<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

// Require composer dependencies if necessary
if (file_exists(__DIR__ . '/vendor/autoload.php')) {
    require_once(__DIR__ . '/vendor/autoload.php');
}

if (!class_exists('ViisonCommon_Plugin_BootstrapV15')) {
    require_once('ViisonCommon/PluginBootstrapV15.php');
}

use Doctrine\DBAL\Connection as DbalConnection;
use Shopware\Components\License\Struct\LicenseUnpackRequest;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\BarcodeLabel\BarcodeLabelPreset;
use Shopware\CustomModels\ViisonPickwareERP\StockValuation\Report;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentStatus;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry;
use Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderItemStatus;
use Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderStatus;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\Models\Config\Element;
use Shopware\Models\Config\Form;
use Shopware\Models\Document\Document as DocumentType;
use Shopware\Models\Document\Element as DocumentElement;
use Shopware\Models\Mail\Mail;
use Shopware\Models\Media\Album;
use Shopware\Models\Shop\Locale;
use Shopware\Models\Shop\Shop;
use Shopware\Models\Snippet\Snippet;
use Shopware\Models\User\User;
use Shopware\Plugins\ViisonCommon\Classes\Document\DocumentBoxCopier;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\InstallationException;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\NumberRangeNotFoundException;
use Shopware\Plugins\ViisonCommon\Classes\Installation\AttributeColumn\AttributeColumnDescription;
use Shopware\Plugins\ViisonCommon\Classes\Installation\AttributeColumn\AttributeColumnInstaller;
use Shopware\Plugins\ViisonCommon\Classes\Installation\AttributeColumn\AttributeColumnUninstaller;
use Shopware\Plugins\ViisonCommon\Classes\Installation\BackendSessionLocaleClassMigration;
use Shopware\Plugins\ViisonCommon\Classes\Installation\Document\DocumentInstallationHelper;
use Shopware\Plugins\ViisonCommon\Classes\Installation\ExpiringLock;
use Shopware\Plugins\ViisonCommon\Classes\Installation\InstallationCheckpointWriter;
use Shopware\Plugins\ViisonCommon\Classes\Installation\MediaAlbum\InstallationHelper as MediaAlbumInstallationHelper;
use Shopware\Plugins\ViisonCommon\Classes\Installation\MediaAlbum\UninstallationHelper as MediaAlbumUninstallationHelper;
use Shopware\Plugins\ViisonCommon\Classes\Installation\Menu\InstallationHelper as MenuInstallationHelper;
use Shopware\Plugins\ViisonCommon\Classes\Installation\OrderNumber\InstallationHelper as OrderNumberInstallationHelper;
use Shopware\Plugins\ViisonCommon\Classes\Installation\SQLHelper;
use Shopware\Plugins\ViisonCommon\Classes\Subscribers as ViisonCommonSubscribers;
use Shopware\Plugins\ViisonCommon\Classes\Util\Currency as CurrencyUtil;
use Shopware\Plugins\ViisonCommon\Classes\Util\Document as DocumentUtil;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Components\HiddenConfigStorageService;
use Shopware\Plugins\ViisonCommon\Components\Plugins\SwagImportExportProfileService;
use Shopware\Plugins\ViisonPickwareERP\Commands;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BinLocation\BinLocationBarcodeLabelType;
use Shopware\Plugins\ViisonPickwareERP\Components\ReturnShipment\ReturnShipmentMailingService;
use Shopware\Plugins\ViisonPickwareERP\Components\ReturnShipment\ReturnShipmentProcessorService;
use Shopware\Plugins\ViisonPickwareERP\Components\Supplier\SupplierNumberGeneratorService;
use Shopware\Plugins\ViisonPickwareERP\Subscribers;
use Shopware\Plugins\ViisonPickwareERP\Subscribers\PluginIntegration\SwagImportExportIntegrationSubscriber;

/*
 * "Declare" some methods that are not required in our bootstrap, but whose existence is checked during Shopware's
 * code review:
 *
 * public function getVersion() {}
 */

final class Shopware_Plugins_Core_ViisonPickwareERP_Bootstrap extends ViisonCommon_Plugin_BootstrapV15
{
    const MIN_SUPPORTED_UPDATE_VERSION = '4.0.13';

    const UNINSTALL_CONFIRM_LOCK = 'ViisonPickwareERPUninstallationUnconfirmed';
    const UNINSTALL_CONFIRM_LOCK_WAIT_TIME_IN_SECONDS = 120;

    const CONFIG_ELEMENT_NAME_LAST_SAFE_UNINSTALL_VERSION = 'pickwareErpLastSafeUninstallVersion';
    const CONFIG_ELEMENT_NAME_INSTALL_ID  = 'viisonPickwareERPInstallId';

    const DOCUMENT_MAILER_MAIL_TEMPLATE_NAME = 'ViisonPickwareERPCommonDocumentMailerEMail';

    const SUPPLIER_ORDER_DOCUMENT_TEMPLATE_NAME = 'Lieferantenbestellung';
    const SUPPLIER_ORDER_DOCUMENT_NUMBERS = 'viison_supplier_order';
    const SUPPLIER_ORDER_DOCUMENT_TEMPLATE_KEY = 'viison_supplier_order';

    const PICK_LIST_NUMBER_RANGE_START = 1000;
    const PICK_LIST_NUMBER_RANGE_NAME = 'viison_pick_list';
    const PICK_LIST_NUMBER_RANGE_DESCRIPTION = 'Pickware Pickliste';
    const PICK_LIST_DOCUMENT_TYPE_KEY = 'viison_pick_list';
    const PICK_LIST_DOCUMENT_TYPE_NAME = 'Pickware Pickliste';
    const PICK_LIST_DOCUMENT_TYPE_TEMPLATE_NAME = 'pick_list.tpl';

    // These are snippets that are part of the ini files provided by Shopware, whose values we want to change. They are
    // automatically reset to the original Shopware values on uninstall.
    const SHOPWARE_SNIPPET_REPLACEMENTS = [
        [
            'namespace' => 'documents/index_sr',
            'name' => 'DocumentIndexCancelationNumber',
            'values' => [
                'en_GB' => 'Reversal invoice {$Document.id} {if $Document.invoiceNumber} for invoice No. {$Document.invoiceNumber} {/if}',
                'de_DE' => 'Stornorechnung Nr. {$Document.id} {if $Document.invoiceNumber} zu Rechnung {$Document.invoiceNumber} {/if}',
            ],
        ],
        [
            'namespace' => 'backend/article/view/main',
            'name' => 'detail/settings/on_sale_box', // article detail settings
            'values' => [
                'en_GB' => 'If available is <= 0, the item can not be ordered',
                'de_DE' => 'Artikel bei Verfügbar <= 0 nicht bestellbar',
            ],
        ],
        [
            'namespace' => 'frontend/account/order_item',
            'name' => 'OrderItemColumnQuantity', // table header on frontend order listing
            'values' => [
                'en_GB' => 'Quantity (canceled)',
                'de_DE' => 'Anzahl (storniert)',
            ],
        ],
    ];

    /**
     * @inheritdoc
     */
    protected $codeVersion = '6.10.2';

    /**
     * @inheritdoc
     */
    protected $minRequiredShopwareVersion = '5.2.4';

    /**
     * @inheritdoc
     */
    protected function runUpdate($oldVersion)
    {
        // Load the plugin's services
        $this->loadServicesForInstallation();

        // Prepare some helpers
        $form = $this->Form();
        /** @var ModelManager $modelManager */
        $modelManager = $this->get('models');
        /* @var $database \Enlight_Components_Db_Adapter_Pdo_Mysql */
        $database = $this->get('db');
        $sqlHelper = new SQLHelper($database);
        $menuInstallationHelper = new MenuInstallationHelper($this->get('models'), $this->getPlugin(), function (array $options) {
            // See Shopware_Components_Plugin_Bootstrap::createMenuItem for possible 'options' fields
            return $this->createMenuItem($options);
        });
        $mediaAlbumInstallationHelper = new MediaAlbumInstallationHelper($modelManager);
        $documentInstallationHelper = new DocumentInstallationHelper($modelManager);
        $orderNumberInstallationHelper = new OrderNumberInstallationHelper($database);
        $attributeColumnInstallationHelper = new AttributeColumnInstaller($modelManager, $sqlHelper);
        $attributeColumnUninstallationHelper = new AttributeColumnUninstaller($modelManager, $sqlHelper);
        $snippetNamespace = $this->getBootstrapSnippetManager()->getNamespace('bootstrap/viison_pickware_erp/main');
        $installationCheckpointWriter = new InstallationCheckpointWriter($this);

        switch ($oldVersion) {
            case 'install':
                // Prevent installation if the plugin was previously uninstalled without deleting data, and the version
                // installed at that time is not supported for incremental updates
                $lastSafeUninstallVersion = $this->get('viison_common.hidden_config_storage')->getConfigValue(
                    self::CONFIG_ELEMENT_NAME_LAST_SAFE_UNINSTALL_VERSION
                );
                if ($lastSafeUninstallVersion && version_compare($lastSafeUninstallVersion, self::MIN_SUPPORTED_UPDATE_VERSION, '<')) {
                    throw InstallationException::invalidReInstall(
                        $this,
                        $lastSafeUninstallVersion,
                        $this->getVersion(),
                        self::MIN_SUPPORTED_UPDATE_VERSION
                    );
                }

                /* Subscribes */

                // Add base subscriber
                $this->subscribeEvent(
                    'Enlight_Controller_Front_StartDispatch',
                    'onStartDispatch'
                );
                $this->subscribeEvent(
                    'Shopware_Console_Add_Command',
                    'onAddConsoleCommand'
                );

                /* Menu items */

                // Create dummy top-level menu item wrapping supplier and supplier order management related menu items
                $supplierMenuItem = $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $this->Menu()->findOneBy(['controller' => 'Article']),
                    'controller' => 'ViisonSupplier',
                    'action' => null,
                    'label' => 'Einkauf',
                    'class' => 'sprite-shopping-basket--plus',
                    'active' => 1,
                    'position' => 4,
                ]);
                // Create sub menu item in the 'ViisonSupplier' main item for managing suppliers
                $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $supplierMenuItem,
                    'controller' => 'ViisonPickwareERPSupplierManagement',
                    'action' => 'Index',
                    'label' => 'Lieferanten',
                    'class' => 'sprite-truck-box-label',
                    'active' => 1,
                ]);
                // Create sub menu item in the 'ViisonSupplier' main item for managing supplier orders
                $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $supplierMenuItem,
                    'controller' => 'ViisonPickwareERPSupplierOrders',
                    'action' => 'Index',
                    'label' => 'Bestellwesen',
                    'class' => 'sprite-documents-stack',
                    'active' => 1,
                ]);
                // Create dummy top-level menu item wrapping stock related menu items
                $stockMenuItem = $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $this->Menu()->findOneBy(['controller' => 'Article']),
                    'controller' => 'ViisonStock',
                    'action' => null,
                    'label' => 'Lager',
                    'class' => 'sprite-wooden-box--pencil',
                    'active' => 1,
                    'position' => 5,
                ]);
                // Create menu item for launching the warehouse management app
                $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $stockMenuItem,
                    'controller' => 'ViisonPickwareERPWarehouseManagement',
                    'action' => 'Index',
                    'label' => 'Lagerverwaltung',
                    'class' => 'sprite-home--pencil',
                    'active' => 1,
                    'position' => -5,
                ]);
                // Create menu item for launching the stock overview app
                $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $stockMenuItem,
                    'controller' => 'ViisonPickwareERPStockOverview',
                    'action' => 'Index',
                    'label' => 'Bestandsübersicht',
                    'class' => 'sprite-report-paper',
                    'active' => 1,
                    'position' => 5,
                ]);
                // Create menu item for rated stock module
                $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $stockMenuItem,
                    'controller' => 'ViisonPickwareERPRatedStock',
                    'action' => 'Index',
                    'label' => 'Bewerteter Warenbestand',
                    'class' => 'sprite-book-open-list',
                    'active' => 1,
                    'position' => 10,
                ]);
                // Create dummy top-level menu item wrapping 'about Pickware' menu items
                $aboutPickwareItem = $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $this->Menu()->findOneBy(['controller' => 'HelpMenu']),
                    'controller' => 'ViisonPickwareERPAbout',
                    'action' => null,
                    'label' => $this->getLabel(),
                    'class' => 'c-sprite-viison-pickware-erp-plugin',
                    'active' => 1,
                    'position' => 50,
                ]);
                // Create menu item for opening the about 'Pickware' module
                $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $aboutPickwareItem,
                    'controller' => 'ViisonPickwareERPAboutBackend',
                    'action' => 'Index',
                    'label' => 'Info',
                    'class' => 'c-sprite-viison-pickware-erp-plugin',
                    'active' => 1,
                    'position' => 1,
                ]);
                // Create menu item for opening the 'Pickware Documentation', link: http://www.pickware.de/dokumentation
                $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $aboutPickwareItem,
                    'controller' => 'ViisonPickwareERPAboutDocumentation',
                    'onclick' => "window.open('https://support.pickware.de/hc/de/articles/360008735018-Shopware-ERP', '_blank');",
                    'label' => 'Documentation',
                    'class' => 'sprite-inbox-document-folder',
                    'active' => 1,
                    'position' => 3,
                ]);
                // Create menu item for opening the 'Contact' module
                $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $aboutPickwareItem,
                    'controller' => 'ViisonPickwareERPAboutContact',
                    'action' => 'Index',
                    'label' => 'Contact (email)',
                    'class' => 'sprite-mail',
                    'active' => 1,
                    'position' => 4,
                ]);
                // Create menu item for opening the 'Pickware modules' module
                $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $this->Menu()->findOneBy(['controller' => 'ConfigurationMenu']),
                    'controller' => 'PluginManager',
                    'action' => 'ViisonPickwareERPPluginManagerPickwareModules',
                    'label' => 'Pickware Module',
                    'class' => 'settings--plugin-manager',
                    'active' => 1,
                    'position' => 50,
                ]);

                // Move the 'fabricator' (Shopware supplier) and reviews (vote) menu items down to the bottom
                $fabricatorMenuItem = $this->Menu()->findOneBy(['controller' => 'Supplier']);
                $fabricatorMenuItem->setPosition(10);
                $reviewsMenuItem = $this->Menu()->findOneBy(['controller' => 'Vote']);
                $reviewsMenuItem->setPosition(15);

                /* Config elements */

                // Add a config element for selecting whether purchase prices are gross or net values
                $form->setElement(
                    'select',
                    'purchasePriceMode',
                    [
                        'label' => 'Eingabe von Einkaufspreisen',
                        'description' => 'Wählen Sie den Modus, in dem Einkaufspreise eingegeben/angezeigt werden sollen.',
                        'value' => 'net',
                        'store' => [
                            [
                                'net',
                                'Netto',
                            ],
                            [
                                'gross',
                                'Brutto',
                            ],
                        ],
                    ]
                );
                // Add a config element for the specifying comments that are available e.g. when manually changing the stock
                $form->setElement(
                    'text',
                    'stockEntryComments',
                    [
                        'label' => 'Bestands-Eintrag Kommentare',
                        'description' => 'Tragen Sie hier einen oder mehrere Kommentare ein, die in z.B. beim manuellen Ändern des Warenbestandes zur Verfügung stehen sollen. Kommentare bitte in "Anführungszeichen" schreiben und mehrere Kommentare durch Kommata trennen.',
                        'value' => sprintf('"%s","%s","%s"', $snippetNamespace->get('config_stock_entry_comment/stocktake'), $snippetNamespace->get('config_stock_entry_comment/writeoff'), $snippetNamespace->get('config_stock_entry_comment/relocation')),
                    ]
                );
                // Add a config element for selecting the columns that are contained in a supplier order CSV export
                $form->setElement(
                    'select',
                    'supplierOrderCSVFields',
                    [
                        'label' => 'Bestellwesen - Felder im CSV-Export',
                        'description' => 'Wählen Sie hier die Felder aus, die im CSV-Export von Lieferantenbestellungen enthalten sein sollen.',
                        'value' => [
                            'fabricatorNumber',
                            'fabricator',
                            'articleName',
                            'articleNumber',
                            'unit',
                            'quantity',
                        ],
                        'store' => [
                            [
                                'fabricatorNumber',
                                'Hersteller-Nr.',
                            ],
                            [
                                'fabricator',
                                'Hersteller',
                            ],
                            [
                                'articleName',
                                'Artikel-Bezeichnung',
                            ],
                            [
                                'articleNumber',
                                'Eigene Artikel-Nr.',
                            ],
                            [
                                'supplierArticleNumber',
                                'Lieferanten Artikel-Nr.',
                            ],
                            [
                                'ean',
                                'EAN',
                            ],
                            [
                                'unit',
                                'Verpackungseinheit',
                            ],
                            [
                                'quantity',
                                'Menge',
                            ],
                            [
                                'purchasePrice',
                                'Einkaufspreis',
                            ],
                            [
                                'price',
                                'Verkaufspreis',
                            ],
                            [
                                'orderNumber',
                                'Bestell-Nr.',
                            ],
                            [
                                'warehouseAddress',
                                'Adresse des Ziellagers',
                            ],
                        ],
                        'multiSelect' => true,
                        'editable' => false,
                    ]
                );
                // Add a config element for enabling/disabling the auto invoice creation
                $form->setElement(
                    'checkbox',
                    'automaticInvoiceCreation',
                    [
                        'label' => 'Autom. Rechnungserzeugung & -versand',
                        'description' => 'Aktivieren Sie dieses Feld, um Rechnung direkt beim Bestellabschluss zu erzeugen und per E-Mail an den Kunden und ggf. das Archiv zu verschicken. Hinweis: Pickware POS Bestellung sind hiervon ausgenommen; Die Pickware Versand App erstellt alle benötigen Dokumente selbst.',
                        'value' => false,
                    ]
                );
                // Add a config element for selecting those dispatch methods, for which the invoice is created automatically
                $form->setElement(
                    'select',
                    'automaticInvoiceCreationDispatchMethodIds',
                    [
                        'label' => 'Autom. Rechnungserzeugung - Versandarten',
                        'description' => 'Wählen Sie hier die Versandarten aus, für die beim Bestellabschluss Rechnungen erzeugt und verschickt werden sollen. Lassen Sie die Liste leer, um die Rechnungen für alle Versandarten zu erzeugen.',
                        'value' => [],
                        'store' => 'Shopware.apps.Base.store.Dispatch',
                        'multiSelect' => true,
                        'editable' => false,
                    ]
                );
                // Add a config element for selecting those payment methods, for which the invoice is created automatically
                $form->setElement(
                    'select',
                    'automaticInvoiceCreationPaymentMethodIds',
                    [
                        'label' => 'Autom. Rechnungserzeugung - Zahlungsarten',
                        'description' => 'Wählen Sie hier die Zahlungsarten aus, für die beim Bestellabschluss Rechnungen erzeugt und verschickt werden sollen. Lassen Sie die Liste leer, um die Rechnungen für alle Zahlungsarten zu erzeugen.',
                        'value' => [],
                        'store' => 'Shopware.apps.Base.store.Payment',
                        'displayField' => 'description',
                        'multiSelect' => true,
                        'editable' => false,
                    ]
                );
                // Add a config element for specifying one or more email addresses, to which all generated invoices will be sent
                $form->setElement(
                    'text',
                    'invoiceArchiveEmailAddresses',
                    [
                        'label' => 'Adressen für Rechnungsarchivierung',
                        'description' => 'Tragen Sie hier die E-Mail Adressen ein, an die alle Rechnungen und Stornorechnungen gesendet werden sollen, nachdem sie generiert wurden. Mehrere E-Mail Adressen bitte durch Kommata trennen.',
                        'value' => '',
                    ]
                );
                // Add a config element for displaying/hiding the about window
                $form->setElement(
                    'checkbox',
                    'displayPickwareAboutWindow',
                    [
                        'label' => 'Pickware Info-Dialog anzeigen',
                        'description' => 'Bestimmt, ob der Pickware Info-Dialog beim Laden des Backends angezeigt werden soll',
                        'value' => true,
                    ]
                );

                /* Attribute fields */

                $attributeColumnInstallationHelper->addAttributeColumnsIfNotExist([
                    // Articles: Add column to store the incoming stock
                    new AttributeColumnDescription(
                        's_articles_attributes',
                        'viison_incoming_stock',
                        'int(11)',
                        0,
                        false
                    ),
                    // Articles: Add column to store current physical stock of all warehouses
                    new AttributeColumnDescription(
                        's_articles_attributes',
                        'viison_physical_stock',
                        'int(11)',
                        0,
                        false
                    ),
                    // Articles: Add column to store the current physical stock of all warehouses, whose stock is
                    // available for sale
                    new AttributeColumnDescription(
                        's_articles_attributes',
                        'viison_physical_stock_for_sale',
                        'int(11)',
                        0,
                        false
                    ),
                    // Articles: Add column to store the 'not relevant for stock manager' flag
                    new AttributeColumnDescription(
                        's_articles_attributes',
                        'viison_not_relevant_for_stock_manager',
                        'tinyint(1)',
                        0,
                        false
                    ),
                    // Order: Add column for optimistic order locking
                    new AttributeColumnDescription(
                        's_order_attributes',
                        'viison_last_changed',
                        'DATETIME',
                        '2000-01-01 00:00:00',
                        false
                    ),
                    // Order details: Add column to store the canceled quantity
                    new AttributeColumnDescription(
                        's_order_details_attributes',
                        'viison_canceled_quantity',
                        'int(11)',
                        0,
                        false
                    ),
                ]);

                /* Warehouse tables */

                // Create the table for Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_warehouses` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `code` varchar(255) NOT NULL,
                        `name` varchar(255) NOT NULL,
                        `stockAvailable` tinyint(1) NOT NULL,
                        `defaultWarehouse` tinyint(1) NOT NULL,
                        `defaultBinLocationId` int(11) DEFAULT NULL,
                        `contact` varchar(255) DEFAULT NULL,
                        `email` varchar(255) DEFAULT NULL,
                        `phone` varchar(255) DEFAULT NULL,
                        `address` longtext,
                        `comment` longtext,
                        `binLocationFormatComponents` longtext COMMENT \'(DC2Type:array)\',
                        PRIMARY KEY (`id`),
                        UNIQUE KEY `code` (`code`),
                        UNIQUE KEY `name` (`name`),
                        UNIQUE KEY `UNIQ_BD6085D427714D0D` (`defaultBinLocationId`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_warehouse_bin_locations` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `warehouseId` int(11) NOT NULL,
                        `code` varchar(255) NOT NULL,
                        PRIMARY KEY (`id`),
                        UNIQUE KEY `warehouseId_code` (`warehouseId`,`code`),
                        KEY `IDX_32025EA5E8DE5B58` (`warehouseId`),
                        CONSTRAINT `FK_32025EA5E8DE5B58` FOREIGN KEY (`warehouseId`) REFERENCES `s_plugin_pickware_warehouses` (`id`) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Manually add a foreign key constraint between 's_plugin_pickware_warehouses' and
                // 's_plugin_pickware_warehouse_bin_locations'
                $sqlHelper->ensureForeignKeyConstraint(
                    's_plugin_pickware_warehouses',
                    'defaultBinLocationId',
                    's_plugin_pickware_warehouse_bin_locations',
                    'id',
                    'SET NULL'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\Warehouse\ArticleDetailBinLocationMapping
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_warehouse_bin_location_article_details` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `binLocationId` int(11) NOT NULL,
                        `articleDetailId` int(11) NOT NULL,
                        `stock` int(11) NOT NULL DEFAULT \'0\',
                        `defaultMapping` tinyint(1) NOT NULL,
                        `reservedStock` int(11) NOT NULL,
                        `lastStocktake` datetime DEFAULT NULL,
                        PRIMARY KEY (`id`),
                        UNIQUE KEY `binLocationId_articleDetailId` (`binLocationId`,`articleDetailId`),
                        KEY `IDX_AEF214EBEC8281FB` (`binLocationId`),
                        KEY `IDX_AEF214EBD0871F1C` (`articleDetailId`),
                        CONSTRAINT `FK_AEF214EBEC8281FB` FOREIGN KEY (`binLocationId`) REFERENCES `s_plugin_pickware_warehouse_bin_locations` (`id`) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\StockLedger\WarehouseArticleDetailStockCount
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_warehouse_stocks` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `warehouseId` int(11) NOT NULL,
                        `articleDetailId` int(11) NOT NULL,
                        `stock` int(11) NOT NULL,
                        `reservedStock` int(11) NOT NULL,
                        PRIMARY KEY (`id`),
                        UNIQUE KEY `warehouseId_articleDetailId` (`warehouseId`,`articleDetailId`),
                        KEY `IDX_58E33076E8DE5B58` (`warehouseId`),
                        KEY `IDX_58E33076D0871F1C` (`articleDetailId`),
                        CONSTRAINT `FK_58E33076E8DE5B58` FOREIGN KEY (`warehouseId`) REFERENCES `s_plugin_pickware_warehouses` (`id`) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\StockLedger\OrderStockReservation
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_warehouse_reserved_stocks` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `binLocationArticleDetailId` int(11) NOT NULL,
                        `orderDetailId` int(11) DEFAULT NULL,
                        `quantity` int(11) NOT NULL,
                        PRIMARY KEY (`id`),
                        UNIQUE KEY `binLocationArticleDetailId_orderDetailId` (`binLocationArticleDetailId`,`orderDetailId`),
                        KEY `IDX_F2418CF7AD7BACD9` (`binLocationArticleDetailId`),
                        KEY `IDX_F2418CF7CB2FF2E0` (`orderDetailId`),
                        CONSTRAINT `FK_F2418CF7AD7BACD9` FOREIGN KEY (`binLocationArticleDetailId`) REFERENCES `s_plugin_pickware_warehouse_bin_location_article_details` (`id`) ON DELETE CASCADE,
                        CONSTRAINT `FK_F2418CF7CB2FF2E0` FOREIGN KEY (`orderDetailId`) REFERENCES `s_order_details` (`id`) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );

                /* Supplier tables */

                // Create the table for Shopware\CustomModels\ViisonPickwareERP\Supplier\Supplier
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_suppliers` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `salutation` varchar(255) COLLATE utf8_unicode_ci DEFAULT NULL,
                        `name` varchar(255) COLLATE utf8_unicode_ci NOT NULL,
                        `contact` varchar(255) COLLATE utf8_unicode_ci NOT NULL,
                        `address` longtext COLLATE utf8_unicode_ci,
                        `email` varchar(255) COLLATE utf8_unicode_ci NOT NULL,
                        `phone` varchar(40) COLLATE utf8_unicode_ci NOT NULL,
                        `fax` varchar(40) COLLATE utf8_unicode_ci NOT NULL,
                        `comment` longtext COLLATE utf8_unicode_ci,
                        `customerNumber` varchar(40) COLLATE utf8_unicode_ci DEFAULT NULL,
                        `deliveryTime` int(11) DEFAULT NULL,
                        PRIMARY KEY (`id`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\Supplier\ArticleDetailSupplierMapping
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_supplier_article_details` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `supplierId` int(11) NOT NULL,
                        `articleDetailId` int(11) unsigned NOT NULL,
                        `defaultSupplier` tinyint(1) NOT NULL,
                        `purchasePrice` decimal(10,2) DEFAULT NULL,
                        `supplierArticleNumber` varchar(255) DEFAULT NULL,
                        `orderAmount` int(11) DEFAULT NULL,
                        PRIMARY KEY (`id`),
                        UNIQUE KEY `supplier_articleDetail` (`supplierId`,`articleDetailId`),
                        KEY `IDX_CD418A33DF05A1D3` (`supplierId`),
                        KEY `IDX_CD418A33D0871F1C` (`articleDetailId`),
                        CONSTRAINT `FK_CD418A33D0871F1C` FOREIGN KEY (`articleDetailId`) REFERENCES `s_articles_details` (`id`) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\Supplier\ManufacturerSupplierMapping
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_supplier_fabricators` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `supplierId` int(11) NOT NULL,
                        `fabricatorId` int(11) NOT NULL,
                        PRIMARY KEY (`id`),
                        UNIQUE KEY `supplier_fabricator` (`supplierId`,`fabricatorId`),
                        KEY `IDX_D34D7EB7DF05A1D3` (`supplierId`),
                        KEY `IDX_D34D7EB7C3F3041F` (`fabricatorId`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );

                /* Supplier order tables */

                // Create the table for Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderStatus
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_supplier_order_states` (
                        `id` int(11) NOT NULL,
                        `name` varchar(255) NOT NULL,
                        `description` varchar(255) NOT NULL,
                        PRIMARY KEY (`id`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrder
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_supplier_orders` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `status` int(11) NOT NULL,
                        `created` datetime NOT NULL,
                        `orderNumber` varchar(40) NOT NULL,
                        `total` decimal(10,2) DEFAULT NULL,
                        `supplierId` int(11) DEFAULT NULL,
                        `warehouseId` int(11) NOT NULL,
                        `comment` varchar(255) DEFAULT NULL,
                        `documentComment` varchar(255) DEFAULT NULL,
                        `deliveryDate` date DEFAULT NULL,
                        `paymentDueDate` date DEFAULT NULL,
                        `paymentStatus` int(11) DEFAULT NULL,
                        `userId` int(11) DEFAULT NULL,
                        PRIMARY KEY (`id`),
                        KEY `IDX_700C12FFDF05A1D3` (`supplierId`),
                        KEY `IDX_700C12FFE8DE5B58` (`warehouseId`),
                        KEY `IDX_700C12FF64B64DCC` (`userId`),
                        KEY `IDX_700C12FF7B00651C` (`status`),
                        CONSTRAINT `FK_700C12FF7B00651C` FOREIGN KEY (`status`) REFERENCES `s_plugin_pickware_supplier_order_states` (`id`),
                        CONSTRAINT `FK_700C12FFE8DE5B58` FOREIGN KEY (`warehouseId`) REFERENCES `s_plugin_pickware_warehouses` (`id`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderItemStatus
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_supplier_order_article_detail_states` (
                        `id` int(11) NOT NULL,
                        `name` varchar(255) NOT NULL,
                        `description` varchar(255) NOT NULL,
                        PRIMARY KEY (`id`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderItem
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_supplier_order_article_details` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `status` int(11) NOT NULL,
                        `orderId` int(11) DEFAULT NULL,
                        `articleDetailId` int(11) DEFAULT NULL,
                        `name` varchar(255) DEFAULT NULL,
                        `orderNumber` varchar(40) NOT NULL,
                        `orderedQuantity` int(11) NOT NULL,
                        `deliveredQuantity` int(11) NOT NULL,
                        `price` decimal(10,2) DEFAULT NULL,
                        `deliveryTime` int(11) DEFAULT NULL,
                        `fabricatorName` varchar(255) DEFAULT NULL,
                        `fabricatorNumber` varchar(255) DEFAULT NULL,
                        `supplierArticleNumber` varchar(255) DEFAULT NULL,
                        PRIMARY KEY (`id`),
                        KEY `IDX_A96CECCFA237437` (`orderId`),
                        KEY `IDX_A96CECCD0871F1C` (`articleDetailId`),
                        KEY `IDX_A96CECC7B00651C` (`status`),
                        CONSTRAINT `FK_A96CECC7B00651C` FOREIGN KEY (`status`) REFERENCES `s_plugin_pickware_supplier_order_article_detail_states` (`id`),
                        CONSTRAINT `FK_A96CECCFA237437` FOREIGN KEY (`orderId`) REFERENCES `s_plugin_pickware_supplier_orders` (`id`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderAttachment
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_supplier_order_attachments` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `orderId` int(11) NOT NULL,
                        `filename` varchar(255) NOT NULL,
                        `date` datetime NOT NULL,
                        `mediaId` int(11) NOT NULL,
                        PRIMARY KEY (`id`),
                        KEY `IDX_29FBCC54FA237437` (`orderId`),
                        KEY `IDX_29FBCC5427D9F5AC` (`mediaId`),
                        CONSTRAINT `FK_29FBCC54FA237437` FOREIGN KEY (`orderId`) REFERENCES `s_plugin_pickware_supplier_orders` (`id`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );

                /* Return shipment tables */

                // Create the table for Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipment
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_reshipments` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `orderId` int(11) NOT NULL,
                        `userId` int(11) NOT NULL,
                        `statusId` int(11) NOT NULL,
                        `comment` longtext,
                        `created` datetime NOT NULL,
                        PRIMARY KEY (`id`),
                        KEY `IDX_5CCEA235FA237437` (`orderId`),
                        KEY `IDX_5CCEA23564B64DCC` (`userId`),
                        KEY `IDX_5CCEA235F112F078` (`statusId`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentItem
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_reshipment_items` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `reshipmentId` int(11) NOT NULL,
                        `orderDetailId` int(11) NOT NULL,
                        `returnedQuantity` int(11) NOT NULL,
                        `depreciatedQuantity` int(11) NOT NULL,
                        PRIMARY KEY (`id`),
                        KEY `IDX_92EC636A4DF360EA` (`reshipmentId`),
                        KEY `IDX_92EC636ACB2FF2E0` (`orderDetailId`),
                        CONSTRAINT `FK_92EC636A4DF360EA` FOREIGN KEY (`reshipmentId`) REFERENCES `s_plugin_pickware_reshipments` (`id`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentAttachment
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_reshipment_attachments` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `reshipmentId` int(11) NOT NULL,
                        `mediaId` int(11) NOT NULL,
                        PRIMARY KEY (`id`),
                        KEY `IDX_DCE36F4E4DF360EA` (`reshipmentId`),
                        KEY `IDX_DCE36F4E27D9F5AC` (`mediaId`),
                        CONSTRAINT `FK_DCE36F4E4DF360EA` FOREIGN KEY (`reshipmentId`) REFERENCES `s_plugin_pickware_reshipments` (`id`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentStatus
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_reshipment_status` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `name` varchar(255) NOT NULL,
                        PRIMARY KEY (`id`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );

                /* Stock entry tables */

                // Create the table for Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_stocks` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `type` varchar(255) DEFAULT NULL,
                        `oldInstockQuantity` int(11) NOT NULL,
                        `newInstockQuantity` int(11) NOT NULL,
                        `changeAmount` int(11) DEFAULT NULL,
                        `purchasePrice` decimal(10,2) DEFAULT NULL,
                        `comment` varchar(255) DEFAULT NULL,
                        `created` datetime NOT NULL,
                        `parentId` int(11) DEFAULT NULL,
                        `articleDetailId` int(11) DEFAULT NULL,
                        `orderDetailId` int(11) DEFAULT NULL,
                        `userId` int(11) DEFAULT NULL,
                        `supplierOrderItemId` int(11) DEFAULT NULL,
                        `reshipmentItemId` int(11) DEFAULT NULL,
                        `warehouseId` int(11) NOT NULL,
                        `transactionId` varchar(255) NOT NULL,
                        PRIMARY KEY (`id`),
                        KEY `IDX_6CA6919D10EE4CEE` (`parentId`),
                        KEY `IDX_6CA6919DD0871F1C` (`articleDetailId`),
                        KEY `IDX_6CA6919DCB2FF2E0` (`orderDetailId`),
                        KEY `IDX_6CA6919D64B64DCC` (`userId`),
                        KEY `IDX_6CA6919D3FD49860` (`supplierOrderItemId`),
                        KEY `IDX_6CA6919DA0DEA3FE` (`reshipmentItemId`),
                        KEY `IDX_6CA6919DE8DE5B58` (`warehouseId`),
                        CONSTRAINT `FK_6CA6919D10EE4CEE` FOREIGN KEY (`parentId`) REFERENCES `s_plugin_pickware_stocks` (`id`) ON DELETE CASCADE,
                        CONSTRAINT `FK_6CA6919D3FD49860` FOREIGN KEY (`supplierOrderItemId`) REFERENCES `s_plugin_pickware_supplier_order_article_details` (`id`),
                        CONSTRAINT `FK_6CA6919DA0DEA3FE` FOREIGN KEY (`reshipmentItemId`) REFERENCES `s_plugin_pickware_reshipment_items` (`id`),
                        CONSTRAINT `FK_6CA6919DE8DE5B58` FOREIGN KEY (`warehouseId`) REFERENCES `s_plugin_pickware_warehouses` (`id`) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\StockLedger\BinLocationStockSnapshot
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_warehouse_bin_location_stock_snapshots` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `stockEntryId` int(11) NOT NULL,
                        `binLocationId` int(11) NOT NULL,
                        `stock` int(11) NOT NULL,
                        PRIMARY KEY (`id`),
                        UNIQUE KEY `stockEntryId_binLocationId` (`stockEntryId`,`binLocationId`),
                        KEY `IDX_E40AFB037794F255` (`stockEntryId`),
                        KEY `IDX_E40AFB03EC8281FB` (`binLocationId`),
                        CONSTRAINT `FK_E40AFB037794F255` FOREIGN KEY (`stockEntryId`) REFERENCES `s_plugin_pickware_stocks` (`id`) ON DELETE CASCADE,
                        CONSTRAINT `FK_E40AFB03EC8281FB` FOREIGN KEY (`binLocationId`) REFERENCES `s_plugin_pickware_warehouse_bin_locations` (`id`) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\ItemProperty\ItemProperty
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_property_types` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `name` varchar(255) COLLATE utf8_unicode_ci NOT NULL,
                        PRIMARY KEY (`id`),
                        UNIQUE KEY `UNIQ_DA9DD9DF5E237E06` (`name`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\ItemProperty\ArticleDetailItemProperty
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_article_detail_property_types` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `articleDetailId` int(11) unsigned NOT NULL, # Must be unsigned to match s_articles_details.id
                        `propertyTypeId` int(11) NOT NULL,
                        PRIMARY KEY (`id`),
                        UNIQUE KEY `articleDetailId_propertyTypeId` (`articleDetailId`,`propertyTypeId`),
                        KEY `IDX_F0624D96D0871F1C` (`articleDetailId`),
                        KEY `IDX_F0624D96D073BC37` (`propertyTypeId`),
                        CONSTRAINT `FK_F0624D96D0871F1C` FOREIGN KEY (`articleDetailId`) REFERENCES `s_articles_details` (`id`) ON DELETE CASCADE,
                        CONSTRAINT `FK_F0624D96D073BC37` FOREIGN KEY (`propertyTypeId`) REFERENCES `s_plugin_pickware_property_types` (`id`) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockItem
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_stock_entry_items` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `stockEntryId` int(11) NOT NULL,
                        PRIMARY KEY (`id`),
                        KEY `IDX_542F0D847794F255` (`stockEntryId`),
                        CONSTRAINT `FK_542F0D847794F255` FOREIGN KEY (`stockEntryId`) REFERENCES `s_plugin_pickware_stocks` (`id`) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Create the table for Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockItemPropertyValue
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `s_plugin_pickware_stock_entry_item_properties` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `stockEntryItemId` int(11) NOT NULL,
                        `propertyTypeId` int(11) NOT NULL,
                        `value` longtext COLLATE utf8_unicode_ci DEFAULT NULL,
                        PRIMARY KEY (`id`),
                        UNIQUE KEY `stockEntryItemId_propertyTypeId` (`stockEntryItemId`,`propertyTypeId`),
                        KEY `IDX_E228DFA5F7E8B57E` (`stockEntryItemId`),
                        KEY `IDX_E228DFA5D073BC37` (`propertyTypeId`),
                        CONSTRAINT `FK_E228DFA5F7E8B57E` FOREIGN KEY (`stockEntryItemId`) REFERENCES `s_plugin_pickware_stock_entry_items` (`id`) ON DELETE CASCADE,
                        CONSTRAINT `FK_E228DFA5D073BC37` FOREIGN KEY (`propertyTypeId`) REFERENCES `s_plugin_pickware_property_types` (`id`) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );

                /* Custom number ranges */

                // Add a range for supplier orders
                $orderNumberInstallationHelper->createSequenceNumberGeneratorIfNotExists(
                    'viison_supplier_order',
                    10000,
                    'Lieferantenbestellungen'
                );
                // Add a range for invoice cancellation documents, but don't overwrite a possibly existing
                // range, because Shopware uses this number already without creating it. Hence someone
                // might already have created it manually.
                $document = $modelManager->getRepository(DocumentType::class)->findOneBy([
                    'template' => 'index_sr.tpl',
                ]);
                $orderNumberInstallationHelper->createSequenceNumberGeneratorIfNotExists(
                    ($document) ? $document->getNumbers() : 'doc_3',
                    30000,
                    ($document) ? $document->getName() : 'Stornorechnung'
                );

                /* Custom media albums */

                // Add a new media album for pictures/documents related to supplier orders
                $mediaAlbumInstallationHelper->createMediaAlbumUnlessExists(
                    'Pickware Lieferantenbestellungen',
                    'sprite-truck-box-label',
                    31
                );

                /* Widgets */

                // Check for existing Pickware news widget (ensures idempotent installation)
                $pickwareNewsWidgetName = 'viison-pickware-erp-news-widget';
                $plugin = $this->getPlugin();
                $pickwareNewsWidget = $plugin->getWidgets()->filter(function ($widget) use ($pickwareNewsWidgetName) {
                    return $widget->getName() === $pickwareNewsWidgetName;
                })->first();
                if (!$pickwareNewsWidget) {
                    // Add Pickware news widget
                    $this->createWidget($pickwareNewsWidgetName);
                    $pickwareNewsWidget = $plugin->getWidgets()->last();
                    $modelManager->persist($pickwareNewsWidget);
                    $modelManager->flush($pickwareNewsWidget);
                    $modelManager->refresh($pickwareNewsWidget);
                    $newlyCreatedWidget = true;
                }
                // Add view to widget for each existing backend user
                $users = $modelManager->getRepository(User::class)->findAll();
                foreach ($users as $user) {
                    if (!$newlyCreatedWidget) {
                        // Check for existing assignment (ensures idempotent installation)
                        $existingView = $pickwareNewsWidget->getViews()->filter(function ($view) use ($user) {
                            return $view->getAuth()->getId() == $user->getId();
                        })->first();
                        if ($existingView) {
                            continue;
                        }
                    }
                    // Create new assignment
                    $view = new \Shopware\Models\Widget\View();
                    $view->setWidget($pickwareNewsWidget);
                    $view->setAuth($user);
                    $view->setColumn(0);
                    $view->setPosition(1);
                    $modelManager->persist($view);
                    $modelManager->flush($view);
                }

                /* Document templates */

                // Add configurable supplier order document template
                $supplierOrderDocumentType = $documentInstallationHelper->ensureDocumentType(
                    self::SUPPLIER_ORDER_DOCUMENT_TEMPLATE_KEY,
                    self::SUPPLIER_ORDER_DOCUMENT_TEMPLATE_NAME,
                    'supplier_order.tpl',
                    self::SUPPLIER_ORDER_DOCUMENT_NUMBERS,
                    25,
                    10,
                    20,
                    20,
                    10
                );
                // Copy all document elements from index.tpl that are not on the blacklist
                $documentElementBlackList = [
                    'Content_Info',
                    'Content_Amount',
                ];
                /** @var \Shopware\Models\Document\Element[] $indexElements */
                $indexElements = $modelManager->getRepository(DocumentElement::class)->findBy([
                    'documentId' => 1,
                ]);
                foreach ($indexElements as $element) {
                    if (in_array($element->getName(), $documentElementBlackList, true)) {
                        continue;
                    }
                    $documentInstallationHelper->ensureDocumentElement(
                        $supplierOrderDocumentType,
                        $element->getName(),
                        $element->getValue(),
                        $element->getStyle()
                    );
                }

                /* Other */

                // Add custom email templates
                $this->createEmailTemplates();
                // Add new email templates for sending other document types per email than invoices, rename old invoice mail
                /** @var DocumentType $document */
                $this->forEachDocumentType(function ($document) {
                    $this->createOrderDocumentEmailTemplate($document);
                });

                // Fix bug in SQ Demo data (if necessary): exchange unit and description in s_core_unit of ml
                $database->query(
                    "UPDATE s_core_units
                    SET unit = 'ml', description = 'Milliliter'
                    WHERE unit = 'Milliliter' AND description = 'ml'"
                );

                // Add the default supplier order statuses and make sure that the 'open' status has ID 0
                $database->query(
                    "INSERT INTO s_plugin_pickware_supplier_order_states (id, name, description)
                    VALUES
                        (:sentToSupplierId, :sentToSupplierName, 'An Lieferant gesendet'),
                        (:dispatchedBySupplierId, :dispatchedBySupplierName, 'Vom Lieferant versandt'),
                        (:partlyReceivedId, :partlyReceivedName, 'Teilweise geliefert'),
                        (:completelyReceivedId, :completelyReceivedName, 'Komplett geliefert'),
                        (:canceledId, :canceledName, 'Storniert / Abgeschrieben')
                    ON DUPLICATE KEY UPDATE
                        id = VALUES(id), name = VALUES(name), description = VALUES(description);",
                    [
                        'sentToSupplierId' => SupplierOrderStatus::VALUE_SENT_TO_SUPPLIER,
                        'sentToSupplierName' => SupplierOrderStatus::SENT_TO_SUPPLIER,
                        'dispatchedBySupplierId' => SupplierOrderStatus::VALUE_DISPATCHED_BY_SUPPLIER,
                        'dispatchedBySupplierName' => SupplierOrderStatus::DISPATCHED_BY_SUPPLIER,
                        'partlyReceivedId' => SupplierOrderStatus::VALUE_PARTLY_RECEIVED,
                        'partlyReceivedName' => SupplierOrderStatus::PARTLY_RECEIVED,
                        'completelyReceivedId' => SupplierOrderStatus::VALUE_COMPLETELY_RECEIVED,
                        'completelyReceivedName' => SupplierOrderStatus::COMPLETELY_RECEIVED,
                        'canceledId' => SupplierOrderStatus::VALUE_CANCELED,
                        'canceledName' => SupplierOrderStatus::CANCELED,
                    ]
                );
                // Check whether the 'open' status already exists (an 'ON DUPLICATE KEY' check does not
                // work in that case, since the ID is manually set to 0)
                $openStatus = $database->fetchRow(
                    'SELECT *
                    FROM s_plugin_pickware_supplier_order_states
                    WHERE id = :openId',
                    [
                        'openId' => SupplierOrderStatus::VALUE_OPEN,
                    ]
                );
                if (empty($openStatus)) {
                    // Add 'open' status with ID 0
                    $database->query(
                        "INSERT INTO s_plugin_pickware_supplier_order_states (name, description)
                        VALUES (:openName, 'Offen');

                        UPDATE s_plugin_pickware_supplier_order_states
                        SET id = :openId
                        WHERE name = :openName;",
                        [
                            'openId' => SupplierOrderStatus::VALUE_OPEN,
                            'openName' => SupplierOrderStatus::OPEN,
                        ]
                    );
                }

                // Add the default supplier order article statuses and make sure the 'open' status has ID 0
                $database->query(
                    "INSERT INTO s_plugin_pickware_supplier_order_article_detail_states (id, name, description)
                    VALUES
                        (:partlyReceivedId, :partlyReceivedName, 'Teilweise geliefert'),
                        (:completelyReceivedId, :completelyReceivedName, 'Komplett geliefert'),
                        (:canceledId, :canceledName, 'Storniert / Abgeschrieben'),
                        (:refundId, :refundName, 'Gutschrift'),
                        (:missingId, :missingName, 'Nicht geliefert')
                    ON DUPLICATE KEY UPDATE
                        id = VALUES(id), name = VALUES(name), description = VALUES(description);",
                    [
                        'partlyReceivedId' => SupplierOrderItemStatus::VALUE_PARTLY_RECEIVED,
                        'partlyReceivedName' => SupplierOrderItemStatus::PARTLY_RECEIVED,
                        'completelyReceivedId' => SupplierOrderItemStatus::VALUE_COMPLETELY_RECEIVED,
                        'completelyReceivedName' => SupplierOrderItemStatus::COMPLETELY_RECEIVED,
                        'canceledId' => SupplierOrderItemStatus::VALUE_CANCELED,
                        'canceledName' => SupplierOrderItemStatus::CANCELED,
                        'refundId' => SupplierOrderItemStatus::VALUE_REFUND,
                        'refundName' => SupplierOrderItemStatus::REFUND,
                        'missingId' => SupplierOrderItemStatus::VALUE_MISSING,
                        'missingName' => SupplierOrderItemStatus::MISSING,
                    ]
                );
                // Check whether the 'open' status already exists (an 'ON DUPLICATE KEY' check does not
                // work in that case, since the ID is manually set to 0)
                $openStatus = $database->fetchRow(
                    'SELECT *
                    FROM s_plugin_pickware_supplier_order_article_detail_states
                    WHERE id = :openId',
                    [
                        'openId' => SupplierOrderItemStatus::VALUE_OPEN,
                    ]
                );
                if (empty($openStatus)) {
                    // Add 'open' status with ID 0
                    $database->query(
                        "INSERT INTO s_plugin_pickware_supplier_order_article_detail_states (name, description)
                        VALUES (:openName, 'Offen');

                        UPDATE s_plugin_pickware_supplier_order_article_detail_states
                        SET id = :openId
                        WHERE name = :openName;",
                        [
                            'openId' => SupplierOrderItemStatus::VALUE_OPEN,
                            'openName' => SupplierOrderItemStatus::OPEN,
                        ]
                    );
                }

                // Add the return shipment statuses
                $database->query(
                    'INSERT IGNORE INTO s_plugin_pickware_reshipment_status (id, name)
                    VALUES
                        (:receivedId, :receivedName),
                        (:completedId, :completedName)',
                    [
                        'completedId' => ReturnShipmentStatus::STATUS_COMPLETED_ID,
                        'completedName' => ReturnShipmentStatus::STATUS_COMPLETED_NAME,
                        'receivedId' => ReturnShipmentStatus::STATUS_RECEIVED_ID,
                        'receivedName' => ReturnShipmentStatus::STATUS_RECEIVED_NAME,
                    ]
                );

                // Check whether any warehouse exists
                $warehouseCount = $database->fetchOne(
                    'SELECT COUNT(*)
                    FROM s_plugin_pickware_warehouses'
                );
                if (intval($warehouseCount) === 0) {
                    // Create a default warehouse and its null bin location
                    $database->query(
                        'INSERT IGNORE INTO s_plugin_pickware_warehouses (code, name, stockAvailable, defaultWarehouse)
                        VALUES (\'HL\', \'Hauptlager\', 1, 1)'
                    );
                    $defaultWarehouseId = $database->lastInsertId();
                    $database->query(
                        'INSERT IGNORE INTO s_plugin_pickware_warehouse_bin_locations (warehouseId, code)
                        VALUES (:warehouseId, :code)',
                        [
                            'code' => 'pickware_default_location', // Use pre-5.0.0 code for initial setup
                            'warehouseId' => $defaultWarehouseId,
                        ]
                    );
                    $nullBinLocationId = $database->lastInsertId();
                    $database->query(
                        'UPDATE s_plugin_pickware_warehouses
                        SET defaultBinLocationId = :nullBinLocationId
                        WHERE id = :warehouseId',
                        [
                            'nullBinLocationId' => $nullBinLocationId,
                            'warehouseId' => $defaultWarehouseId,
                        ]
                    );

                    // Add mappings to the warehouse's null bin location for all article details
                    $database->query(
                        'INSERT IGNORE INTO s_plugin_pickware_warehouse_bin_location_article_details (binLocationId, articleDetailId)
                            SELECT :nullBinLocationId, articleDetail.id
                            FROM s_articles_details AS articleDetail',
                        [
                            'nullBinLocationId' => $nullBinLocationId,
                        ]
                    );

                    // Create the cached warehouse stocks for all bin location article details
                    $database->query(
                        'INSERT IGNORE INTO s_plugin_pickware_warehouse_stocks (warehouseId, articleDetailId)
                            SELECT :warehouseId, articleDetail.id
                            FROM s_articles_details AS articleDetail',
                        [
                            'warehouseId' => $defaultWarehouseId,
                        ]
                    );
                }

                // Update Shopware snippets and mark them dirty
                $this->loadSnippetReplacementsToDB();

            /* Incremental Updates */

            case self::MIN_SUPPORTED_UPDATE_VERSION:
                // Nothing to do
            case '4.0.14':
                // Nothing to do
            case '4.0.15':
                // Nothing to do
            case '4.0.16':
                // Nothing to do
            case '4.0.17':
                // Nothing to do
            case '4.0.18':
                // Nothing to do
            case '4.0.19':
                // Nothing to do
            case '4.0.20':
                // Nothing to do
            case '4.0.21':
                // Nothing to do
            case '4.0.22':
                // Set s_plugin_pickware_warehouse_bin_location_article_details.reservedStock to 0 by default
                $sqlHelper->alterColumnIfExists(
                    's_plugin_pickware_warehouse_bin_location_article_details',
                    'reservedStock',
                    'reservedStock',
                    'int(11) NOT NULL DEFAULT \'0\''
                );
                // Set s_plugin_pickware_warehouse_bin_location_article_details.defaultMapping to 0 by default
                $sqlHelper->alterColumnIfExists(
                    's_plugin_pickware_warehouse_bin_location_article_details',
                    'defaultMapping',
                    'defaultMapping',
                    'tinyint(1) NOT NULL DEFAULT \'0\''
                );
                // Add any missing bin location article detail mappings in all warehouses
                $warehouses = $database->fetchAll(
                    'SELECT
                        `id` AS `warehouseId`,
                        `defaultBinLocationId` AS `nullBinLocationId`
                    FROM `s_plugin_pickware_warehouses`'
                );
                foreach ($warehouses as $warehouse) {
                    // Add a mapping to the warehouse's null bin location, if no mappings exist
                    $database->query(
                        'INSERT IGNORE INTO s_plugin_pickware_warehouse_bin_location_article_details (binLocationId, articleDetailId)
                        SELECT :nullBinLocationId, ad.id
                        FROM s_articles_details ad
                        WHERE ad.id NOT IN (
                            SELECT tmp.id
                            FROM (
                                -- Find all article details with at least one mapping in the same warehouse
                                SELECT inner_wblad.articleDetailId AS id
                                FROM s_plugin_pickware_warehouse_bin_location_article_details inner_wblad
                                INNER JOIN s_plugin_pickware_warehouse_bin_locations wbl
                                    ON wbl.id = inner_wblad.binLocationId
                                WHERE wbl.warehouseId = :warehouseId
                                GROUP BY inner_wblad.articleDetailId
                                HAVING COUNT(inner_wblad.id) > 0
                            ) tmp
                        )',
                        $warehouse
                    );
                }

                // Add sub shop id to supplier model table. Also add a default sub shop id for suppliers to which no
                // "preferred" sub shop can be assigned later on
                $sqlHelper->addColumnIfNotExists(
                    's_plugin_pickware_suppliers',
                    'documentLocalizationSubShopId',
                    'int(11) unsigned NULL'
                );
                $sqlHelper->addIndexIfNotExists(
                    's_plugin_pickware_suppliers',
                    $sqlHelper->getConstraintIdentifier('s_plugin_pickware_suppliers', 'documentLocalizationSubShopId', 'idx'),
                    [
                        'documentLocalizationSubShopId'
                    ]
                );
                // Clear any documentLocalizationSubShopId values, whose associated shop does not exist anymore
                $database->query(
                    'UPDATE s_plugin_pickware_suppliers supplier
                    LEFT JOIN s_core_shops shop
                        ON supplier.documentLocalizationSubShopId = shop.id
                    SET supplier.documentLocalizationSubShopId = NULL
                    WHERE shop.id IS NULL'
                );
                // Add a foreign key constraint to s_plugin_pickware_suppliers.documentLocalizationSubShopId
                $sqlHelper->ensureForeignKeyConstraint(
                    's_plugin_pickware_suppliers',
                    'documentLocalizationSubShopId',
                    's_core_shops',
                    'id',
                    'SET NULL' // ON DELETE action
                );
                // Add a "preferred" sub shop to each supplier that is a sub shop with the same locale as the supplier
                // and has the lowest id among shops with the same locale. This migration can only be done if the
                // languageId field existed before (which is not the case in a fresh installation)
                if ($sqlHelper->doesColumnExist('s_plugin_pickware_suppliers', 'languageId')) {
                    $database->query(
                        'UPDATE s_plugin_pickware_suppliers supplier
                        LEFT JOIN s_core_shops shop
                            ON supplier.languageId = shop.locale_id
                        SET supplier.documentLocalizationSubShopId = shop.id
                        WHERE shop.id IN (
                            SELECT MIN(s_core_shops.id)
                            FROM s_core_shops
                            WHERE s_core_shops.locale_id = supplier.languageId
                        )'
                    );
                    // Remove supplier language field, since it is not necessary anymore
                    $sqlHelper->dropColumnIfExists('s_plugin_pickware_suppliers', 'languageId');
                }
            case '4.0.23':
                // Update SwagImportExport profiles if necessary
                if (ViisonCommonUtil::isPluginInstalledAndActive('Backend', 'SwagImportExport')) {
                    /** @var SwagImportExportProfileService $profileLoader */
                    $profileLoader = $this->get('viison_common.swag_import_export_profile_service');
                    $profileLoader->updateExistingProfiles(
                        SwagImportExportIntegrationSubscriber::PROFILE_NAMES,
                        $this->Path() . 'Components/SwagImportExportIntegration/Profiles/'
                    );
                }
            case '4.0.24':
                // Nothing to do
            case '4.0.25':
                // Nothing to do
            case '4.0.26':
                // Nothing to do
            case '4.0.27':
                // Nothing to do
            case '4.0.28':
                // Nothing to do
            case '4.0.29':
                // Nothing to do
            case '4.0.30':
                // Nothing to do
            case '4.0.31':
                // Nothing to do
            case '4.0.32':
                // Clear any stock reservations that don't have an associated order detail
                $database->query(
                    'DELETE FROM s_plugin_pickware_warehouse_reserved_stocks
                    WHERE orderDetailId IS NULL'
                );
                // Update the cached reserved stocks for all article details in all warehouses
                $database->query(
                    'UPDATE s_plugin_pickware_warehouse_bin_location_article_details binLocationMappings
                    LEFT JOIN (
                        SELECT
                            binLocationArticleDetailId AS binLocationMappingId,
                            SUM(quantity) AS reservedStock
                        FROM s_plugin_pickware_warehouse_reserved_stocks
                        GROUP BY binLocationArticleDetailId
                    ) AS reservedStocks
                        ON reservedStocks.binLocationMappingId = binLocationMappings.id
                    SET binLocationMappings.reservedStock = IFNULL(reservedStocks.reservedStock, 0)'
                );
                $database->query(
                    'UPDATE s_plugin_pickware_warehouse_stocks warehouseStocks
                    LEFT JOIN (
                        SELECT
                            binLocations.warehouseId AS warehouseId,
                            binLocationMappings.articleDetailId AS articleDetailId,
                            SUM(binLocationMappings.reservedStock) AS reservedStock
                        FROM s_plugin_pickware_warehouse_bin_location_article_details binLocationMappings
                        LEFT JOIN s_plugin_pickware_warehouse_bin_locations binLocations
                            ON binLocations.id = binLocationMappings.binLocationId
                        GROUP BY
                            binLocations.warehouseId,
                            binLocationMappings.articleDetailId
                    ) AS reservedStocks
                        ON reservedStocks.warehouseId = warehouseStocks.warehouseId
                        AND reservedStocks.articleDetailId = warehouseStocks.articleDetailId
                    SET warehouseStocks.reservedStock = IFNULL(reservedStocks.reservedStock, 0)'
                );

                // Ensure unique supplier number column and number sequence exists. Fill empty supplier numbers if
                // needed.
                $orderNumberInstallationHelper->createSequenceNumberGeneratorIfNotExists(
                    SupplierNumberGeneratorService::NUMBER_RANGE_NAME,
                    SupplierNumberGeneratorService::START_NUMBER,
                    SupplierNumberGeneratorService::NUMBER_RANGE_DESCRIPTION
                );
                if (!$sqlHelper->doesColumnExist('s_plugin_pickware_suppliers', 'number')) {
                    $sqlHelper->addColumnIfNotExists(
                        's_plugin_pickware_suppliers',
                        'number',
                        'VARCHAR(255)'
                    );

                    // Update all empty supplier numbers with incrementing, unused numbers. Since the column was added
                    // just now, we can assume that all numbers are currently empty and simple generate as many numbers
                    // as required.
                    $curentSupplierNumber = $database->fetchOne(
                        'SELECT `number`
                        FROM `s_order_number`
                        WHERE `name` = :numberRangeName',
                        [
                            'numberRangeName' => SupplierNumberGeneratorService::NUMBER_RANGE_NAME,
                        ]
                    );
                    $database->query(
                        'SET @supplierNumber = :nextSupplierNumber;
                        UPDATE s_plugin_pickware_suppliers
                        SET number = @supplierNumber := @supplierNumber + 1;',
                        [
                            'nextSupplierNumber' => intval($curentSupplierNumber) + 1,
                        ]
                    );
                    // Update the number range again to speed up the generation of the next number by
                    // avoiding collisions
                    $database->query(
                        'UPDATE `s_order_number`
                        SET `number` = (
                            SELECT MAX(`number`)
                            FROM `s_plugin_pickware_suppliers`
                        )
                        WHERE `name` = :numberRangeName',
                        [
                            'numberRangeName' => SupplierNumberGeneratorService::NUMBER_RANGE_NAME,
                        ]
                    );

                    // Add not null and unique constraint for productive use of the number column
                    $sqlHelper->addNotNullConstraint('s_plugin_pickware_suppliers', 'number', 0);
                    $sqlHelper->addUniqueConstraintIfNotExists('s_plugin_pickware_suppliers', ['number']);
                }

                // Add attributes 'viison_stock_initialized' and 'viison_stock_initialization_time' to save whether and
                // when the stock of an article detail was initialized
                $attributeColumnInstallationHelper->addAttributeColumnsIfNotExist([
                    // This column will be created as nullable with null as default value, which allows us to migrate
                    // the field for all existing article details in an idempotent manner. The nullability constraint
                    // will be added after their migration.
                    new AttributeColumnDescription(
                        's_articles_attributes',
                        'viison_stock_initialized',
                        'TINYINT(1)',
                        null,
                        true
                    ),
                    new AttributeColumnDescription(
                        's_articles_attributes',
                        'viison_stock_initialization_time',
                        'DATETIME',
                        null,
                        true
                    ),
                ]);
                // Set the initialization attributes for all existing article details based on their
                // stock entries
                $database->query(
                    'UPDATE s_articles_attributes AS attributes
                    LEFT JOIN (
                        SELECT
                            stocks.articleDetailId AS articleDetailId,
                            MIN(stocks.created) AS initTime
                        FROM s_plugin_pickware_stocks AS stocks
                        WHERE type = :typeInitialization
                        GROUP BY stocks.articleDetailId
                    ) AS initStockEntry
                        ON initStockEntry.articleDetailId = attributes.articleDetailsID
                    SET
                        attributes.viison_stock_initialized = IF(initStockEntry.initTime IS NULL, 0, 1),
                        attributes.viison_stock_initialization_time = initStockEntry.initTime
                    WHERE attributes.viison_stock_initialized IS NULL',
                    [
                        'typeInitialization' => StockLedgerEntry::TYPE_INITIALIZATION,
                    ]
                );
                // Add the nullability constraint to the 'viison_stock_initialized' attribute
                $sqlHelper->alterColumnIfExists(
                    's_articles_attributes',
                    'viison_stock_initialized',
                    'viison_stock_initialized',
                    'TINYINT(1) NOT NULL DEFAULT 0'
                );
                // We need to re-generate the attribute model explicitly since we've changed the nullability specifier
                $modelManager->generateAttributeModels(['s_articles_attributes']);

                // Add a foreign key constraint from return shipment attachments to return shipments
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    's_plugin_pickware_reshipment_attachments',
                    'reshipmentId',
                    's_plugin_pickware_reshipments',
                    'id'
                );
                // Add a foreign key constraint from return shipment attachments to media items
                $sqlHelper->cleanUpAndEnsureRestrictingForeignKeyConstraint(
                    's_plugin_pickware_reshipment_attachments',
                    'mediaId',
                    's_media',
                    'id'
                );
                // Fix values before adding the foreign key constraint from return shipments to return shipment status
                $database->query(
                    'UPDATE s_plugin_pickware_reshipments
                    SET statusId = :statusReceived
                    WHERE statusId NOT IN (:statusReceived, :statusCompleted)',
                    [
                        'statusCompleted' => ReturnShipmentStatus::STATUS_COMPLETED_ID,
                        'statusReceived' => ReturnShipmentStatus::STATUS_RECEIVED_ID,
                    ]
                );
                // Add a foreign key constraint from return shipments to return shipment status
                $sqlHelper->ensureForeignKeyConstraint(
                    's_plugin_pickware_reshipments',
                    'statusId',
                    's_plugin_pickware_reshipment_status',
                    'id'
                );
                // Remove stale warehouse stocks before changing the column type of
                // 's_plugin_pickware_warehouse_stocks.articleDetailId' to prevent errors
                $sqlHelper->cleanUpForRestrictingOrCascadingForeignKeyConstraint(
                    's_plugin_pickware_warehouse_stocks',
                    'articleDetailId',
                    's_articles_details',
                    'id'
                );
                // Change the column type of 's_plugin_pickware_warehouse_stocks.articleDetailId' to allow a foreign
                // key constraint
                $sqlHelper->alterColumnIfExists(
                    's_plugin_pickware_warehouse_stocks',
                    'articleDetailId',
                    'articleDetailId',
                    'int(11) unsigned NOT NULL'
                );
                // Add a foreign key constraint from warehouse stocks to article details
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    's_plugin_pickware_warehouse_stocks',
                    'articleDetailId',
                    's_articles_details',
                    'id'
                );
                // Add missing default value of 0 to the 's_plugin_pickware_warehouse_stocks.stock'
                $sqlHelper->alterColumnIfExists(
                    's_plugin_pickware_warehouse_stocks',
                    'stock',
                    'stock',
                    'int(11) NOT NULL DEFAULT \'0\''
                );
                // Add missing default value of 0 to the 's_plugin_pickware_warehouse_stocks.reservedStock'
                $sqlHelper->alterColumnIfExists(
                    's_plugin_pickware_warehouse_stocks',
                    'reservedStock',
                    'reservedStock',
                    'int(11) NOT NULL DEFAULT \'0\''
                );
                // Remove stale bin location mappings before changing the column type of
                // 's_plugin_pickware_warehouse_bin_location_article_details.articleDetailId' to prevent errors
                $sqlHelper->cleanUpForRestrictingOrCascadingForeignKeyConstraint(
                    's_plugin_pickware_warehouse_bin_location_article_details',
                    'articleDetailId',
                    's_articles_details',
                    'id'
                );
                // Change the column type of 's_plugin_pickware_warehouse_bin_location_article_details.articleDetailId'
                // to allow a foreign key constraint
                $sqlHelper->alterColumnIfExists(
                    's_plugin_pickware_warehouse_bin_location_article_details',
                    'articleDetailId',
                    'articleDetailId',
                    'int(11) unsigned NOT NULL'
                );
                // Add a foreign key constraint from warehouse stocks to article details
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    's_plugin_pickware_warehouse_bin_location_article_details',
                    'articleDetailId',
                    's_articles_details',
                    'id'
                );
                // Ensure s_plugin_pickware_suppliers uses InnoDB
                // Due to a bug in v4.0.23 where a CREATE TABLE IF NOT EXISTS statement omitted the engine
                // specification, some customers have a MyISAM table here, which will cause constraint errors in the
                // following migrations. Because of this, the following line was added with v5.0.0 (after this migration
                // step was released initially).
                $database->query('ALTER TABLE s_plugin_pickware_suppliers ENGINE=InnoDB');
                // Update the foreign key constraint from supplier order attachments to supplier orders
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    's_plugin_pickware_supplier_order_attachments',
                    'orderId',
                    's_plugin_pickware_supplier_orders',
                    'id'
                );
                // Add a foreign key constraint from supplier orders attachments to media items
                $sqlHelper->cleanUpAndEnsureRestrictingForeignKeyConstraint(
                    's_plugin_pickware_supplier_order_attachments',
                    'mediaId',
                    's_media',
                    'id'
                );
                // Add a foreign key constraint from supplier article mappings to suppliers
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    's_plugin_pickware_supplier_article_details',
                    'supplierId',
                    's_plugin_pickware_suppliers',
                    'id'
                );
                // Add a foreign key constraint from supplier fabricator mappings to suppliers
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    's_plugin_pickware_supplier_fabricators',
                    'supplierId',
                    's_plugin_pickware_suppliers',
                    'id'
                );
                // Add a foreign key constraint from supplier fabricator mappings to fabricators
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    's_plugin_pickware_supplier_fabricators',
                    'fabricatorId',
                    's_articles_supplier',
                    'id'
                );
                // Remove stale stock entries before changing the column type of
                // 's_plugin_pickware_stocks.articleDetailId' to prevent errors
                $sqlHelper->cleanUpForRestrictingOrCascadingForeignKeyConstraint(
                    's_plugin_pickware_stocks',
                    'articleDetailId',
                    's_articles_details',
                    'id'
                );
                // Change the column type of 's_plugin_pickware_stocks.articleDetailId' to allow a foreign
                // key constraint
                $sqlHelper->alterColumnIfExists(
                    's_plugin_pickware_stocks',
                    'articleDetailId',
                    'articleDetailId',
                    'int(11) unsigned NOT NULL'
                );
                // Add a foreign key constraint from stock entries to article details
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    's_plugin_pickware_stocks',
                    'articleDetailId',
                    's_articles_details',
                    'id'
                );
                // Add a foreign key constraint from stock entries to order details
                $sqlHelper->cleanUpAndEnsureNullingForeignKeyConstraint(
                    's_plugin_pickware_stocks',
                    'orderDetailId',
                    's_order_details',
                    'id'
                );
                // Add a foreign key constraint from stock entries to (backend) users
                $sqlHelper->cleanUpAndEnsureNullingForeignKeyConstraint(
                    's_plugin_pickware_stocks',
                    'userId',
                    's_core_auth',
                    'id'
                );
            case '4.1.0':
                // Update the foreign key constraint from stock entries to supplier order details, before cleaning up
                // any supplier order details
                $sqlHelper->cleanUpAndEnsureNullingForeignKeyConstraint(
                    's_plugin_pickware_stocks',
                    'supplierOrderItemId',
                    's_plugin_pickware_supplier_order_article_details',
                    'id'
                );
                // Update the foreign key constraint from supplier order items to supplier orders
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    's_plugin_pickware_supplier_order_article_details',
                    'orderId',
                    's_plugin_pickware_supplier_orders',
                    'id'
                );
                // Update the foreign key constraint from stock entries to return shipment items, before cleaning up any
                // return shipment items
                $sqlHelper->cleanUpAndEnsureNullingForeignKeyConstraint(
                    's_plugin_pickware_stocks',
                    'reshipmentItemId',
                    's_plugin_pickware_reshipment_items',
                    'id'
                );
                // Add a foreign key constraint from return shipment items to return shipments
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    's_plugin_pickware_reshipment_items',
                    'reshipmentId',
                    's_plugin_pickware_reshipments',
                    'id'
                );
                // Add a foreign key constraint from return shipment items to order details
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    's_plugin_pickware_reshipment_items',
                    'orderDetailId',
                    's_order_details',
                    'id'
                );
            case '4.1.1':
                // Add a foreign key constraint from supplier orders to suppliers, after cleaning up any supplier order
                // items and attachments
                $sqlHelper->cleanUpAndEnsureRestrictingForeignKeyConstraint(
                    's_plugin_pickware_supplier_orders',
                    'supplierId',
                    's_plugin_pickware_suppliers',
                    'id'
                );
                // Add a foreign key constraint from return shipments to orders, after cleaning up any return shipment
                // items and attachments
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    's_plugin_pickware_reshipments',
                    'orderId',
                    's_order',
                    'id'
                );
                // Allow 's_plugin_pickware_reshipments.userId' to be nullable
                $sqlHelper->alterColumnIfExists(
                    's_plugin_pickware_reshipments',
                    'userId',
                    'userId',
                    'int(11) DEFAULT NULL'
                );
                // Add a foreign key constraint from return shipments to users, after cleaning up any return shipment
                // items and attachments
                $sqlHelper->cleanUpAndEnsureNullingForeignKeyConstraint(
                    's_plugin_pickware_reshipments',
                    'userId',
                    's_core_auth',
                    'id'
                );
            case '4.1.2':
                // Regenerate attributes for article details because we forgot to do this after an 'ALTER TABLE' in the
                // previous update step (case '4.0.32'). That caused doctrine attribute models to lack a non-nullability
                // constraint which was present on the database, which in turn broke variant generation.
                $modelManager->generateAttributeModels(['s_articles_attributes']);

                // Fix backend sessions
                $backendSessionMigration = new BackendSessionLocaleClassMigration(
                    $this->get('models'),
                    $this->get('db'),
                    $this->get('viison_common.logger'),
                    $this->get('application')->Container()
                );
                $backendSessionMigration->fixSessions();
            case '4.1.3':
                // Nothing to do
            case '4.1.4':
                // Nothing to do
            case '4.1.5':
                // Generate attributes for articles without attributes and repeat the stock initialization migration.
                // Remark: The migration in 4.2.0 used to be in this update step, but contained an error. We moved the
                // fixed migration into the 4.2.0 update step.
                $database->query(
                    'INSERT IGNORE INTO s_articles_attributes (articledetailsID)
                        SELECT id from s_articles_details'
                );
            case '4.2.0':
                // Fix of articles as 'initialized' that was done in the update to 4.0.32. Remark: This SQL is a fixed
                // version of the migration that used to be in the 4.1.5 update step.
                $database->query(
                    'UPDATE s_articles_attributes AS attributes
                    LEFT JOIN (
                        SELECT
                            stocks.articleDetailId AS articleDetailId,
                            MIN(stocks.created) AS initTime
                        FROM s_plugin_pickware_stocks AS stocks
                        WHERE type = :typeInitialization
                        GROUP BY stocks.articleDetailId
                    ) AS initStockEntry
                        ON initStockEntry.articleDetailId = attributes.articleDetailsID
                    SET
                        attributes.viison_stock_initialized = IF(initStockEntry.initTime IS NULL, 0, 1),
                        attributes.viison_stock_initialization_time = initStockEntry.initTime
                    WHERE attributes.viison_stock_initialized = 0',
                    [
                        'typeInitialization' => StockLedgerEntry::TYPE_INITIALIZATION,
                    ]
                );
            case '4.2.1':
                // Add columns to store the minimum stock, target stock and incoming stock
                // on an per article detail / warehouse base
                $sqlHelper->addColumnIfNotExists(
                    's_plugin_pickware_warehouse_stocks',
                    'minimumStock',
                    'INT(11) NOT NULL DEFAULT 0'
                );
                $sqlHelper->addColumnIfNotExists(
                    's_plugin_pickware_warehouse_stocks',
                    'targetStock',
                    'INT(11) NOT NULL DEFAULT 0'
                );
                $sqlHelper->addColumnIfNotExists(
                    's_plugin_pickware_warehouse_stocks',
                    'incomingStock',
                    'INT(11) NOT NULL DEFAULT 0'
                );

                // In order to keep the update method idempotent, only migrate minimum stock values
                // for articles, whose minimum stock value is not set yet
                $articleDetailIdsToMigrate = $database->fetchCol(
                    'SELECT articleDetailId
                        FROM s_plugin_pickware_warehouse_stocks
                        GROUP BY articleDetailId
                        HAVING MAX(minimumStock) = 0'
                );

                // Migrate minimum stock values (for all article details set the minimun stock and target stock values
                // of the default warehouse to the current stockMin value of the article detail )
                $defaultWarehouseId = $database->fetchOne(
                    'SELECT `id`
                    FROM `s_plugin_pickware_warehouses`
                    WHERE `defaultWarehouse` = 1'
                );
                $this->get('dbal_connection')->executeQuery(
                    'UPDATE s_plugin_pickware_warehouse_stocks warehouseStock
                    LEFT JOIN s_articles_details articleDetail ON articleDetail.id = warehouseStock.articleDetailId
                    SET
                        warehouseStock.minimumStock = articleDetail.stockmin,
                        warehouseStock.targetStock = articleDetail.stockmin
                    WHERE
                        warehouseStock.warehouseId = :warehouseId
                        AND warehouseStock.articleDetailId IN (:articleDetailIds)',
                    [
                        'warehouseId' => $defaultWarehouseId,
                        'articleDetailIds' => $articleDetailIdsToMigrate,
                    ],
                    [
                        'articleDetailIds' => \Doctrine\DBAL\Connection::PARAM_INT_ARRAY,
                    ]
                );
            case '4.3.0':
                // Nothing to do
            case '4.3.1':
                // Nothing to do
            case '4.3.2':
                // Add a checkpoint to ensure none of the migrations above this version are executed after the tables
                // are renamed
                $installationCheckpointWriter->commitCheckpointAtVersion('4.3.1');

                // Add pick list document type
                $this->ensurePickListDocumentType();

                // Drop the foreign key constrains of all (old) tables of this plugin
                $erpTableNames = [
                    's_plugin_pickware_article_detail_property_types',
                    's_plugin_pickware_property_types',
                    's_plugin_pickware_reshipment_attachments',
                    's_plugin_pickware_reshipment_items',
                    's_plugin_pickware_reshipment_status',
                    's_plugin_pickware_reshipments',
                    's_plugin_pickware_stock_entry_item_properties',
                    's_plugin_pickware_stock_entry_items',
                    's_plugin_pickware_stocks',
                    's_plugin_pickware_supplier_article_details',
                    's_plugin_pickware_supplier_fabricators',
                    's_plugin_pickware_supplier_order_article_detail_states',
                    's_plugin_pickware_supplier_order_article_details',
                    's_plugin_pickware_supplier_order_attachments',
                    's_plugin_pickware_supplier_order_states',
                    's_plugin_pickware_supplier_orders',
                    's_plugin_pickware_suppliers',
                    's_plugin_pickware_warehouse_bin_location_article_details',
                    's_plugin_pickware_warehouse_bin_location_stock_snapshots',
                    's_plugin_pickware_warehouse_bin_locations',
                    's_plugin_pickware_warehouse_reserved_stocks',
                    's_plugin_pickware_warehouse_stocks',
                    's_plugin_pickware_warehouses',
                ];
                $erpTableConstraints = $this->get('dbal_connection')->fetchAll(
                    'SELECT
                        `TABLE_NAME` AS `tableName`,
                        `COLUMN_NAME` AS `columnName`,
                        `REFERENCED_TABLE_NAME` AS `referencedTableName`,
                        `REFERENCED_COLUMN_NAME` AS `referencedColumnName`
                    FROM `information_schema`.`KEY_COLUMN_USAGE`
                    WHERE
                        `TABLE_SCHEMA` = (SELECT DATABASE())
                        AND `TABLE_NAME` IN (:erpTableNames)
                        AND `REFERENCED_TABLE_SCHEMA` = `TABLE_SCHEMA`
                        AND `REFERENCED_TABLE_NAME` IS NOT NULL
                        AND `REFERENCED_COLUMN_NAME` IS NOT NULL',
                    [
                        'erpTableNames' => $erpTableNames,
                    ],
                    [
                        'erpTableNames' => DbalConnection::PARAM_STR_ARRAY,
                    ]
                );
                foreach ($erpTableConstraints as $constraint) {
                    $sqlHelper->dropForeignKeyConstraintIfExists(
                        $constraint['tableName'],
                        $constraint['columnName'],
                        $constraint['referencedTableName'],
                        $constraint['referencedColumnName']
                    );
                }

                // Rename all ERP tables in an idempotency-compatible manner
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_article_detail_property_types', 'pickware_erp_article_detail_item_properties');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_property_types', 'pickware_erp_item_properties');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_reshipment_attachments', 'pickware_erp_return_shipment_attachments');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_reshipment_items', 'pickware_erp_return_shipment_items');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_reshipment_status', 'pickware_erp_return_shipment_statuses');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_reshipments', 'pickware_erp_return_shipments');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_stock_entry_item_properties', 'pickware_erp_stock_item_property_values');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_stock_entry_items', 'pickware_erp_stock_items');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_stocks', 'pickware_erp_stock_ledger_entries');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_supplier_article_details', 'pickware_erp_article_detail_supplier_mappings');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_supplier_fabricators', 'pickware_erp_manufacturer_supplier_mappings');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_supplier_order_article_detail_states', 'pickware_erp_supplier_order_item_statuses');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_supplier_order_article_details', 'pickware_erp_supplier_order_items');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_supplier_order_attachments', 'pickware_erp_supplier_order_attachments');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_supplier_order_states', 'pickware_erp_supplier_order_statuses');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_supplier_orders', 'pickware_erp_supplier_orders');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_suppliers', 'pickware_erp_suppliers');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_warehouse_bin_location_article_details', 'pickware_erp_article_detail_bin_location_mappings');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_warehouse_bin_location_stock_snapshots', 'pickware_erp_bin_location_stock_snapshots');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_warehouse_bin_locations', 'pickware_erp_bin_locations');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_warehouse_reserved_stocks', 'pickware_erp_order_stock_reservations');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_warehouse_stocks', 'pickware_erp_warehouse_article_detail_stock_counts');
                $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_warehouses', 'pickware_erp_warehouses');

                // Rename some non-attribute columns in an idempotency-compatible manner
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_article_detail_item_properties',
                    'propertyTypeId',
                    'itemPropertyId',
                    'int(11) NOT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_bin_location_stock_snapshots',
                    'stockEntryId',
                    'stockLedgerEntryId',
                    'int(11) NOT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_manufacturer_supplier_mappings',
                    'fabricatorId',
                    'manufacturerId',
                    'int(11) NOT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_order_stock_reservations',
                    'binLocationArticleDetailId',
                    'articleDetailBinLocationMappingId',
                    'int(11) NOT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_return_shipment_attachments',
                    'reshipmentId',
                    'returnShipmentId',
                    'int(11) NOT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_return_shipment_items',
                    'depreciatedQuantity',
                    'writtenOffQuantity',
                    'int(11) NOT NULL DEFAULT \'0\''
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_return_shipment_items',
                    'reshipmentId',
                    'returnShipmentId',
                    'int(11) NOT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_stock_item_property_values',
                    'propertyTypeId',
                    'itemPropertyId',
                    'int(11) NOT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_stock_item_property_values',
                    'stockEntryItemId',
                    'stockItemId',
                    'int(11) NOT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_stock_items',
                    'stockEntryId',
                    'stockLedgerEntryId',
                    'int(11) NOT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_stock_ledger_entries',
                    'newInstockQuantity',
                    'newStock',
                    'int(11) NOT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_stock_ledger_entries',
                    'oldInstockQuantity',
                    'oldStock',
                    'int(11) NOT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_stock_ledger_entries',
                    'parentId',
                    'sourceLotEntryId',
                    'int(11) DEFAULT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_stock_ledger_entries',
                    'reshipmentItemId',
                    'returnShipmentItemId',
                    'int(11) DEFAULT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_supplier_order_attachments',
                    'orderId',
                    'supplierOrderId',
                    'int(11) NOT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_supplier_order_items',
                    'fabricatorName',
                    'manufacturerName',
                    'varchar(255) DEFAULT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_supplier_order_items',
                    'fabricatorNumber',
                    'manufacturerArticleNumber',
                    'varchar(255) DEFAULT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_supplier_order_items',
                    'orderId',
                    'supplierOrderId',
                    'int(11) DEFAULT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_supplier_order_items',
                    'status',
                    'statusId',
                    'int(11) NOT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_supplier_orders',
                    'status',
                    'statusId',
                    'int(11) NOT NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_warehouses',
                    'stockAvailable',
                    'stockAvailableForSale',
                    'tinyint(1) NOT NULL DEFAULT \'1\''
                );

                // Update the code of the 'null bin location'
                $database->query(
                    'UPDATE `pickware_erp_bin_locations`
                    SET `code` = :nullBinLocationCode
                    WHERE `code` = \'pickware_default_location\'',
                    [
                        'nullBinLocationCode' => BinLocation::NULL_BIN_LOCATION_CODE,
                    ]
                );

                // Don't allow `pickware_erp_order_stock_reservations.orderDetail` to be null anymore
                $database->query(
                    'DELETE FROM `pickware_erp_order_stock_reservations`
                    WHERE `orderDetailId` IS NULL'
                );
                $database->query(
                    'ALTER TABLE `pickware_erp_order_stock_reservations`
                    CHANGE `orderDetailId` `orderDetailId` int(11) NOT NULL'
                );

                // Add a foreign key constraint from article detail item property mappings to article details
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_article_detail_item_properties',
                    'articleDetailId',
                    's_articles_details',
                    'id'
                );
                // Add a foreign key constraint from article detail item property mappings to item properties
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_article_detail_item_properties',
                    'itemPropertyId',
                    'pickware_erp_item_properties',
                    'id'
                );
                // Add a foreign key constraint from stock item property values to item properties
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_stock_item_property_values',
                    'itemPropertyId',
                    'pickware_erp_item_properties',
                    'id'
                );
                // Add a foreign key constraint from stock item property values to stock items
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_stock_item_property_values',
                    'stockItemId',
                    'pickware_erp_stock_items',
                    'id'
                );
                // Add a foreign key constraint from stock items to stock ledger entries
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_stock_items',
                    'stockLedgerEntryId',
                    'pickware_erp_stock_ledger_entries',
                    'id'
                );
                // Add a foreign key constraint from stock ledger entries to article details
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_stock_ledger_entries',
                    'articleDetailId',
                    's_articles_details',
                    'id'
                );
                // Add a foreign key constraint from stock ledger entries to warehouses
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_stock_ledger_entries',
                    'warehouseId',
                    'pickware_erp_warehouses',
                    'id'
                );
                // Add a foreign key constraint from stock ledger entries to stock ledger entries that forms the
                // source lot entry relationship, but don't clean up anything (the column is nullable)
                $sqlHelper->ensureForeignKeyConstraint(
                    'pickware_erp_stock_ledger_entries',
                    'sourceLotEntryId',
                    'pickware_erp_stock_ledger_entries',
                    'id',
                    'CASCADE'
                );
                // Add a foreign key constraint from stock ledger entries to (backend) users
                $sqlHelper->cleanUpAndEnsureNullingForeignKeyConstraint(
                    'pickware_erp_stock_ledger_entries',
                    'userId',
                    's_core_auth',
                    'id'
                );
                // Add a foreign key constraint from stock ledger entries to order details
                $sqlHelper->cleanUpAndEnsureNullingForeignKeyConstraint(
                    'pickware_erp_stock_ledger_entries',
                    'orderDetailId',
                    's_order_details',
                    'id'
                );
                // Add a foreign key constraint from stock ledger entries to supplier order items
                $sqlHelper->cleanUpAndEnsureNullingForeignKeyConstraint(
                    'pickware_erp_stock_ledger_entries',
                    'supplierOrderItemId',
                    'pickware_erp_supplier_order_items',
                    'id'
                );
                // Add a foreign key constraint from stock ledger entries to return shipment items
                $sqlHelper->cleanUpAndEnsureNullingForeignKeyConstraint(
                    'pickware_erp_stock_ledger_entries',
                    'returnShipmentItemId',
                    'pickware_erp_return_shipment_items',
                    'id'
                );
                // Add a foreign key constraint from return shipment attachments to return shipments
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_return_shipment_attachments',
                    'returnShipmentId',
                    'pickware_erp_return_shipments',
                    'id'
                );
                // Add a restricting foreign key constraint from return shipment attachments to media items
                $sqlHelper->cleanUpAndEnsureRestrictingForeignKeyConstraint(
                    'pickware_erp_return_shipment_attachments',
                    'mediaId',
                    's_media',
                    'id'
                );
                // Add a foreign key constraint from return shipment items to return shipments
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_return_shipment_items',
                    'returnShipmentId',
                    'pickware_erp_return_shipments',
                    'id'
                );
                // Add a foreign key constraint from return shipment items to order details
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_return_shipment_items',
                    'orderDetailId',
                    's_order_details',
                    'id'
                );
                // Add a restricting foreign key constraint from return shipments to return shipment statuses, without
                // cleaning up beforehand
                $sqlHelper->ensureForeignKeyConstraint(
                    'pickware_erp_return_shipments',
                    'statusId',
                    'pickware_erp_return_shipment_statuses',
                    'id'
                );
                // Add a foreign key constraint from return shipments to orders
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_return_shipments',
                    'orderId',
                    's_order',
                    'id'
                );
                // Add a foreign key constraint from return shipments to (backend) users
                $sqlHelper->cleanUpAndEnsureNullingForeignKeyConstraint(
                    'pickware_erp_return_shipments',
                    'userId',
                    's_core_auth',
                    'id'
                );
                // Add a foreign key constraint from supplier order attachments to supplier orders
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_supplier_order_attachments',
                    'supplierOrderId',
                    'pickware_erp_supplier_orders',
                    'id'
                );
                // Add a restricting foreign key constraint from supplier orders attachments to media items
                $sqlHelper->cleanUpAndEnsureRestrictingForeignKeyConstraint(
                    'pickware_erp_supplier_order_attachments',
                    'mediaId',
                    's_media',
                    'id'
                );
                // Add a foreign key constraint from supplier order items to supplier orders
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_supplier_order_items',
                    'supplierOrderId',
                    'pickware_erp_supplier_orders',
                    'id'
                );
                // Add a restricting foreign key constraint from supplier order items to supplier order item statuses
                $sqlHelper->cleanUpAndEnsureRestrictingForeignKeyConstraint(
                    'pickware_erp_supplier_order_items',
                    'statusId',
                    'pickware_erp_supplier_order_item_statuses',
                    'id'
                );
                // Update stale supplier order items before changing the column type of
                // 'pickware_erp_supplier_order_items.articleDetailId' to prevent errors
                $sqlHelper->cleanUpForNullingForeignKeyConstraint(
                    'pickware_erp_supplier_order_items',
                    'articleDetailId',
                    's_articles_details',
                    'id'
                );
                // Change the column type of 'pickware_erp_supplier_order_items.articleDetailId' to allow a foreign
                // key constraint
                $sqlHelper->alterColumnIfExists(
                    'pickware_erp_supplier_order_items',
                    'articleDetailId',
                    'articleDetailId',
                    'int(11) unsigned DEFAULT NULL'
                );
                // Add a foreign key constraint from supplier order items to article details
                $sqlHelper->cleanUpAndEnsureNullingForeignKeyConstraint(
                    'pickware_erp_supplier_order_items',
                    'articleDetailId',
                    's_articles_details',
                    'id'
                );
                // Add a restricting foreign key constraint from supplier orders to suppliers, but don't clean up
                // anything (the column is nullable)
                $sqlHelper->ensureForeignKeyConstraint(
                    'pickware_erp_supplier_orders',
                    'supplierId',
                    'pickware_erp_suppliers',
                    'id'
                );
                // Add a restricting foreign key constraint from supplier orders to supplier order statuses
                $sqlHelper->cleanUpAndEnsureRestrictingForeignKeyConstraint(
                    'pickware_erp_supplier_orders',
                    'statusId',
                    'pickware_erp_supplier_order_statuses',
                    'id'
                );
                // Add a restricting foreign key constraint from supplier orders to warehouses
                $sqlHelper->cleanUpAndEnsureRestrictingForeignKeyConstraint(
                    'pickware_erp_supplier_orders',
                    'warehouseId',
                    'pickware_erp_warehouses',
                    'id'
                );
                // Add a foreign key constraint from supplier orders to (backend) users
                $sqlHelper->cleanUpAndEnsureNullingForeignKeyConstraint(
                    'pickware_erp_supplier_orders',
                    'userId',
                    's_core_auth',
                    'id'
                );
                // Add a foreign key constraint from article detail supplier mappings to suppliers
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_article_detail_supplier_mappings',
                    'supplierId',
                    'pickware_erp_suppliers',
                    'id'
                );
                // Add a foreign key constraint from article detail supplier mappings to article details
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_article_detail_supplier_mappings',
                    'articleDetailId',
                    's_articles_details',
                    'id'
                );
                // Add a foreign key constraint from manufacturer supplier mappings to suppliers
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_manufacturer_supplier_mappings',
                    'supplierId',
                    'pickware_erp_suppliers',
                    'id'
                );
                // Add a foreign key constraint from manufacturer supplier mappings to manufacturers
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_manufacturer_supplier_mappings',
                    'manufacturerId',
                    's_articles_supplier',
                    'id'
                );
                // Add a foreign key constraint from suppliers to shops
                $sqlHelper->cleanUpAndEnsureNullingForeignKeyConstraint(
                    'pickware_erp_suppliers',
                    'documentLocalizationSubShopId',
                    's_core_shops',
                    'id'
                );
                // Add a foreign key constraint from stock reservations to article detail bin location mappings
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_order_stock_reservations',
                    'articleDetailBinLocationMappingId',
                    'pickware_erp_article_detail_bin_location_mappings',
                    'id'
                );
                // Add a foreign key constraint from stock reservations to order details
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_order_stock_reservations',
                    'orderDetailId',
                    's_order_details',
                    'id'
                );
                // Add a foreign key constraint from article detail bin location mappings to bin locations
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_article_detail_bin_location_mappings',
                    'binLocationId',
                    'pickware_erp_bin_locations',
                    'id'
                );
                // Add a foreign key constraint from article detail bin location mappings to article details
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_article_detail_bin_location_mappings',
                    'articleDetailId',
                    's_articles_details',
                    'id'
                );
                // Add a foreign key constraint from bin location stock snapshots to stock ledger entries
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_bin_location_stock_snapshots',
                    'stockLedgerEntryId',
                    'pickware_erp_stock_ledger_entries',
                    'id'
                );
                // Add a foreign key constraint from bin location stock snapshots to bin locations
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_bin_location_stock_snapshots',
                    'binLocationId',
                    'pickware_erp_bin_locations',
                    'id'
                );
                // Add a foreign key constraint from bin locations to warehouses
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_bin_locations',
                    'warehouseId',
                    'pickware_erp_warehouses',
                    'id'
                );
                // Add a foreign key constraint from warehouse article detail stock counts to warehouses
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_warehouse_article_detail_stock_counts',
                    'warehouseId',
                    'pickware_erp_warehouses',
                    'id'
                );
                // Add a foreign key constraint from warehouse article detail stock counts to article details
                $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
                    'pickware_erp_warehouse_article_detail_stock_counts',
                    'articleDetailId',
                    's_articles_details',
                    'id'
                );

                // Remove all supplier order items, which are neither associated with a supplier order, nor with an
                // article detail
                $database->query(
                    'DELETE FROM `pickware_erp_supplier_order_items`
                    WHERE
                        `supplierOrderId` IS NULL
                        AND `articleDetailId` IS NULL'
                );

                // Remove the attribute ``
                $attributeColumnUninstallationHelper->removeAttributeColumnsIfExist([
                    new AttributeColumnDescription(
                        's_articles_attributes',
                        'viison_physical_stock'
                    ),
                ]);

                // Rename attribute columns for Pickware 5
                $sqlHelper->ensureColumnIsRenamed(
                    's_articles_attributes',
                    'viison_incoming_stock',
                    'pickware_incoming_stock',
                    'int(11) NOT NULL DEFAULT \'0\''
                );
                $sqlHelper->ensureColumnIsRenamed(
                    's_articles_attributes',
                    'viison_not_relevant_for_stock_manager',
                    'pickware_stock_management_disabled',
                    'tinyint(1) NOT NULL DEFAULT \'0\''
                );
                $sqlHelper->ensureColumnIsRenamed(
                    's_articles_attributes',
                    'viison_physical_stock_for_sale',
                    'pickware_physical_stock_for_sale',
                    'int(11) NOT NULL DEFAULT \'0\''
                );
                $sqlHelper->ensureColumnIsRenamed(
                    's_articles_attributes',
                    'viison_stock_initialized',
                    'pickware_stock_initialized',
                    'tinyint(1) NOT NULL DEFAULT \'0\''
                );
                $sqlHelper->ensureColumnIsRenamed(
                    's_articles_attributes',
                    'viison_stock_initialization_time',
                    'pickware_stock_initialization_time',
                    'datetime NULL'
                );
                $sqlHelper->ensureColumnIsRenamed(
                    's_order_attributes',
                    'viison_last_changed',
                    'pickware_last_changed',
                    'datetime NOT NULL DEFAULT \'2000-01-01 00:00:00\''
                );
                $sqlHelper->ensureColumnIsRenamed(
                    's_order_details_attributes',
                    'viison_canceled_quantity',
                    'pickware_canceled_quantity',
                    'int(11) NOT NULL DEFAULT \'0\''
                );
                $modelManager->generateAttributeModels([
                    's_articles_attributes',
                    's_order_attributes',
                    's_order_details_attributes',
                ]);

                // Add new table for warehouse article detail configurations
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `pickware_erp_warehouse_article_detail_configurations` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `warehouseId` int(11) NOT NULL,
                        `articleDetailId` int(11) unsigned NOT NULL,
                        `minimumStock` int(11) NOT NULL DEFAULT \'0\',
                        `targetStock` int(11) NOT NULL DEFAULT \'0\',
                        PRIMARY KEY (`id`),
                        UNIQUE KEY `warehouseId_articleDetailId` (`warehouseId`,`articleDetailId`),
                        KEY `IDX_4EC7E13CE8DE5B58` (`warehouseId`),
                        KEY `IDX_4EC7E13CD0871F1C` (`articleDetailId`),
                        CONSTRAINT `FK_4EC7E13CE8DE5B58` FOREIGN KEY (`warehouseId`) REFERENCES `pickware_erp_warehouses` (`id`) ON DELETE CASCADE,
                        CONSTRAINT `FK_4EC7E13CD0871F1C` FOREIGN KEY (`articleDetailId`) REFERENCES `s_articles_details` (`id`) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );
                // Add entries for all article details in all warehouses
                $allWarehouseIds = $database->fetchCol(
                    'SELECT `id`
                    FROM `pickware_erp_warehouses`'
                );
                foreach ($allWarehouseIds as $warehouseId) {
                    $database->query(
                        'INSERT IGNORE INTO `pickware_erp_warehouse_article_detail_configurations` (`warehouseId`, `articleDetailId`)
                            SELECT :warehouseId, `articleDetail`.`id`
                            FROM `s_articles_details` AS `articleDetail`',
                        [
                            'warehouseId' => $warehouseId,
                        ]
                    );
                }
                // Copy the minimum and target stocks for each warehouse to the new configurations table, if necessary
                $stockCountsTableNeedsMigration = (
                    $sqlHelper->doesColumnExist('pickware_erp_warehouse_article_detail_stock_counts', 'minimumStock')
                    && $sqlHelper->doesColumnExist('pickware_erp_warehouse_article_detail_stock_counts', 'targetStock')
                );
                if ($stockCountsTableNeedsMigration) {
                    // Copy minimumStock and targetStock from the stock counts to the configuration
                    $database->query(
                        'UPDATE `pickware_erp_warehouse_article_detail_configurations` AS `configs`
                        LEFT JOIN `pickware_erp_warehouse_article_detail_stock_counts` AS `stockCounts`
                            ON `stockCounts`.`warehouseId` = `configs`.`warehouseId`
                            AND `stockCounts`.`articleDetailId` = `configs`.`articleDetailId`
                        SET
                            `configs`.`minimumStock` = `stockCounts`.`minimumStock`,
                            `configs`.`targetStock` = `stockCounts`.`targetStock`
                        WHERE
                            `stockCounts`.`minimumStock` != 0
                            OR `stockCounts`.`targetStock` != 0'
                    );
                    // Drop the columns from the stock counts table
                    $sqlHelper->dropColumnIfExists(
                        'pickware_erp_warehouse_article_detail_stock_counts',
                        'minimumStock'
                    );
                    $sqlHelper->dropColumnIfExists('pickware_erp_warehouse_article_detail_stock_counts', 'targetStock');
                }
            case '5.0.0':
                // Nothing to do
            case '5.0.1':
                // Nothing to do
            case '5.0.2':
                // Rename 'pickware_erp_warehouses.defaultBinLocationId' to 'nullBinLocationId'
                $sqlHelper->removeForeignKeyConstraintIfExists('pickware_erp_warehouses', 'defaultBinLocationId');
                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_warehouses',
                    'defaultBinLocationId',
                    'nullBinLocationId',
                    'int(11) DEFAULT NULL'
                );
                $sqlHelper->cleanUpAndEnsureNullingForeignKeyConstraint(
                    'pickware_erp_warehouses',
                    'nullBinLocationId',
                    'pickware_erp_bin_locations',
                    'id'
                );
            case '5.1.0':
                // Nothing to do
            case '5.2.0':
                // Nothing to do
            case '5.2.1':
                // Change the foreign key constraint from stock item property values to item properties to be resticting
                $sqlHelper->cleanUpAndEnsureRestrictingForeignKeyConstraint(
                    'pickware_erp_stock_item_property_values',
                    'itemPropertyId',
                    'pickware_erp_item_properties',
                    'id'
                );
                // Remove the shop configuration element for handling the backend about window and use hidden config
                // field instead. Flush the form once so we can be sure that every config element is persisted in the db
                // and the following code is consistent for both an update and an installation process.
                $modelManager->flush($form);
                $backendAboutPopUpConfigElementValue = true;
                $backendAboutPopUpConfigElement = $form->getElement('displayPickwareAboutWindow');
                if ($backendAboutPopUpConfigElement) {
                    $backendAboutPopUpConfigElementValue = $backendAboutPopUpConfigElement->getValue();
                    $values = $backendAboutPopUpConfigElement->getValues();
                    if ($values !== null && count($values) > 0) {
                        // Since the HiddenConfigStorageService uses the default value of the config element, look for sub
                        // shop configurations (config element value) and override the config element default value.
                        $backendAboutPopUpConfigElementValue = $values[0]->getValue();
                    }
                    $form->removeElement($backendAboutPopUpConfigElement);
                    $modelManager->remove($backendAboutPopUpConfigElement);
                    $modelManager->flush([
                        $form,
                        $backendAboutPopUpConfigElement,
                    ]);
                }
                $this->get('viison_common.hidden_config_storage')->setConfigValue(
                    'pickwareErpDisplayAboutWindow',
                    'boolean',
                    $backendAboutPopUpConfigElementValue
                );
            case '5.3.0':
                // Nothing to do
            case '5.3.1':
                // Nothing to do
            case '5.3.2':
                // Nothing to do
            case '5.4.0':
                // Nothing to do
            case '5.4.1':
                // Nothing to do
            case '5.4.2':
                // In a previous release the wrong value and style was set for the document box 'Signature' and 'Info'
                // of pick list documents. Reset this document box now with the default values.
                $pickListDocumentType = DocumentUtil::getDocumentType(
                    self::PICK_LIST_DOCUMENT_TYPE_KEY,
                    'Pickware Pickliste'
                );
                $database->query(
                    'UPDATE s_core_documents_box
                        SET style = :style, value = :value
                        WHERE name = :name
                        AND documentID = :documentId
                        AND style = :value
                        AND value = :style',
                    [
                        'style' => 'font-size:14px; width: 38%; text-align:center; float: left; margin-top:16mm; border-top: 2px solid black;',
                        'value' => 'Unterschrift',
                        'name' => 'Signature',
                        'documentId' => $pickListDocumentType->getId(),
                    ]
                );
                $database->query(
                    'UPDATE s_core_documents_box
                        SET style = :style, value = :value
                        WHERE name = :name
                        AND documentID = :documentId
                        AND style = :value
                        AND value = :style',
                    [
                        'style' => 'float: left; width: 60%; text-align:left; height: 20mm;',
                        'value' => '',
                        'name' => 'Info',
                        'documentId' => $pickListDocumentType->getId(),
                    ]
                );
            case '5.5.0':
                // Update all cached incoming stocks
                $database->query(
                    'UPDATE `pickware_erp_warehouse_article_detail_stock_counts` AS `stockCounts`
                    LEFT JOIN (
                        SELECT
                            `supplierOrders`.`warehouseId`,
                            `supplierOrderItems`.`articleDetailId`,
                            GREATEST(
                                SUM(`supplierOrderItems`.`orderedQuantity`) - SUM(`supplierOrderItems`.`deliveredQuantity`),
                                0
                            ) AS `incomingStock`
                        FROM `pickware_erp_supplier_order_items` AS `supplierOrderItems`
                        INNER JOIN `pickware_erp_supplier_orders` AS `supplierOrders`
                            ON `supplierOrderItems`.`supplierOrderId` = `supplierOrders`.`id`
                        WHERE
                            `supplierOrders`.`statusId` NOT IN (4, 5)
                            AND `supplierOrderItems`.`statusId` NOT IN (4, 5)
                        GROUP BY `supplierOrders`.`warehouseId`, `supplierOrderItems`.`articleDetailId`
                    ) AS `supplierOrderIncomingStocks`
                        ON `supplierOrderIncomingStocks`.`warehouseId` = `stockCounts`.`warehouseId`
                        AND `supplierOrderIncomingStocks`.`articleDetailId` = `stockCounts`.`articleDetailId`
                    SET `stockCounts`.`incomingStock` = IFNULL(`supplierOrderIncomingStocks`.`incomingStock`, 0)'
                );
            case '5.5.1':
                $menuInstallationHelper->deleteMenuItemFromDatabase([
                    'parent' => $this->Menu()->findOneBy([
                        'controller' => 'Article',
                        'action' => null,
                    ]),
                    'controller' => 'ViisonPickwareCommonBarcode',
                    'action' => null,
                    'label' => 'Etiketten',
                    'class' => 'sprite-barcode',
                    'active' => 1,
                    'position' => 50,
                ]);
                $menuInstallationHelper->deleteMenuItemFromDatabase([
                    'parent' => $this->Menu()->findOneBy([
                        'controller' => 'ViisonPickwareCommonBarcode',
                        'action' => null,
                    ]),
                    'controller' => 'ViisonPickwareCommonBarcodeLabelPrinting',
                    'action' => 'Index',
                    'label' => 'Etikettendruck',
                    'class' => 'sprite-blog',
                    'active' => 1,
                    'position' => 1,
                ]);
                $menuInstallationHelper->deleteMenuItemFromDatabase([
                    'parent' => $this->Menu()->findOneBy([
                        'controller' => 'ViisonPickwareCommonBarcode',
                        'action' => null,
                    ]),
                    'controller' => 'ViisonPickwareCommonBarcodeLabelPresets',
                    'action' => 'Index',
                    'label' => 'Vorlagenkonfiguration',
                    'class' => 'sprite-layout-3-mix',
                    'active' => 1,
                    'position' => 2,
                ]);

                $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $this->Menu()->findOneBy([
                        'controller' => 'Article',
                        'action' => null,
                    ]),
                    'controller' => 'ViisonPickwareERPBarcode',
                    'action' => null,
                    'label' => 'Etiketten',
                    'class' => 'sprite-barcode',
                    'active' => 1,
                    'position' => 50,
                ]);
                $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $this->Menu()->findOneBy([
                        'controller' => 'ViisonPickwareERPBarcode',
                        'action' => null,
                    ]),
                    'controller' => 'ViisonPickwareERPBarcodeLabelPrinting',
                    'action' => 'Index',
                    'label' => 'Etikettendruck',
                    'class' => 'sprite-blog',
                    'active' => 1,
                    'position' => 1,
                ]);
                $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $this->Menu()->findOneBy([
                        'controller' => 'ViisonPickwareERPBarcode',
                        'action' => null,
                    ]),
                    'controller' => 'ViisonPickwareERPBarcodeLabelPresets',
                    'action' => 'Index',
                    'label' => 'Vorlagenkonfiguration',
                    'class' => 'sprite-layout-3-mix',
                    'active' => 1,
                    'position' => 2,
                ]);

                $this->migrateOrCreateBarcodeLabelPrinting();

                $sqlHelper->ensureColumnIsRenamed(
                    'pickware_erp_article_detail_supplier_mappings',
                    'orderAmount',
                    'minimumOrderAmount',
                    'INT(11) DEFAULT NULL'
                );
                $sqlHelper->addColumnIfNotExists(
                    'pickware_erp_article_detail_supplier_mappings',
                    'packingUnit',
                    'INT(11) NOT NULL DEFAULT 1'
                );

                // Add a column that marks a StockLedgerEntry as a correction (its stock change is in the opposite of
                // its non-correction version)
                $sqlHelper->addColumnIfNotExists('pickware_erp_stock_ledger_entries', 'correction', 'TINYINT(1) NOT NULL DEFAULT 0');

                // Add attribute for summary return shipment status
                $attributeColumnInstallationHelper->addAttributeColumnsIfNotExist([
                    new AttributeColumnDescription(
                        's_order_attributes',
                        'pickware_return_shipment_status_id',
                        'INT(11)'
                    )
                ]);

                // When the user has PickwareMobile, there will be an album that is named "Pickware Rücksendungen". This
                // album should be used instead of creating a new one.
                /** @var Album $mediaAlbum */
                $mediaAlbum = $modelManager->getRepository(Album::class)->findOneBy([
                    'name' => 'Pickware Rücksendungen',
                ]);
                if ($mediaAlbum) {
                    $mediaAlbum->setName('Pickware Retouren');
                    $modelManager->flush($mediaAlbum);
                } else {
                    $mediaAlbum = $mediaAlbumInstallationHelper->createMediaAlbumUnlessExists(
                        'Pickware Retouren',
                        'sprite-box--arrow',
                        30
                    );
                }

                if (method_exists($mediaAlbum, 'setGarbageCollectable')) {
                    // property was introduces with SW5.4
                    $mediaAlbum->setGarbageCollectable(true);
                    $modelManager->flush($mediaAlbum);
                }

                $this->addReturnEmailTemplates();

                // Add the warehouse to every return shipment
                $sqlHelper->addColumnIfNotExists('pickware_erp_return_shipments', 'targetWarehouseId', 'INT(11) NULL');
                // Assign a warehouse to every existing return shipment. The warehouse is determined by the assigned
                // stock entry of the return shipment items.
                $database->query(
                    'UPDATE pickware_erp_return_shipments AS returnShipmentsToUpdate
                    LEFT JOIN (
                        SELECT
                            id AS returnShipmentId,
                            (
                                SELECT stockLedgerEntries.warehouseId
                                FROM pickware_erp_stock_ledger_entries stockLedgerEntries
                                INNER JOIN pickware_erp_return_shipment_items returnItems
                                    ON stockLedgerEntries.returnShipmentItemId = returnItems.id
                                WHERE returnItems.returnShipmentId = returnShipments.id
                                ORDER BY created DESC
                                LIMIT 1
                            ) AS warehouseId
                        FROM pickware_erp_return_shipments returnShipments
                    ) AS latestStockLedgerEntryOfReturnShipments
                        ON latestStockLedgerEntryOfReturnShipments.returnShipmentId = returnShipmentsToUpdate.id
                    SET returnShipmentsToUpdate.targetWarehouseId = latestStockLedgerEntryOfReturnShipments.warehouseId
                    WHERE returnShipmentsToUpdate.targetWarehouseId IS NULL'
                );

                $sqlHelper->cleanUpAndEnsureNullingForeignKeyConstraint(
                    'pickware_erp_return_shipments',
                    'targetWarehouseId',
                    'pickware_erp_warehouses',
                    'id'
                );

                // Cleanup the table pickware_erp_return_shipment_items before adding a unique constraint on the columns
                // orderDetailId, returnShipmentId. Usually no duplicates should exist, but better to be safe than
                // sorry.
                $database->beginTransaction();
                // For each orderDetailId-returnShipmentId combination, sum up the quantities and write them back to
                // all of the duplicated items. Since we are going to delete every item but the first later, it
                // doesn't matter that we update all items of a orderDetailId-returnShipmentId combination.
                $database->query(
                    'UPDATE pickware_erp_return_shipment_items AS allReturnItems
                    LEFT JOIN (
                        SELECT
                            returnShipmentId,
                            orderDetailId,
                            SUM(returnedQuantity) AS summedReturnedQuantity,
                            SUM(writtenOffQuantity) AS summedWrittenOffQuantity
                        FROM pickware_erp_return_shipment_items
                        GROUP BY
                            returnShipmentId,
                            orderDetailId
                    ) AS summedItems
                        ON allReturnItems.orderDetailId = summedItems.orderDetailId
                        AND allReturnItems.returnShipmentId = summedItems.returnShipmentId
                    SET allReturnItems.returnedQuantity = summedItems.summedReturnedQuantity,
                        allReturnItems.writtenOffQuantity = summedItems.summedWrittenOffQuantity'
                );
                // Update all stock ledger entries referencing one of the duplicate that are going to be removed
                // to reference the one to keep
                $database->query(
                    'UPDATE pickware_erp_stock_ledger_entries stockLedgerEntries
                    INNER JOIN (
                        SELECT
                            rsi.id AS idOfEntryToRemove,
                            duplicatedEntries.idOfEntryToKeep
                        FROM pickware_erp_return_shipment_items rsi
                        INNER JOIN (
                            SELECT
                                MIN(id) AS idOfEntryToKeep,
                                orderDetailId,
                                returnShipmentId
                            FROM pickware_erp_return_shipment_items
                            GROUP BY
                                orderDetailId,
                                returnShipmentId
                        ) AS duplicatedEntries
                            ON duplicatedEntries.orderDetailId = rsi.orderDetailId
                            AND duplicatedEntries.returnShipmentId = rsi.returnShipmentId
                        WHERE rsi.id != duplicatedEntries.idOfEntryToKeep
                    ) AS duplicateItems
                        ON duplicateItems.idOfEntryToRemove = stockLedgerEntries.returnShipmentItemId
                    SET stockLedgerEntries.returnShipmentItemId = duplicateItems.idOfEntryToKeep'
                );
                // Remove all but one duplicate return shipment items for each orderDetailId-returnShipmentId.
                $database->query(
                    'DELETE rsi
                    FROM pickware_erp_return_shipment_items rsi
                    INNER JOIN (
                        SELECT
                            MIN(id) AS idOfEntryToKeep,
                            orderDetailId,
                            returnShipmentId
                        FROM pickware_erp_return_shipment_items
                        GROUP BY
                            orderDetailId,
                            returnShipmentId
                    ) AS duplicatedEntries
                        ON duplicatedEntries.orderDetailId = rsi.orderDetailId
                        AND duplicatedEntries.returnShipmentId = rsi.returnShipmentId
                    WHERE rsi.id != duplicatedEntries.idOfEntryToKeep'
                );
                $database->commit();
                // Do not allow multiple items of the same return shipment to reference the same order detail
                $sqlHelper->addUniqueConstraintIfNotExists(
                    'pickware_erp_return_shipment_items',
                    [
                        'orderDetailId',
                        'returnShipmentId',
                    ]
                );

                // Create the new status 'Cancelation invoice created'
                $database->query(
                    'INSERT IGNORE INTO pickware_erp_return_shipment_statuses (id, name)
                    VALUES
                        (:idCanceled, :nameCanceled),
                        (:idNew, :nameNew)',
                    [
                        'idCanceled' => ReturnShipmentStatus::STATUS_CANCELLED_ID,
                        'nameCanceled' => ReturnShipmentStatus::STATUS_CANCELLED_NAME,
                        'idNew' => ReturnShipmentStatus::STATUS_NEW_ID,
                        'nameNew' => ReturnShipmentStatus::STATUS_NEW_NAME,
                    ]
                );
                $sqlHelper->addColumnIfNotExists(
                    'pickware_erp_return_shipment_statuses',
                    'processStep',
                    'INT(11) NULL'
                );
                $processSteps = [
                    1 => ReturnShipmentStatus::STATUS_NEW_ID,
                    2 => ReturnShipmentStatus::STATUS_RECEIVED_ID,
                    3 => ReturnShipmentStatus::STATUS_CANCELLED_ID,
                    4 => ReturnShipmentStatus::STATUS_COMPLETED_ID,
                ];
                foreach ($processSteps as $processStep => $statusId) {
                    $database->query(
                        'UPDATE pickware_erp_return_shipment_statuses
                        SET processStep = :processStep
                        WHERE id = :statusId
                            AND processStep IS NULL',
                        [
                            'processStep' => $processStep,
                            'statusId' => $statusId,
                        ]
                    );
                }
                // Do not allow null anymore as update is done
                $database->query('ALTER TABLE pickware_erp_return_shipment_statuses MODIFY COLUMN processStep INT(11) NOT NULL');

                $sqlHelper->addUniqueConstraintIfNotExists('pickware_erp_return_shipment_statuses', ['processStep']);
                $sqlHelper->addUniqueConstraintIfNotExists('pickware_erp_return_shipment_statuses', ['name']);

                $sqlHelper->addColumnIfNotExists(
                    'pickware_erp_return_shipment_items',
                    'cancelledQuantity',
                    'INT(11) NOT NULL DEFAULT 0'
                );

                // ViisonPickwareMobile cancelled the order positions immediately after creating the return shipment items.
                // So we can assume that all returned items are cancelled, too. That's why we can set the
                // cancelledQuantity to returnedQuantity. Also the status of all return shipments is set to at least
                // cancelled.
                $database->query(
                    'UPDATE pickware_erp_return_shipment_items
                    SET cancelledQuantity = returnedQuantity'
                );

                $hiddenConfigFieldId = $database->fetchOne(
                    'SELECT id
                    FROM s_core_config_elements
                    WHERE name = "pickware_erp_return_shipment_migration"'
                );
                if ($hiddenConfigFieldId === false) {
                    $database->beginTransaction();
                    // In ViisonPickwareMobile there were only 2 statuses: received and completed. But ViisonPickwareMobile
                    // immediately cancelled all created return shipment items. Hence the old status 'received' is
                    // now the status 'cancelled'.
                    $database->query(
                        'UPDATE pickware_erp_return_shipments
                        SET statusId = :statusIdCancelled
                        WHERE statusId = :statusIdReceived',
                        [
                            'statusIdCancelled' => ReturnShipmentStatus::STATUS_CANCELLED_ID,
                            'statusIdReceived' => ReturnShipmentStatus::STATUS_RECEIVED_ID,
                        ]
                    );
                    // In ViisonPickwareMobile shipped was reduced by returned quantity. We need to revert this change
                    $database->query(
                        'UPDATE s_order_details sod
                        LEFT JOIN (
                            SELECT
                                rsi.orderDetailId,
                                SUM(rsi.returnedQuantity) AS returnedTotal
                            FROM pickware_erp_return_shipment_items rsi
                            GROUP BY rsi.orderDetailId
                        ) AS shippedCorrection
                            ON shippedCorrection.orderDetailId = sod.id
                        SET sod.shipped = sod.shipped + IFNULL(shippedCorrection.returnedTotal, 0)'
                    );
                    $database->query(
                        'INSERT INTO s_core_config_elements (
                            form_id,
                            name,
                            value,
                            type,
                            required,
                            position,
                            scope
                        )
                        VALUES (
                            0,
                            "pickware_erp_return_shipment_migration",
                            "b:1;",
                            "boolean",
                            0,
                            0,
                            0
                        )'
                    );
                    $database->commit();
                }

                // Set the pickware_return_shipment_status_id attribute for all orders having return shipments
                $database->query(
                    'UPDATE s_order_attributes orderAttributes
                    INNER JOIN (
                        SELECT
                            orderId,
                            MIN(statusId) AS statusId
                        FROM pickware_erp_return_shipments
                        GROUP BY orderId
                    ) AS returnShipments
                        ON returnShipments.orderId = orderAttributes.orderID
                    SET orderAttributes.pickware_return_shipment_status_id = returnShipments.statusId'
                );

                // Create menu item for launching the return shipments app
                $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $this->Menu()->findOneBy(['controller' => 'Customer']),
                    'controller' => 'ViisonPickwareERPReturnShipment',
                    'action' => 'Index',
                    'label' => 'Retouren',
                    'class' => 'sprite-arrow-return-180',
                    'active' => 1,
                    'position' => 1,
                ]);

                $database->query(
                    'CREATE TABLE IF NOT EXISTS `pickware_erp_return_shipment_internal_comments` (
                        `id` INT(11) NOT NULL AUTO_INCREMENT,
                        `returnShipmentId` INT(11) NOT NULL,
                        `userId` INT(11) NULL,
                        `created` DATETIME NOT NULL,
                        `comment` TEXT NOT NULL,
                        PRIMARY KEY (`id`),
                        CONSTRAINT `FK_2E925611BF396750` FOREIGN KEY (`returnShipmentId`) REFERENCES `pickware_erp_return_shipments` (`id`) ON DELETE CASCADE,
                        CONSTRAINT `FK_686A3F19BF396750` FOREIGN KEY (`userId`) REFERENCES `s_core_auth` (`id`) ON DELETE SET NULL
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
                );

                if ($sqlHelper->doesColumnExist('pickware_erp_return_shipments', 'comment')) {
                    $database->query(
                        'INSERT INTO `pickware_erp_return_shipment_internal_comments` (
                            `returnShipmentId`,
                            `comment`,
                            `userId`,
                            `created`
                        )
                        SELECT
                            `returnShipment`.`id` AS `returnShipmentId`,
                            `returnShipment`.`comment`,
                            `returnShipment`.`userId`,
                            `returnShipment`.`created`
                        FROM `pickware_erp_return_shipments` AS `returnShipment`
                        LEFT JOIN (
                            SELECT
                                `id`,
                                `returnShipmentId`
                            FROM `pickware_erp_return_shipment_internal_comments`
                        ) AS `existingComment`
                            ON `existingComment`.`returnShipmentId` = `returnShipment`.`id`
                        WHERE
                            `returnShipment`.`comment` IS NOT NULL
                            AND `returnShipment`.`comment` != ""
                            AND `existingComment`.`id` IS NULL'
                    );
                }
                $sqlHelper->dropColumnIfExists('pickware_erp_return_shipments', 'comment');

                $orderNumberInstallationHelper->createSequenceNumberGeneratorIfNotExists(
                    ReturnShipmentProcessorService::NUMBER_RANGE_NAME,
                    20000,
                    'Retouren'
                );

                $sqlHelper->addColumnIfNotExists('pickware_erp_return_shipments', 'number', 'VARCHAR(255) NULL');

                // Generating numbers for existing return shipments.
                // To ensure that the corresponding number range is always updated after generating the numbers for the
                // return shipments, this update step is done in a transaction.
                $database->beginTransaction();
                $currentNumber = $database->fetchOne(
                    'SELECT number
                    FROM s_order_number
                    WHERE name = :numberRangeName
                    FOR UPDATE',
                    [
                        'numberRangeName' => ReturnShipmentProcessorService::NUMBER_RANGE_NAME,
                    ]
                );
                if ($currentNumber === false) {
                    throw new NumberRangeNotFoundException(ReturnShipmentProcessorService::NUMBER_RANGE_NAME);
                }

                // Find return shipments without numbers and create numbers for it. The numbers are created
                // automatically based on the previously fetched number range value $currentNumber. The MySQL query
                // automatically increments the number for each return shipment by one.
                $updatedRows = $database->query(
                    'UPDATE `pickware_erp_return_shipments` AS `leftSide`
                    INNER JOIN (
                        SELECT
                            `id`,
                            @rownum := @rownum + 1 AS `number`
                        FROM `pickware_erp_return_shipments`
                        CROSS JOIN (SELECT @rownum := :currentNumber) r
                        WHERE `number` IS NULL
                    ) AS `rightSide`
                        ON `rightSide`.`id` = `leftSide`.`id`
                    SET `leftSide`.`number` = `rightSide`.`number`',
                    [
                        'currentNumber' => $currentNumber,
                    ]
                )->rowCount();

                // Increase the number range value by exactly the amount of created return shipments numbers.
                $currentNumber += $updatedRows;
                $database->query(
                    'UPDATE s_order_number
                    SET number = :currentNumber
                    WHERE name = :numberRangeName',
                    [
                        'currentNumber' => $currentNumber,
                        'numberRangeName' => ReturnShipmentProcessorService::NUMBER_RANGE_NAME,
                    ]
                );
                $database->commit();

                $database->query('ALTER TABLE `pickware_erp_return_shipments` MODIFY COLUMN `number` VARCHAR(255) NOT NULL');
                $sqlHelper->addUniqueConstraintIfNotExists('pickware_erp_return_shipments', ['number']);

                $sqlHelper->addColumnIfNotExists('pickware_erp_return_shipments', 'documentId', 'INT(11) NULL');
                $sqlHelper->ensureForeignKeyConstraint(
                    'pickware_erp_return_shipments',
                    'documentId',
                    's_order_documents',
                    'id',
                    'SET NULL'
                );

                // Create a column to mark a warehouse as the default for return shipments and use the current default
                // warehouse as the default return shipment warehouse.
                $sqlHelper->addColumnIfNotExists(
                    'pickware_erp_warehouses',
                    'defaultReturnShipmentWarehouse',
                    'TINYINT(1) NOT NULL DEFAULT 0'
                );
                $defaultReturnShipmentWarehouse = $database->fetchOne(
                    'SELECT id
                    FROM pickware_erp_warehouses
                    WHERE defaultReturnShipmentWarehouse = 1'
                );
                if ($defaultReturnShipmentWarehouse === false) {
                    $database->query(
                        'UPDATE pickware_erp_warehouses
                        SET defaultReturnShipmentWarehouse = 1
                        WHERE defaultWarehouse = 1'
                    );
                }

                // Migrate old 'manual' stock ledger entries for write offs
                $database->query(
                    'UPDATE `pickware_erp_stock_ledger_entries`
                    SET `type` = :writeOffType,
                        `comment` = "Migrated from type \\"manual\\""
                    WHERE
                        `type` = :manualType
                        AND `comment` = "Abschreibung"',
                    [
                        'writeOffType' => StockLedgerEntry::TYPE_WRITE_OFF,
                        'manualType' => StockLedgerEntry::TYPE_MANUAL,
                    ]
                );

                // Since we changed the stock entry type of stock entries that are written when using the stock import,
                // we need to migrate existing stock entries of the old type.
                $csvImportStockEntryComment = 'SwagImportExport: "physicalStock" Import%';
                $database->query(
                    'UPDATE pickware_erp_stock_ledger_entries
                    SET type = :incomingType
                    WHERE
                        type = :manualType
                        AND changeAmount > 0
                        AND comment LIKE :csvImportStockEntryComment',
                    [
                        'incomingType' => StockLedgerEntry::TYPE_INCOMING,
                        'manualType' => StockLedgerEntry::TYPE_MANUAL,
                        'csvImportStockEntryComment' => $csvImportStockEntryComment,
                    ]
                );
                $database->query(
                    'UPDATE `pickware_erp_stock_ledger_entries`
                    SET `type` = :outgoingType
                    WHERE
                        `type` = :manualType
                        AND `changeAmount` < 0
                        AND `comment` LIKE :csvImportStockEntryComment',
                    [
                        'outgoingType' => StockLedgerEntry::TYPE_OUTGOING,
                        'manualType' => StockLedgerEntry::TYPE_MANUAL,
                        'csvImportStockEntryComment' => $csvImportStockEntryComment,
                    ]
                );

                // Add custom currency to supplier and supplier orders
                $sqlHelper->addColumnIfNotExists(
                    'pickware_erp_suppliers',
                    'currencyId',
                    'int(11) NULL'
                );
                $sqlHelper->addColumnIfNotExists(
                    'pickware_erp_supplier_orders',
                    'currencyId',
                    'int(11) NULL'
                );
                // Migrate existing suppliers and supplier orders by setting the currency to the default currency where
                // no currency is set.
                $defaultCurrency = CurrencyUtil::getDefaultCurrency();
                $database->query(
                    'UPDATE pickware_erp_suppliers
                    SET currencyId = :defaultCurrencyId
                    WHERE currencyId IS NULL',
                    [
                        'defaultCurrencyId' => $defaultCurrency->getId(),
                    ]
                );
                $database->query(
                    'UPDATE pickware_erp_supplier_orders
                    SET currencyId = :defaultCurrencyId
                    WHERE currencyId IS NULL',
                    [
                        'defaultCurrencyId' => $defaultCurrency->getId(),
                    ]
                );
                $database->query('ALTER TABLE `pickware_erp_suppliers` CHANGE `currencyId` `currencyId` int(11) NOT NULL');
                $database->query('ALTER TABLE `pickware_erp_supplier_orders` CHANGE `currencyId` `currencyId` int(11) NOT NULL');
                $sqlHelper->ensureForeignKeyConstraint(
                    'pickware_erp_suppliers',
                    'currencyId',
                    's_core_currencies',
                    'id',
                    'RESTRICT'
                );
                $sqlHelper->ensureForeignKeyConstraint(
                    'pickware_erp_supplier_orders',
                    'currencyId',
                    's_core_currencies',
                    'id',
                    'RESTRICT'
                );

                // Create menu item for the REST API log module. Remove ViisonPickwareCommon's menu item first, since
                // the REST API log is being moved from there.
                $logMenuItem = $this->Menu()->findOneBy([
                    'controller' => 'Log',
                ]);
                $menuInstallationHelper->deleteMenuItemFromDatabase([
                    'parent' => $logMenuItem,
                    'controller' => 'ViisonPickwareCommonApiLog',
                    'action' => 'Index',
                    'label' => 'API-Log',
                    'class' => 'sprite-cards-stack',
                    'active' => 1,
                ]);
                $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $logMenuItem,
                    'controller' => 'ViisonPickwareERPApiLog',
                    'action' => 'Index',
                    'label' => 'API-Log',
                    'class' => 'sprite-cards-stack',
                    'active' => 1,
                ]);

                // Ensure the table "pickware_erp_rest_api_idempotent_operations" for the idempotent operation exist.
                // There are 3 possible migration scenarios:
                //  1. Customer is updating from Pickware < 5 with WMS or POS installed:
                //     -> Table exists as "s_plugin_pickware_idempotency_key" and must be renamed
                //  2. Customer is updating from Pickware 5 with WMS or POS installed:
                //     -> Table exists as "pickware_rest_api_idempotency_keys" and must be renamed
                //  3. Every other case:
                //     -> No table exists and it must be created.
                if ($sqlHelper->doesTableExist('s_plugin_pickware_idempotency_key')) {
                    $sqlHelper->ensureTableIsRenamed(
                        's_plugin_pickware_idempotency_key',
                        'pickware_erp_rest_api_idempotent_operations'
                    );
                } elseif ($sqlHelper->doesTableExist('pickware_rest_api_idempotency_keys')) {
                    $sqlHelper->ensureTableIsRenamed(
                        'pickware_rest_api_idempotency_keys',
                        'pickware_erp_rest_api_idempotent_operations'
                    );
                } else {
                    $database->query(
                        'CREATE TABLE IF NOT EXISTS `pickware_erp_rest_api_idempotent_operations` (
                            `id` int(11) NOT NULL AUTO_INCREMENT,
                            `createdAt` datetime NOT NULL,
                            `idempotencyKey` varchar(255) NOT NULL DEFAULT "",
                            `controllerName` varchar(1000) NOT NULL DEFAULT "",
                            `actionName` varchar(1000) NOT NULL DEFAULT "",
                            `recoveryPoint` varchar(255) NOT NULL DEFAULT "",
                            `responseCode` int(11) DEFAULT NULL,
                            `responseData` longtext,
                            PRIMARY KEY (`id`),
                            UNIQUE KEY `idempotencyKey` (`idempotencyKey`)
                        ) ENGINE=InnoDB DEFAULT CHARSET=utf8;'
                    );
                }

                // Add new colums to the idempontent operations, which did not exist the original, migrated table
                $sqlHelper->addColumnIfNotExists(
                    'pickware_erp_rest_api_idempotent_operations',
                    'intermediateState',
                    'TEXT DEFAULT NULL'
                );
                $sqlHelper->addColumnIfNotExists(
                    'pickware_erp_rest_api_idempotent_operations',
                    'initialRequestRawBodyChecksum',
                    'VARCHAR(255) DEFAULT NULL'
                );
                $database->query(
                    'UPDATE `pickware_erp_rest_api_idempotent_operations`
                    SET `initialRequestRawBodyChecksum` = ""'
                );
                $database->query(
                    'ALTER TABLE `pickware_erp_rest_api_idempotent_operations`
                    MODIFY COLUMN `initialRequestRawBodyChecksum` VARCHAR(255) NOT NULL'
                );

                // Ensure the table "pickware_erp_rest_api_requests" for the api log entries. There are 3
                // possible migration scenarios:
                //  1. Customer is updating from Pickware < 5 with WMS or POS installed:
                //     -> Table exists as "s_plugin_pickware_api_log" and must be renamed
                //  2. Customer is updating from Pickware 5 with WMS or POS installed:
                //     -> Table exists as "pickware_rest_api_requests" and must be renamed
                //  3. Every other case:
                //     -> No table exists and it must be created.
                if ($sqlHelper->doesTableExist('s_plugin_pickware_api_log')) {
                    // Before renaming, any possible existing FK constraints and indexes must be removed. They will be
                    // re-added afterwards.
                    $sqlHelper->dropForeignKeyConstraintIfExists(
                        's_plugin_pickware_api_log',
                        'idempotencyKeyId',
                        'pickware_erp_rest_api_idempotent_operations',
                        'id'
                    );
                    $sqlHelper->dropIndexIfExists(
                        's_plugin_pickware_api_log',
                        $sqlHelper->getConstraintIdentifier('s_plugin_pickware_api_log', 'idempotencyKeyId', 'idx')
                    );
                    $sqlHelper->ensureTableIsRenamed('s_plugin_pickware_api_log', 'pickware_erp_rest_api_requests');
                } elseif ($sqlHelper->doesTableExist('pickware_rest_api_requests')) {
                    // Before renaming, any possible existing FK constraints and indexes must be removed. They will be
                    // re-added afterwards.
                    $sqlHelper->dropForeignKeyConstraintIfExists(
                        'pickware_rest_api_requests',
                        'idempotencyKeyId',
                        'pickware_erp_rest_api_idempotent_operations',
                        'id'
                    );
                    $sqlHelper->dropIndexIfExists(
                        'pickware_rest_api_requests',
                        $sqlHelper->getConstraintIdentifier('pickware_rest_api_requests', 'idempotencyKeyId', 'idx')
                    );
                    $sqlHelper->ensureTableIsRenamed('pickware_rest_api_requests', 'pickware_erp_rest_api_requests');
                } else {
                    $database->query(
                        'CREATE TABLE IF NOT EXISTS `pickware_erp_rest_api_requests` (
                            `id` int(11) NOT NULL AUTO_INCREMENT,
                            `date` datetime NOT NULL,
                            `method` varchar(55) COLLATE utf8_unicode_ci DEFAULT NULL,
                            `url` varchar(2000) COLLATE utf8_unicode_ci DEFAULT NULL,
                            `requestHeaders` longtext COLLATE utf8_unicode_ci,
                            `requestData` longtext COLLATE utf8_unicode_ci,
                            `responseCode` int(11) DEFAULT NULL,
                            `responseHeaders` longtext COLLATE utf8_unicode_ci,
                            `responseData` longtext COLLATE utf8_unicode_ci,
                            `exception` longtext COLLATE utf8_unicode_ci,
                            `user` varchar(255) COLLATE utf8_unicode_ci DEFAULT NULL,
                            `ipAddress` varchar(255) COLLATE utf8_unicode_ci DEFAULT NULL,
                            `userAgent` varchar(255) COLLATE utf8_unicode_ci DEFAULT NULL,
                            `computationTime` double DEFAULT NULL,
                            `idempotencyKeyId` int(11) DEFAULT NULL,
                            PRIMARY KEY (`id`)
                        ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci ROW_FORMAT=COMPRESSED;'
                    );
                }

                // Since we can encounter the table `pickware_erp_rest_api_requests` in different versions, depending on
                // the previously installed version of ViisonPickwareCommon, there may be additional migrations steps
                // we need to peform:
                $sqlHelper->addColumnIfNotExists(
                    'pickware_erp_rest_api_requests',
                    'computationTime',
                    'double DEFAULT NULL'
                );
                $database->query(
                    'ALTER TABLE `pickware_erp_rest_api_requests`
                    MODIFY COLUMN `url` TEXT NULL'
                );
                if (!$sqlHelper->doesColumnExist('pickware_erp_rest_api_requests', 'idempotentOperationId')) {
                    // Add the old `idempotencyKeyId` column first before renaming it. This ensures that the field
                    // exists even if migration from a very old version of ViisonPickwareCommon, which did not contain
                    // the column.
                    $sqlHelper->addColumnIfNotExists(
                        'pickware_erp_rest_api_requests',
                        'idempotencyKeyId',
                        'int(11) DEFAULT NULL'
                    );
                    $sqlHelper->ensureColumnIsRenamed(
                        'pickware_erp_rest_api_requests',
                        'idempotencyKeyId',
                        'idempotentOperationId',
                        'int(11) DEFAULT NULL'
                    );
                }
                $sqlHelper->addIndexIfNotExists(
                    'pickware_erp_rest_api_requests',
                    $sqlHelper->getConstraintIdentifier(
                        'pickware_erp_rest_api_requests',
                        'idempotentOperationId',
                        'idx'
                    ),
                    ['idempotentOperationId']
                );
                $sqlHelper->cleanUpAndEnsureNullingForeignKeyConstraint(
                    'pickware_erp_rest_api_requests',
                    'idempotentOperationId',
                    'pickware_erp_rest_api_idempotent_operations',
                    'id'
                );
            case '6.0.0':
                // Nothing to do
            case '6.1.0':
                // Nothing to do
            case '6.1.1':
                // Nothing to do
            case '6.1.2':
                // Nothing to do
            case '6.1.3':
                // Nothing to do
            case '6.1.4':
                // Nothing to do
            case '6.1.5':
                // Nothing to do
            case '6.1.6':
                // Add the stocktaking module config field (which were part of Pickware WMS before)
                $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $this->Menu()->findOneBy(['controller' => 'ViisonStock']),
                    'controller' => 'ViisonPickwareERPStockTakeExport',
                    'action' => 'Index',
                    'label' => 'Inventurexport',
                    'class' => 'sprite-clipboard--arrow',
                    'active' => 1,
                    'position' => 15,
                ]);

                $showCurrentStockWhenStocktakingWMSValue = $database->fetchOne(
                    'SELECT cv.value
                    FROM s_core_config_elements ce
                    INNER JOIN s_core_config_forms cf ON cf.id = ce.form_id
                    INNER JOIN s_core_plugins plugin ON plugin.id = cf.plugin_id
                    INNER JOIN s_core_config_values cv ON ce.id = cv.element_id
                    WHERE plugin.name = :pluginName
                    AND ce.name = :configElementName',
                    [
                        'pluginName' => 'ViisonPickwareMobile',
                        'configElementName' => 'showCurrentStockWhenStocktaking',
                    ]
                );
                $showCurrentStockWhenStocktaking = false;
                if ($showCurrentStockWhenStocktakingWMSValue) {
                    $showCurrentStockWhenStocktaking = unserialize(
                        $showCurrentStockWhenStocktakingWMSValue,
                        [
                            'allowed_classes' => false,
                        ]
                    );
                }
                $form->setElement(
                    'checkbox',
                    'showCurrentStockWhenStocktaking',
                    [
                        'label' => 'Erwarteten Lagerbestand bei Inventur anzeigen',
                        'description' => 'Bei der Einstellung "Ja" wird während einer Inventur der erwartete ' .
                            '(aktuelle) Bestand für einen Artikel auf einem Lagerplatz angezeigt. Stellen Sie diese ' .
                            'Option auf "Nein" um den erwarteten Bestand nicht anzuzeigen (Blindinventur).',
                        'value' => $showCurrentStockWhenStocktaking,
                    ]
                );
            case '6.2.0':
                // Nothing to do
            case '6.2.1':
                // Nothing to do
            case '6.2.2':
                // Nothing to do
            case '6.3.0':
                // Nothing to do
            case '6.4.0':
                // Nothing to do
            case '6.5.0':
                // Nothing to do
            case '6.5.1':
                // Nothing to do
            case '6.5.2':
                // Nothing to do
            case '6.5.3':
                // Nothing to do
            case '6.5.4':
                // Nothing to do
            case '6.5.5':
                // Nothing to do
            case '6.6.0':
                // Nothing to do
            case '6.6.1':
                // Nothing to do
            case '6.6.2':
                // Nothing to do
            case '6.6.3':
                // Nothing to do
            case '6.6.4':
                // Nothing to do
            case '6.6.5':
                // Nothing to do
            case '6.6.6':
                // Nothing to do
            case '6.6.7':
                // Remove legacy stock valuation
                $menuInstallationHelper->deleteMenuItemFromDatabase([
                    'parent' => $this->Menu()->findOneBy(['controller' => 'ViisonStock']),
                    'label' => 'Bewerteter Warenbestand',
                ]);
                $database->query('
                    CREATE TABLE IF NOT EXISTS `pickware_erp_stock_valuation_reports` (
                        `id` INT(11) NOT NULL AUTO_INCREMENT,
                        `reportingDay` DATE NOT NULL,
                        `untilDate` DATETIME NULL,
                        `generated` TINYINT(1) NOT NULL,
                        `comment` TEXT COLLATE utf8_unicode_ci,
                        `preview` TINYINT(1) NOT NULL,
                        `method` VARCHAR(255) NOT NULL,
                        `warehouseId` INT(11) NULL,
                        `warehouseName` VARCHAR(255) NOT NULL,
                        `warehouseCode` VARCHAR(255) NOT NULL,
                        PRIMARY KEY (`id`),
                        CONSTRAINT `FK_D568356FE8DE5B58`
                            FOREIGN KEY (`warehouseId`)
                            REFERENCES `pickware_erp_warehouses` (`id`)
                            ON DELETE SET NULL
                            ON UPDATE RESTRICT
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;
                ');
                $database->query('
                    CREATE TABLE IF NOT EXISTS `pickware_erp_stock_valuation_report_rows` (
                        `id` INT(11) NOT NULL AUTO_INCREMENT,
                        `reportId` INT(11) NOT NULL,
                        `articleDetailId` INT(11) UNSIGNED NULL,
                        `articleDetailName` VARCHAR(255) NOT NULL,
                        `articleDetailNumber` VARCHAR(255) NOT NULL,
                        `stock` INT(11) NOT NULL,
                        `valuationNet` DECIMAL(10,2) NULL,
                        `valuationGross` DECIMAL(10,2) NULL,
                        `taxRate` FLOAT NOT NULL,
                        `averagePurchasePriceNet` DECIMAL(10,2) NOT NULL,
                        `surplusStock` INT(11) DEFAULT 0,
                        `surplusPurchasePriceNet` DECIMAL(10,2) NULL,
                        PRIMARY KEY (`id`),
                        UNIQUE KEY `UNIQ_14C2317ED0871F1C` (`reportId`, `articleDetailId`),
                        CONSTRAINT `FK_14C2317ED0871F1C`
                            FOREIGN KEY (`articleDetailId`)
                            REFERENCES `s_articles_details` (`id`)
                            ON DELETE SET NULL
                            ON UPDATE RESTRICT,
                        CONSTRAINT `FK_14C2317EBB333E0D`
                            FOREIGN KEY (`reportId`)
                            REFERENCES `pickware_erp_stock_valuation_reports` (`id`)
                            ON DELETE CASCADE
                            ON UPDATE RESTRICT
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;
                ');
                $database->query('
                    CREATE TABLE IF NOT EXISTS `pickware_erp_stock_valuation_report_purchases` (
                        `id` INT(11) NOT NULL AUTO_INCREMENT,
                        `reportRowId` INT(11) NOT NULL,
                        `date` DATETIME NOT NULL,
                        `purchasePriceNet` DECIMAL(10,2) NOT NULL,
                        `quantity` INT(11) NOT NULL,
                        `quantityUsedForValuation` INT NOT NULL DEFAULT 0,
                        `type` VARCHAR(255) COLLATE utf8_unicode_ci NOT NULL,
                        `purchaseStockLedgerEntryId` INT(11) NULL,
                        `carryOverReportRowId` INT(11) NULL,
                        PRIMARY KEY (`id`),
                        CONSTRAINT `FK_4B584ED2F3673EDB`
                            FOREIGN KEY (`reportRowId`)
                            REFERENCES `pickware_erp_stock_valuation_report_rows` (`id`)
                            ON DELETE CASCADE
                            ON UPDATE RESTRICT,
                        CONSTRAINT `FK_4B584ED2558A164F`
                            FOREIGN KEY (`purchaseStockLedgerEntryId`)
                            REFERENCES pickware_erp_stock_ledger_entries (`id`)
                            ON DELETE SET NULL
                            ON UPDATE RESTRICT,
                        CONSTRAINT `FK_4B584ED24301653`
                            FOREIGN KEY (`carryOverReportRowId`)
                            REFERENCES pickware_erp_stock_valuation_report_rows (`id`)
                            ON DELETE SET NULL
                            ON UPDATE RESTRICT
                        ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;
                ');
                $menuInstallationHelper->ensureMenuItemInDatabaseIs([
                    'parent' => $this->Menu()->findOneBy(['controller' => 'ViisonStock']),
                    'controller' => 'ViisonPickwareERPStockValuation',
                    'action' => 'Index',
                    'label' => 'Bewerteter Warenbestand',
                    'class' => 'sprite-book-open-list',
                    'active' => 1,
                    'position' => 11,
                ]);
                $form->setElement(
                    'select',
                    'stockValuationMethod',
                    [
                        'label' => 'Methode für Bestandsbewertung',
                        'description' => 'Wählen Sie hier die Methode aus, die verwendet werden soll um den Bestand ' .
                            'ihres Lagers zu bewerten (Bewerteter Warenbestand).',
                        'value' => Report::METHOD_LIFO,
                        'store' => [
                            [
                                Report::METHOD_LIFO,
                                'LIFO (Last In - First Out)',
                            ],
                            [
                                Report::METHOD_FIFO,
                                'FIFO (First In - First Out)',
                            ],
                            [
                                Report::METHOD_AVERAGE,
                                'Gewogener Durchschnitt',
                            ],
                        ],
                        'multiSelect' => false,
                        'editable' => false,
                    ]
                );
                $modelManager->flush($form);
            case '6.7.0':
                // Nothing to do
            case '6.7.1':
                // Nothing to do
            case '6.7.2':
                // Nothing to do
            case '6.7.3':
                // Nothing to do
            case '6.8.0':
                // Nothing to do
            case '6.8.1':
                $database->query(
                    'CREATE TABLE IF NOT EXISTS `pickware_erp_stock_valuation_temp_stocks` (
                        `id` INT(11) NOT NULL AUTO_INCREMENT,
                        `reportId` INT(11),
                        `articleDetailId` INT(11),
                        `stock` INT(11),
                        `averagePurchasePriceNet` DECIMAL(10,2),
                        `valuationNet` DECIMAL(10,2) NULL,
                        -- If there is more stock than the sum of purchased stock, this is the surplus stock, that is not
                        -- included in purchases
                        `surplusStock` INT(11) DEFAULT 0,
                        -- The surplus stock will be valued by a "guessed" purchase price. This purchase price is saved for
                        -- documentation.
                        `surplusPurchasePriceNet` DECIMAL(10,2) NULL,
                        PRIMARY KEY (`id`),
                        INDEX `IDX_CFFF835FBB333E0D` (`reportId`),
                        INDEX `IDX_CFFF835FD0871F1C` (`articleDetailId`),
                        CONSTRAINT `FK_CFFF835FBB333E0D`
                            FOREIGN KEY (`reportId`)
                            REFERENCES pickware_erp_stock_valuation_reports (`id`)
                            ON DELETE CASCADE
                            ON UPDATE CASCADE
                    );'
                );
                $database->query(
                    'CREATE TABLE IF NOT EXISTS pickware_erp_stock_valuation_temp_purchases (
                        `id` INT(11) NOT NULL AUTO_INCREMENT,
                        `reportId` INT(11),
                        `articleDetailId` INT(11),
                        `quantity` INT(11),
                        `quantityUsedForValuation` INT(11) DEFAULT 0,
                        `purchasePriceNet` DECIMAL(10,2),
                        `date` DATETIME,
                        `type` VARCHAR(255),
                        `averagePurchasePriceNet` DECIMAL(10,2) NULL,
                        `purchaseStockLedgerEntryId` INT(11),
                        `carryOverReportRowId` INT(11),
                        PRIMARY KEY (`id`),
                        INDEX `IDX_6CA2D9A9BB333E0D` (`reportId`),
                        INDEX `IDX_6CA2D9A9D0871F1C` (`articleDetailId`),
                        CONSTRAINT `FK_6CA2D9A9BB333E0D`
                            FOREIGN KEY (`reportId`)
                            REFERENCES pickware_erp_stock_valuation_reports (`id`)
                            ON DELETE CASCADE
                            ON UPDATE CASCADE
                    );'
                );
                $sqlHelper->addColumnIfNotExists(
                    'pickware_erp_stock_valuation_reports',
                    'generationStep',
                    'VARCHAR(255) NULL'
                );
            case '6.9.0':
                // Nothing to do
            case '6.9.1':
                // Nothing to do
            case '6.10.0':
                // Nothing to do
            case '6.10.1':
                // Nothing to do
            case '6.10.2':
                // Next release

                // *** *** *** *** ***
                // NEVER REMOVE THE FOLLOWING BREAK! All updates must be added above this comment block!
                // *** *** *** *** ***
                break;
            default:
                throw InstallationException::updateFromVersionNotSupported(
                    $this,
                    $oldVersion,
                    $this->getVersion(),
                    self::MIN_SUPPORTED_UPDATE_VERSION
                );
        }

        // Run these steps after every update
        $this->updateConfigFormTranslations();
        $this->removeObsoletePluginFiles();

        // Make sure the last safe uninstall version is deleted to allow installing the plugin again later
        $this->get('viison_common.hidden_config_storage')->removeConfigValue(
            self::CONFIG_ELEMENT_NAME_LAST_SAFE_UNINSTALL_VERSION
        );
    }

    /**
     * @inheritdoc
     */
    protected function runUninstall($deleteData)
    {
        if (!$deleteData) {
            $this->doUninstall($deleteData);

            return;
        }
        // Prepare a lock forcing the user to confirm the uninstallation
        $snippetNamespace = $this->getBootstrapSnippetManager()->getNamespace('bootstrap/viison_pickware_erp/main');
        $confirmUninstallLock = new ExpiringLock(
            static::UNINSTALL_CONFIRM_LOCK,
            $snippetNamespace->get('lock_message/uninstall_confirm_lock_description'),
            static::UNINSTALL_CONFIRM_LOCK_WAIT_TIME_IN_SECONDS,
            $this->get('models')
        );

        // Check confirm lock
        $uninstallConfirmStatus = $confirmUninstallLock->tryAcquireLock();
        if ($uninstallConfirmStatus->hasBeenAcquired()) {
            // If we could acquire the lock, there was no previous confirmation or the confirmation has expired
            throw new \Exception($uninstallConfirmStatus->getDescription());
        }

        try {
            $this->doUninstall($deleteData);
        } finally {
            // In any case, after an attempt to uninstall, release the confirmation lock, so any further attempts will
            // again require confirmation
            $confirmUninstallLock->releaseUnconditionally();
        }
    }

    /**
     * @param bool $deleteData
     */
    private function doUninstall($deleteData)
    {
        // Make sure neither ViisonPickwareMobile nor ViisonPickwarePOS are installed
        if (ViisonCommonUtil::isPluginInstalled(null, 'ViisonPickwareMobile')) {
            throw InstallationException::dependentPluginExists(
                $this,
                ViisonCommonUtil::getPlugin('ViisonPickwareMobile')->getLabel()
            );
        }
        if (ViisonCommonUtil::isPluginInstalled(null, 'ViisonPickwarePOS')) {
            throw InstallationException::dependentPluginExists(
                $this,
                ViisonCommonUtil::getPlugin('ViisonPickwarePOS')->getLabel()
            );
        }

        if (!$deleteData) {
            // Just save the currently installed plugin version to be able to detect 'safe' uninstalls of
            // incompatible versions
            $this->get('viison_common.hidden_config_storage')->setConfigValue(
                self::CONFIG_ELEMENT_NAME_LAST_SAFE_UNINSTALL_VERSION,
                'text',
                $this->getPlugin()->getVersion()
            );

            return;
        }

        /** @var HiddenConfigStorageService $hiddenConfigStorageService */
        $hiddenConfigStorageService = $this->get('viison_common.hidden_config_storage');
        // Make sure the last safe uninstall version is deleted to allow installing the plugin again later
        $hiddenConfigStorageService->removeConfigValue(self::CONFIG_ELEMENT_NAME_LAST_SAFE_UNINSTALL_VERSION);

        // Prepare some helpers
        /* @var $modelManager Shopware\Components\Model\ModelManager */
        $modelManager = $this->get('models');
        $database = $this->get('db');
        $sqlHelper = new SQLHelper($database);
        $attributeColumnUninstallationHelper = new AttributeColumnUninstaller($modelManager, $sqlHelper);

        // Remove custom attribute columns
        $attributeColumnUninstallationHelper->removeAttributeColumnsIfExist([
            new AttributeColumnDescription(
                's_articles_attributes',
                'pickware_incoming_stock'
            ),
            new AttributeColumnDescription(
                's_articles_attributes',
                'viison_incoming_stock'
            ),
            new AttributeColumnDescription(
                's_articles_attributes',
                'pickware_physical_stock_for_sale'
            ),
            new AttributeColumnDescription(
                's_articles_attributes',
                'viison_physical_stock_for_sale'
            ),
            new AttributeColumnDescription(
                's_articles_attributes',
                'pickware_stock_management_disabled'
            ),
            new AttributeColumnDescription(
                's_articles_attributes',
                'viison_not_relevant_for_stock_manager'
            ),
            new AttributeColumnDescription(
                's_order_attributes',
                'pickware_last_changed'
            ),
            new AttributeColumnDescription(
                's_order_attributes',
                'viison_last_changed'
            ),
            new AttributeColumnDescription(
                's_order_details_attributes',
                'pickware_canceled_quantity'
            ),
            new AttributeColumnDescription(
                's_order_details_attributes',
                'viison_canceled_quantity'
            ),
            new AttributeColumnDescription(
                's_order_attributes',
                'pickware_return_shipment_status_id'
            ),
            new AttributeColumnDescription(
                's_articles_attributes',
                'pickware_stock_initialization_time'
            ),
            new AttributeColumnDescription(
                's_articles_attributes',
                'viison_stock_initialization_time'
            ),
            new AttributeColumnDescription(
                's_articles_attributes',
                'pickware_stock_initialized'
            ),
            new AttributeColumnDescription(
                's_articles_attributes',
                'viison_stock_initialized'
            ),
            // Legacy attributes
            new AttributeColumnDescription(
                's_articles_attributes',
                'viison_physical_stock'
            ),
        ]);

        // Drop all custom tables
        $database->query(
            'SET FOREIGN_KEY_CHECKS = 0;
            -- Old table names (< 5.0.0)
            DROP TABLE IF EXISTS s_plugin_pickware_article_detail_property_types;
            DROP TABLE IF EXISTS s_plugin_pickware_property_types;
            DROP TABLE IF EXISTS s_plugin_pickware_reshipment_attachments;
            DROP TABLE IF EXISTS s_plugin_pickware_reshipment_items;
            DROP TABLE IF EXISTS s_plugin_pickware_reshipment_status;
            DROP TABLE IF EXISTS s_plugin_pickware_reshipments;
            DROP TABLE IF EXISTS s_plugin_pickware_stock_entry_item_properties;
            DROP TABLE IF EXISTS s_plugin_pickware_stock_entry_items;
            DROP TABLE IF EXISTS s_plugin_pickware_stocks;
            DROP TABLE IF EXISTS s_plugin_pickware_supplier_article_details;
            DROP TABLE IF EXISTS s_plugin_pickware_supplier_fabricators;
            DROP TABLE IF EXISTS s_plugin_pickware_supplier_order_article_detail_states;
            DROP TABLE IF EXISTS s_plugin_pickware_supplier_order_article_details;
            DROP TABLE IF EXISTS s_plugin_pickware_supplier_order_attachments;
            DROP TABLE IF EXISTS s_plugin_pickware_supplier_order_states;
            DROP TABLE IF EXISTS s_plugin_pickware_supplier_orders;
            DROP TABLE IF EXISTS s_plugin_pickware_suppliers;
            DROP TABLE IF EXISTS s_plugin_pickware_warehouse_bin_location_article_details;
            DROP TABLE IF EXISTS s_plugin_pickware_warehouse_bin_location_stock_snapshots;
            DROP TABLE IF EXISTS s_plugin_pickware_warehouse_bin_locations;
            DROP TABLE IF EXISTS s_plugin_pickware_warehouse_reserved_stocks;
            DROP TABLE IF EXISTS s_plugin_pickware_warehouse_stocks;
            DROP TABLE IF EXISTS s_plugin_pickware_warehouses;
            -- New table names (>= 5.0.0)
            DROP TABLE IF EXISTS `pickware_erp_article_detail_barcode_labels`;
            DROP TABLE IF EXISTS `pickware_erp_article_detail_bin_location_mappings`;
            DROP TABLE IF EXISTS `pickware_erp_article_detail_item_properties`;
            DROP TABLE IF EXISTS `pickware_erp_article_detail_supplier_mappings`;
            DROP TABLE IF EXISTS `pickware_erp_barcode_label_preset_blocks`;
            DROP TABLE IF EXISTS `pickware_erp_barcode_label_presets`;
            DROP TABLE IF EXISTS `pickware_erp_bin_location_barcode_labels`;
            DROP TABLE IF EXISTS `pickware_erp_bin_location_stock_snapshots`;
            DROP TABLE IF EXISTS `pickware_erp_bin_locations`;
            DROP TABLE IF EXISTS `pickware_erp_item_properties`;
            DROP TABLE IF EXISTS `pickware_erp_manufacturer_supplier_mappings`;
            DROP TABLE IF EXISTS `pickware_erp_order_stock_reservations`;
            DROP TABLE IF EXISTS `pickware_erp_stock_valuation_report_purchases`;
            DROP TABLE IF EXISTS `pickware_erp_stock_valuation_report_rows`;
            DROP TABLE IF EXISTS `pickware_erp_stock_valuation_reports`;
            DROP TABLE IF EXISTS `pickware_erp_rest_api_idempotent_operations`;
            DROP TABLE IF EXISTS `pickware_erp_rest_api_requests`;
            DROP TABLE IF EXISTS `pickware_erp_return_shipment_attachments`;
            DROP TABLE IF EXISTS `pickware_erp_return_shipment_internal_comments`;
            DROP TABLE IF EXISTS `pickware_erp_return_shipment_items`;
            DROP TABLE IF EXISTS `pickware_erp_return_shipment_statuses`;
            DROP TABLE IF EXISTS `pickware_erp_return_shipments`;
            DROP TABLE IF EXISTS `pickware_erp_stock_item_property_values`;
            DROP TABLE IF EXISTS `pickware_erp_stock_items`;
            DROP TABLE IF EXISTS `pickware_erp_stock_ledger_entries`;
            DROP TABLE IF EXISTS `pickware_erp_supplier_order_attachments`;
            DROP TABLE IF EXISTS `pickware_erp_supplier_order_item_statuses`;
            DROP TABLE IF EXISTS `pickware_erp_supplier_order_items`;
            DROP TABLE IF EXISTS `pickware_erp_supplier_order_statuses`;
            DROP TABLE IF EXISTS `pickware_erp_supplier_orders`;
            DROP TABLE IF EXISTS `pickware_erp_suppliers`;
            DROP TABLE IF EXISTS `pickware_erp_warehouse_article_detail_configurations`;
            DROP TABLE IF EXISTS `pickware_erp_warehouse_article_detail_stock_counts`;
            DROP TABLE IF EXISTS `pickware_erp_warehouses`;
            SET FOREIGN_KEY_CHECKS = 1;'
        );

        // Remove custom number ranges
        $database->delete('s_order_number', 'name = \'viison_supplier_order\'');
        $database->delete('s_order_number', 'name = \'' . ReturnShipmentProcessorService::NUMBER_RANGE_NAME . '\'');

        // Remove custom media album
        $mediaAlbumUninstallationHelper = new MediaAlbumUninstallationHelper($modelManager);
        $mediaAlbumUninstallationHelper->removeMediaAlbumsIfExist([
            'Pickware Lieferantenbestellungen',
            'Pickware Retouren',
        ]);

        /* Reload Config\Elements from database. This is necessary so uninstall does not fail during tests.
         * Without reloading, the elements will considered detached entities during a test. */
        $formElements = $this->Form()->getElements();
        $newFormElements = [];
        foreach ($formElements as $key => $formElement) {
            $existingFormElement = $modelManager->getRepository(Element::class)->findOneBy([
                'form' => $this->Form(),
                'name' => $formElement->getName(),
            ]);
            $newFormElements[$key] = ($existingFormElement) ?: $formElement;
        }
        foreach ($newFormElements as $key => $newFormElement) {
            $formElements->set($key, $newFormElement);
        }

        /* Reload Config\Form from database. This is necessary so uninstall does not fail during tests.
         * Without reloading, \Shopware_Components_Plugin_Namespace#removeForm will consider it a detached entity during
         * a test. */
        if ($this->Form()->getId() !== null) {
            $existingForm = $modelManager->find(Form::class, $this->Form()->getId());
            if ($existingForm) {
                $this->form = $existingForm;
            }
        }

        // We need to reset the values of the Shopware snippets that we updated during installation to their original values.
        // To do so, we load the original values from the Shopware ini files using the same method as the sw:snippets:to:db
        // command. To prevent snippets that have been customized by the user from being overwritten, we only import snippets
        // that are marked as non-dirty. Since we want to overwrite the snippets that were replaced by our plugin, we need
        // to mark these as non-dirty.
        foreach (self::SHOPWARE_SNIPPET_REPLACEMENTS as $snippetReplacement) {
            foreach ($snippetReplacement['values'] as $locale => $value) {
                $locale = Shopware()->Models()->getRepository(Locale::class)->findOneBy(['locale' => $locale]);
                $this->changeSnippetDirtyState($snippetReplacement['name'], $snippetReplacement['namespace'], $locale, false);
            }
        }

        /** @var \Shopware\Components\Snippet\DatabaseHandler $databaseLoader */
        $databaseLoader = $this->get('shopware.snippet_database_handler');
        $sourceDir = Shopware()->Container()->getParameter('kernel.root_dir') . DIRECTORY_SEPARATOR . 'snippets' . DIRECTORY_SEPARATOR;
        $databaseLoader->loadToDatabase($sourceDir, false); // Set force to false so that snippets marked as 'dirty' are not overwritten

        // Make sure to remove other hidden config fields since they are not removed with the plugin form
        $hiddenConfigStorageService->removeConfigValue('pickwareErpDisplayAboutWindow');

        // Remove flag that barcode less barcode label template has been installed.
        $hiddenConfigStorage = $this->get('viison_common.hidden_config_storage');
        $hiddenConfigStorage->removeConfigValue('viisonPickwareCommonIsBarcodeLessBarcodeLabelPresetInstalled');
        $hiddenConfigStorage->removeConfigValue('pickwareErpBarcodeLessBarcodeLabelPresetIsInstalled');
    }

    /**
     * @inheritdoc
     */
    protected function runActivation()
    {
        // Nothing to do here
    }

    /**
     * @inheritdoc
     */
    protected function runDeactivation()
    {
        // Make sure neither ViisonPickwareMobile nor ViisonPickwarePOS are installed and active
        if (ViisonCommonUtil::isPluginInstalledAndActive(null, 'ViisonPickwareMobile')) {
            throw InstallationException::dependentPluginExists(
                $this,
                ViisonCommonUtil::getPlugin('ViisonPickwareMobile')->getLabel()
            );
        }
        if (ViisonCommonUtil::isPluginInstalledAndActive(null, 'ViisonPickwarePOS')) {
            throw InstallationException::dependentPluginExists(
                $this,
                ViisonCommonUtil::getPlugin('ViisonPickwarePOS')->getLabel()
            );
        }
    }

    /**
     * Registers the plugin's namespaces.
     */
    public function afterInit()
    {
        $this->loadDependencies();
        $this->loadPlugin();
    }

    /* Events & Hooks */

    /**
     * This callback function is triggered at the very beginning of the dispatch process and allows
     * us to register additional events on the fly.
     *
     * @param \Enlight_Event_EventArgs $args
     */
    public function onStartDispatch(\Enlight_Event_EventArgs $args)
    {
        // Nothing to do here, since the dynamic subscribers were already registered in 'afterInit()'
    }

    /**
     * Adds the pickware:erp:stock:init CLI command.
     *
     * @param \Enlight_Event_EventArgs $args
     * @return Commands\InitializeAllArticlesCommand
     */
    public function onAddConsoleCommand(\Enlight_Event_EventArgs $args)
    {
        return new Commands\InitializeAllArticlesCommand();
    }

    /* Other */

    /**
     * Uses the dependency loader to load the namespaces and susbcribers of all required,
     * shared dependencies.
     */
    private function loadDependencies()
    {
        // Require all shared dependencies
        $loader = VIISON\ShopwarePluginDependencyLoader\Loader::getInstance();
        $loader->requireDependencies($this->Path(), [
            'ViisonCommon'
        ]);
        Shopware\Plugins\ViisonCommon\Classes\PickwareAutoloadDependencyLoader::ensureDependenciesLoaded();

        // Add the subscribers of ViisonCommon
        $viisonCommonSubscriberRegistrator = new Shopware\Plugins\ViisonCommon\Classes\SubscriberRegistrator($this);
        $viisonCommonSubscriberRegistrator->registerSubscribers();

        // Load the Shopware polyfill
        require_once __DIR__ . '/ViisonCommon/Polyfill/Loader.php';
    }

    /**
     * First checks whether the plugin is installed and is active and, ff it is, first the namespaces
     * of this plugin are registered with the class loader, before all subscribers are instanciated
     * and added to the event manager.
     */
    private function loadPlugin()
    {
        // Make sure this plugin is installed and active before registering the susbcribers
        if (!$this->isInstalledAndActive()) {
            return;
        }

        // Create the subscribers, which should be added even when using an expired test version
        $subscribers = [
            new Subscribers\Backend\PluginTestNotificationSubscriber($this),
            new ViisonCommonSubscribers\ViewLoading($this, 'ViisonPickwareERP'),
        ];
        if (!$this->pluginTestExpired()) {
            // Add all other subscribers of this plugin
            $subscribers = array_merge($subscribers, [
                new Subscribers\Backend\BackendAnalyticsSubscriber($this),
                new Subscribers\Backend\BackendArticleListSubscriber($this),
                new Subscribers\Backend\BackendArticleSubscriber($this),
                new Subscribers\Backend\BackendBaseSubscriber($this),
                new Subscribers\Backend\BackendIndexSubscriber($this),
                new Subscribers\Backend\BackendMailSubscriber($this),
                new Subscribers\Backend\BackendOrderSubscriber($this),
                new Subscribers\Backend\BackendOverviewSubscriber($this),
                new Subscribers\Backend\BackendPluginManagerSubscriber($this),
                new Subscribers\Backend\VariantGenerationSubscriber($this),
                new Subscribers\BarcodeLabel\ArticleBarcodeLabelTypeSubscriber($this),
                new Subscribers\BarcodeLabel\BinLocationBarcodeLabelTypeSubscriber($this),
                new Subscribers\CompatibilityCheckSubscriber($this),
                new Subscribers\Components\DocumentComponentSubscriber($this),
                new Subscribers\ControllersSubscriber($this),
                new Subscribers\CoreModules\OrderModuleSubscriber($this),
                new Subscribers\Doctrine\DoctrineSubscriberRegistrationSubscriber($this),
                new Subscribers\Frontend\FrontendAccountSubscriber($this),
                new Subscribers\MediaBundle\GarbageCollectorSubscriber($this),
                new Subscribers\Models\ArticleAttributeEntityLifecycleSubscriber($this),
                new Subscribers\Models\ArticleDetailEntityLifecycleSubscriber($this),
                new Subscribers\Models\OrderDetailEntityLifecycleSubscriber($this),
                new Subscribers\Models\ViisonPickwareERP\SupplierOrderItemEntityLifecycleSubscriber($this),
                new Subscribers\Models\ViisonPickwareERP\WarehouseArticleDetailConfigurationEntityLifecycleSubscriber($this),
                new Subscribers\Models\ViisonPickwareERP\WarehouseEntityLifecycleSubscriber($this),
                new SwagImportExportIntegrationSubscriber($this),
                new Subscribers\RestApi\RestApiControllerSubscriber($this),
                new Subscribers\RestApi\RestApiOrdersSubscriber($this),
                new Subscribers\RestApi\RestApiVariantUpdateSubscriber($this),
                new Subscribers\RestApiRequestLogFilterSubscriber($this),
                new Subscribers\RestApiRouterSubscriber($this),
                new Subscribers\ServicesSubscriber($this),
                new Subscribers\SubApplicationRegistrationSubscriber($this),
                new ViisonCommonSubscribers\DocumentTypeHidingSubscriber($this, [
                    'numberRangeNames' => [
                        self::SUPPLIER_ORDER_DOCUMENT_NUMBERS,
                    ],
                    'ids' => [
                        DocumentUtil::DOCUMENT_TYPE_ID_CANCELLATION,
                    ],
                ]),
                new ViisonCommonSubscribers\IndexPopupWindowQueue($this, 'ViisonPickwareERPAboutPopup'),
                $this->createMigrationSubscriber(),
            ]);
        }

        // Make sure that the subscribers are only added once
        if (!$this->isSubscriberRegistered($subscribers[0])) {
            $eventManager = $this->get('events');
            foreach ($subscribers as $subscriber) {
                $eventManager->addSubscriber($subscriber);
            }
        }
    }

    /**
     * Helper method that registers only the services subscriber for this plugin. This is required in order to use
     * the plugin's services during installation or activation if the plugin is not active at the moment (i.e. fresh
     * installation or activation).
     */
    private function loadServicesForInstallation()
    {
        $servicesSubscriber = new Subscribers\ServicesSubscriber($this);
        if (!$this->isSubscriberRegistered($servicesSubscriber)) {
            $this->get('events')->addSubscriber($servicesSubscriber);
        }
    }

    /**
     * @inheritdoc
     */
    protected function createMigrationSubscriber()
    {
        return new Subscribers\MigrationSubscriber($this);
    }

    /* Setup helper methods */

    /**
     * Creates the custom email templates contained in this plugin, including the following:
     *  * viisonSupplierOrder - The mail used for sending orders to supplier
     *
     * This method should only be used during the initial installation and not in update steps
     * since users will lose their custom changes to any of these emails otherwise.
     */
    private function createEmailTemplates()
    {
        $modelManager = $this->get('models');
        // Add/update the custom email templates
        $customMails = [
            'viisonSupplierOrder' => [
                'fromMail' => '{config name=mail}',
                'fromName' => '{config name=shopName}',
                'subject' => 'Nachbestellung {$ordernumber}{if $supplier.customerNumber} (Kundennummer: {$supplier.customerNumber}){/if}',
                'fileName' => 'viisonSupplierOrder',
            ],
            self::DOCUMENT_MAILER_MAIL_TEMPLATE_NAME . '_archive' => [
                'fromMail' => '{$sender}',
                'fromName' => '',
                'subject' => '{$documentPrefix} {$documentIdentifier} {$subjectOrderString} {$orderNumber}',
                'fileName' => 'archive_mail',
            ],
        ];
        foreach ($customMails as $templateName => $data) {
            // Try to get the mail
            $mail = $modelManager->getRepository(Mail::class)->findOneBy([
                'name' => $templateName,
            ]);
            if (!$mail) {
                // Create a new mail template
                $mail = new Mail();
                $mail->setName($templateName);
                $mail->setFromMail($data['fromMail']);
                $mail->setFromName($data['fromName']);
                $mail->setSubject($data['subject']);
                $mail->setMailtype(1);
                $modelManager->persist($mail);
            }

            // Add the plain email content
            $content = file_get_contents($this->Path() . 'Views/mails/' . $data['fileName'] . '.tpl');
            if ($content !== false) {
                $mail->setContent($content);
            }

            // Save the mail
            $modelManager->flush($mail);
        }
    }

    /**
     * Update the Shopware snippets according to the definitions in the SHOPWARE_SNIPPET_REPLACEMENTS constant. The updated
     * snippets are marked dirty so that they do not get reset on Shopware updates.
     *
     * @throws Exception
     */
    private function loadSnippetReplacementsToDB()
    {
        $databaseWriter = new \Shopware\Components\Snippet\Writer\DatabaseWriter(Shopware()->Models()->getConnection());

        foreach (self::SHOPWARE_SNIPPET_REPLACEMENTS as $snippetReplacement) {
            foreach ($snippetReplacement['values'] as $locale => $value) {
                /** @var \Shopware\Models\Shop\Locale $locale */
                $locale = Shopware()->Models()->getRepository(Locale::class)->findOneBy(['locale' => $locale]);
                $databaseWriter->write([$snippetReplacement['name'] => $value], $snippetReplacement['namespace'], $locale->getId(), 1);
                // Get the snippet that was updated and set it to dirty=true so that it does not get overwritten by Shopware updates
                $this->changeSnippetDirtyState($snippetReplacement['name'], $snippetReplacement['namespace'], $locale, true);
            }
        }
    }

    /**
     * Helper function that changes the dirty state of the snippet with the given name, namespace and locale.
     *
     * @param string $name
     * @param string $namespace
     * @param \Shopware\Models\Shop\Locale $locale
     * @param bool $dirty
     */
    private function changeSnippetDirtyState($name, $namespace, $locale, $dirty)
    {
        $snippet = $this->get('models')->getRepository(Snippet::class)->findOneBy([
            'name' => $name,
            'namespace' => $namespace,
            'localeId' => $locale->getId(),
        ]);
        if (!$snippet) {
            return;
        }

        // Refresh snippet, since it may be updated during the write action
        // of the snippet database writer in self::loadSnippetReplacementsToDB()
        // and an old version is cached by the entity manager.
        $this->get('models')->refresh($snippet);

        $snippet->setDirty($dirty);
        $this->get('models')->flush($snippet);
    }

    /**
     * Calls the given callback function for each document type of the shop.
     *
     * @param callable $callback
     */
    private function forEachDocumentType($callback)
    {
        $documents = $this->get('models')->getRepository(DocumentType::class)->findAll();
        foreach ($documents as $document) {
            $callback($document);
        }
    }

    /**
     * Creates a pick list document type along with its number range, copies all necessary document boxes from
     * Shopware's invoice template and creates its own specific document boxes.
     */
    private function ensurePickListDocumentType()
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        /** @var Zend_Db_Adapter_Abstract $databaseConnection */
        $databaseConnection = $this->get('db');
        $documentInstallationHelper = new DocumentInstallationHelper($entityManager);
        $numberRangeInstallationHelper = new OrderNumberInstallationHelper($databaseConnection);

        // Create number range
        $numberRangeInstallationHelper->createSequenceNumberGeneratorIfNotExists(
            self::PICK_LIST_NUMBER_RANGE_NAME,
            self::PICK_LIST_NUMBER_RANGE_START,
            self::PICK_LIST_NUMBER_RANGE_DESCRIPTION
        );

        // Create document type
        $pickListDocumentType = $documentInstallationHelper->ensureDocumentType(
            self::PICK_LIST_DOCUMENT_TYPE_KEY,
            self::PICK_LIST_DOCUMENT_TYPE_NAME,
            self::PICK_LIST_DOCUMENT_TYPE_TEMPLATE_NAME,
            self::PICK_LIST_NUMBER_RANGE_NAME,
            25,
            10,
            20,
            10,
            15
        );

        $invoiceDocumentType = DocumentUtil::getInvoiceDocumentType();
        // Copy invoice elements. Leave out: Footer, Content-Amount, Content-Info
        $documentBoxCopier = new DocumentBoxCopier(
            $this->get('models'),
            $this->get('db'),
            $invoiceDocumentType,
            $pickListDocumentType
        );

        $copyDocumentBoxes = [
            'Body',
            'Logo',
            'Header',
            'Header_Recipient',
            'Header_Box_Left',
            'Header_Box_Right',
            'Header_Box_Bottom',
            'Content',
            'Td',
            'Td_Name',
            'Td_Line',
            'Td_Head',
        ];
        $copyDocumentBoxStyle = [
            'Body',
            'Logo',
            'Header_Box_Bottom',
            'Td_Name',
            'Td_Line',
            'Td_Head',
        ];
        $copyDocumentBoxValue = [
            'Logo'
        ];

        foreach ($copyDocumentBoxes as $documentBoxName) {
            $documentBoxCopier->ensureCopiedDocumentBoxExists($documentBoxName);
            if (in_array($documentBoxName, $copyDocumentBoxStyle)) {
                $documentBoxCopier->copyDocumentBoxStyle($documentBoxName);
            }
            if (in_array($documentBoxName, $copyDocumentBoxValue)) {
                $documentBoxCopier->copyDocumentBoxValue($documentBoxName);
                $documentBoxCopier->copyDocumentElementTranslations();
            }
        }

        // Add missing boxes
        $documentInstallationHelper->ensureDocumentElement(
            $pickListDocumentType,
            'Signature',
            'Unterschrift',
            'font-size:14px; width: 38%; text-align:center; float: left; margin-top:16mm; border-top: 2px solid black;'
        );
        $documentInstallationHelper->ensureDocumentElement(
            $pickListDocumentType,
            'Info',
            '',
            'float: left; width: 60%; text-align:left; height: 20mm;'
        );

        // Update styles
        $updateStyles = [
            [
                'name' => 'Content',
                'style' => 'width: 170mm;',
            ],
            [
                'name' => 'Header',
                'style' => 'height: 30mm;',
            ],
            [
                'name' => 'Header_Recipient',
                'style' => 'float: left;',
            ],
            [
                'name' => 'Header_Box_Left',
                'style' => 'width: 120mm; height:30mm; float:left;',
            ],
            [
                'name' => 'Header_Box_Right',
                'style' => 'width: 45mm; height: 30mm; float:left; margin-left:5px;',
            ],
            [
                'name' => 'Content',
                'style' => 'width: 170mm; height: 140mm;',
            ],
            [
                'name' => 'Td',
                'style' => 'white-space:nowrap; padding: 5px 0; vertical-align: middle;',
            ],
        ];
        foreach ($updateStyles as $updateStyle) {
            $databaseConnection->query('UPDATE s_core_documents_box SET style = :style WHERE documentID = :documentId AND name = :name', [
                'style' => $updateStyle['style'],
                'documentId' => $pickListDocumentType->getId(),
                'name' => $updateStyle['name'],
            ]);
        }
    }

    /**
     * Inserts the email template, which is used to send a document of a specific type to the customer, into the database.
     *
     * @param DocumentType $document
     */
    private function createOrderDocumentEmailTemplate(DocumentType $document)
    {
        if (ViisonCommonUtil::assertMinimumShopwareVersion('5.5.0')) {
            // Shopware >= 5.5 brings its own document templates for sending order documents via mail.
            return;
        }

        $map = [
            1 => 'die Rechnung',
            2 => 'den Lieferschein',
            3 => 'die Gutschrift',
            4 => 'die Stornorechnung',
        ];

        $documentId = $document->getId();

        $emailTemplate = <<<'EMAIL_CONTENT'
{include file="string:{config name=emailheaderplain}"}

Sehr geehrte{if $userSalutation eq "mr"}r Herr{elseif $userSalutation eq "ms"} Frau{/if} {$userFirstName} {$userLastName},

vielen Dank für Ihre Bestellung bei {config name=shopName}. Im Anhang finden Sie %1$s zu Ihrer Bestellung als PDF.
Wir wünschen Ihnen noch einen schönen Tag.

Ihr Team von {config name=shopName}

{include file="string:{config name=emailfooterplain}"}
EMAIL_CONTENT;
        $documentPart = (array_key_exists($documentId, $map)) ? $map[$documentId] : 'das Dokument \'' . $document->getName() .'\'';
        $emailTemplate = sprintf($emailTemplate, $documentPart);

        $this->get('db')->query(
            'INSERT IGNORE INTO s_core_config_mails (name, frommail, fromname, subject, content, ishtml, mailtype, context)
            VALUES (:name, :fromMail, :fromName, :subject, :content, 0, 1, :context)',
            [
                'name' => self::DOCUMENT_MAILER_MAIL_TEMPLATE_NAME . '_DocType' . $documentId,
                'fromMail' => '{config name=mail}',
                'fromName' => '{config name=shopName}',
                'subject' => sprintf('%s zur Bestellung {$orderNumber}', $document->getName()),
                'content' => $emailTemplate,
                'context' => 'a:13:{s:7:"sConfig";a:0:{}s:5:"sShop";s:13:"Shopware Demo";s:8:"sShopURL";s:20:"http://shopware.demo";s:5:"theme";a:19:{s:10:"mobileLogo";s:47:"frontend/_public/src/img/logos/logo--mobile.png";s:10:"tabletLogo";s:47:"frontend/_public/src/img/logos/logo--tablet.png";s:19:"tabletLandscapeLogo";s:47:"frontend/_public/src/img/logos/logo--tablet.png";s:11:"desktopLogo";s:47:"frontend/_public/src/img/logos/logo--tablet.png";s:14:"appleTouchIcon";s:57:"frontend/_public/src/img/apple-touch-icon-precomposed.png";s:13:"brand-primary";s:7:"#D9400B";s:19:"brand-primary-light";s:41:"saturate(lighten(@brand-primary,12%), 5%)";s:15:"brand-secondary";s:7:"#5F7285";s:20:"brand-secondary-dark";s:29:"darken(@brand-secondary, 15%)";s:7:"body-bg";s:23:"darken(@gray-light, 5%)";s:10:"text-color";s:16:"@brand-secondary";s:15:"text-color-dark";s:21:"@brand-secondary-dark";s:10:"link-color";s:14:"@brand-primary";s:16:"link-hover-color";s:24:"darken(@link-color, 10%)";s:12:"font-size-h1";i:26;s:12:"font-size-h2";i:21;s:12:"font-size-h4";i:16;s:12:"font-size-h5";s:15:"@font-size-base";s:12:"font-size-h6";i:12;}s:6:"sOrder";a:40:{s:7:"orderID";s:2:"15";s:11:"ordernumber";s:5:"20001";s:12:"order_number";s:5:"20001";s:6:"userID";s:1:"2";s:10:"customerID";s:1:"2";s:14:"invoice_amount";s:6:"998.56";s:18:"invoice_amount_net";s:6:"839.13";s:16:"invoice_shipping";s:1:"0";s:20:"invoice_shipping_net";s:1:"0";s:9:"ordertime";s:19:"2012-08-30 10:15:54";s:6:"status";s:1:"0";s:8:"statusID";s:1:"0";s:7:"cleared";s:2:"17";s:9:"clearedID";s:2:"17";s:9:"paymentID";s:1:"4";s:13:"transactionID";s:0:"";s:7:"comment";s:0:"";s:15:"customercomment";s:0:"";s:3:"net";s:1:"1";s:5:"netto";s:1:"1";s:9:"partnerID";s:0:"";s:11:"temporaryID";s:0:"";s:7:"referer";s:0:"";s:11:"cleareddate";N;s:12:"cleared_date";N;s:12:"trackingcode";s:0:"";s:8:"language";s:1:"1";s:8:"currency";s:3:"EUR";s:14:"currencyFactor";s:1:"1";s:9:"subshopID";s:1:"1";s:10:"dispatchID";s:1:"9";s:10:"currencyID";s:1:"1";s:12:"cleared_name";s:4:"open";s:19:"cleared_description";s:5:"Offen";s:11:"status_name";s:4:"open";s:18:"status_description";s:5:"Offen";s:19:"payment_description";s:8:"Rechnung";s:20:"dispatch_description";s:16:"Standard Versand";s:20:"currency_description";s:4:"Euro";s:10:"attributes";a:7:{s:10:"attribute1";s:0:"";s:10:"attribute2";s:0:"";s:10:"attribute3";s:0:"";s:10:"attribute4";s:0:"";s:10:"attribute5";s:0:"";s:10:"attribute6";s:0:"";s:21:"pickware_last_changed";s:19:"2000-01-01 00:00:00";}}s:13:"sOrderDetails";a:3:{i:0;a:20:{s:14:"orderdetailsID";s:2:"42";s:7:"orderID";s:2:"15";s:11:"ordernumber";s:5:"20001";s:9:"articleID";s:3:"197";s:18:"articleordernumber";s:7:"SW10196";s:5:"price";s:7:"836.134";s:8:"quantity";s:1:"1";s:7:"invoice";s:7:"836.134";s:4:"name";s:20:"ESD Download Artikel";s:6:"status";s:1:"0";s:7:"shipped";s:1:"0";s:12:"shippedgroup";s:1:"0";s:11:"releasedate";s:10:"0000-00-00";s:5:"modus";s:1:"0";s:10:"esdarticle";s:1:"1";s:5:"taxID";s:1:"1";s:3:"tax";s:5:"19.00";s:8:"tax_rate";s:2:"19";s:3:"esd";s:1:"1";s:10:"attributes";a:7:{s:10:"attribute1";s:0:"";s:10:"attribute2";s:0:"";s:10:"attribute3";s:0:"";s:10:"attribute4";s:0:"";s:10:"attribute5";s:0:"";s:10:"attribute6";s:0:"";s:26:"pickware_canceled_quantity";s:1:"0";}}i:1;a:20:{s:14:"orderdetailsID";s:2:"43";s:7:"orderID";s:2:"15";s:11:"ordernumber";s:5:"20001";s:9:"articleID";s:1:"0";s:18:"articleordernumber";s:16:"SHIPPINGDISCOUNT";s:5:"price";s:2:"-2";s:8:"quantity";s:1:"1";s:7:"invoice";s:2:"-2";s:4:"name";s:15:"Warenkorbrabatt";s:6:"status";s:1:"0";s:7:"shipped";s:1:"0";s:12:"shippedgroup";s:1:"0";s:11:"releasedate";s:10:"0000-00-00";s:5:"modus";s:1:"4";s:10:"esdarticle";s:1:"0";s:5:"taxID";s:1:"1";s:3:"tax";s:5:"19.00";s:8:"tax_rate";s:2:"19";s:3:"esd";s:1:"0";s:10:"attributes";a:7:{s:10:"attribute1";s:0:"";s:10:"attribute2";s:0:"";s:10:"attribute3";s:0:"";s:10:"attribute4";s:0:"";s:10:"attribute5";s:0:"";s:10:"attribute6";s:0:"";s:26:"pickware_canceled_quantity";s:1:"0";}}i:2;a:20:{s:14:"orderdetailsID";s:2:"44";s:7:"orderID";s:2:"15";s:11:"ordernumber";s:5:"20001";s:9:"articleID";s:1:"0";s:18:"articleordernumber";s:19:"sw-payment-absolute";s:5:"price";s:1:"5";s:8:"quantity";s:1:"1";s:7:"invoice";s:1:"5";s:4:"name";s:25:"Zuschlag für Zahlungsart";s:6:"status";s:1:"0";s:7:"shipped";s:1:"0";s:12:"shippedgroup";s:1:"0";s:11:"releasedate";s:10:"0000-00-00";s:5:"modus";s:1:"4";s:10:"esdarticle";s:1:"0";s:5:"taxID";s:1:"1";s:3:"tax";s:5:"19.00";s:8:"tax_rate";s:2:"19";s:3:"esd";s:1:"0";s:10:"attributes";a:7:{s:10:"attribute1";s:0:"";s:10:"attribute2";s:0:"";s:10:"attribute3";s:0:"";s:10:"attribute4";s:0:"";s:10:"attribute5";s:0:"";s:10:"attribute6";s:0:"";s:26:"pickware_canceled_quantity";s:1:"0";}}}s:5:"sUser";a:86:{s:15:"billing_company";s:3:"B2B";s:18:"billing_department";s:7:"Einkauf";s:18:"billing_salutation";s:2:"mr";s:14:"customernumber";s:5:"20003";s:17:"billing_firstname";s:8:"Händler";s:16:"billing_lastname";s:18:"Kundengruppe-Netto";s:14:"billing_street";s:11:"Musterweg 1";s:32:"billing_additional_address_line1";N;s:32:"billing_additional_address_line2";N;s:15:"billing_zipcode";s:5:"00000";s:12:"billing_city";s:11:"Musterstadt";s:5:"phone";s:13:"012345 / 6789";s:13:"billing_phone";s:13:"012345 / 6789";s:17:"billing_countryID";s:1:"2";s:15:"billing_stateID";s:1:"3";s:15:"billing_country";s:11:"Deutschland";s:18:"billing_countryiso";s:2:"DE";s:19:"billing_countryarea";s:11:"deutschland";s:17:"billing_countryen";s:7:"GERMANY";s:5:"ustid";s:0:"";s:13:"billing_text1";N;s:13:"billing_text2";N;s:13:"billing_text3";N;s:13:"billing_text4";N;s:13:"billing_text5";N;s:13:"billing_text6";N;s:7:"orderID";s:2:"15";s:16:"shipping_company";s:3:"B2B";s:19:"shipping_department";s:7:"Einkauf";s:19:"shipping_salutation";s:2:"mr";s:18:"shipping_firstname";s:8:"Händler";s:17:"shipping_lastname";s:18:"Kundengruppe-Netto";s:15:"shipping_street";s:11:"Musterweg 1";s:33:"shipping_additional_address_line1";N;s:33:"shipping_additional_address_line2";N;s:16:"shipping_zipcode";s:5:"00000";s:13:"shipping_city";s:11:"Musterstadt";s:16:"shipping_stateID";s:1:"3";s:18:"shipping_countryID";s:1:"2";s:16:"shipping_country";s:11:"Deutschland";s:19:"shipping_countryiso";s:2:"DE";s:20:"shipping_countryarea";s:11:"deutschland";s:18:"shipping_countryen";s:7:"GERMANY";s:14:"shipping_text1";N;s:14:"shipping_text2";N;s:14:"shipping_text3";N;s:14:"shipping_text4";N;s:14:"shipping_text5";N;s:14:"shipping_text6";N;s:2:"id";s:1:"2";s:8:"password";s:32:"352db51c3ff06159d380d3d9935ec814";s:7:"encoder";s:3:"md5";s:5:"email";s:17:"mustermann@b2b.de";s:6:"active";s:1:"1";s:11:"accountmode";s:1:"0";s:15:"confirmationkey";s:0:"";s:9:"paymentID";s:1:"4";s:19:"doubleOptinRegister";s:1:"0";s:24:"doubleOptinEmailSentDate";N;s:22:"doubleOptinConfirmDate";N;s:10:"firstlogin";s:10:"2012-08-30";s:9:"lastlogin";s:19:"2012-08-30 11:43:17";s:9:"sessionID";s:40:"66e9b10064a19b1fcf6eb9310c0753866c764836";s:10:"newsletter";s:1:"0";s:10:"validation";s:1:"0";s:9:"affiliate";s:1:"0";s:13:"customergroup";s:1:"H";s:13:"paymentpreset";s:1:"4";s:8:"language";s:1:"1";s:9:"subshopID";s:1:"1";s:7:"referer";s:0:"";s:12:"pricegroupID";N;s:15:"internalcomment";s:0:"";s:12:"failedlogins";s:1:"0";s:11:"lockeduntil";N;s:26:"default_billing_address_id";s:1:"2";s:27:"default_shipping_address_id";s:1:"4";s:5:"title";N;s:10:"salutation";s:2:"mr";s:9:"firstname";s:8:"Händler";s:8:"lastname";s:18:"Kundengruppe-Netto";s:8:"birthday";N;s:11:"login_token";N;s:7:"changed";N;s:11:"preisgruppe";s:1:"2";s:11:"billing_net";s:1:"0";}s:9:"sDispatch";a:3:{s:2:"id";s:1:"9";s:4:"name";s:16:"Standard Versand";s:11:"description";s:0:"";}s:16:"viisonConfigShop";a:3:{s:15:"__initializer__";N;s:10:"__cloner__";N;s:17:"__isInitialized__";b:1;}s:11:"orderNumber";s:5:"20001";s:14:"userSalutation";s:2:"mr";s:13:"userFirstName";s:8:"Händler";s:12:"userLastName";s:18:"Kundengruppe-Netto";}',
            ]
        );
    }

    /* Pickware test version methods */

    /**
     * Returns true, if this plugin binary is a test version, which has expired. Otherwise false.
     *
     * @return boolean
     */
    public function pluginTestExpired()
    {
        $status = $this->pluginTestStatus();

        return $status && $status['daysLeft'] < 0;
    }

    /**
     * First checks whether a plugin test validity date is set and, if it is, determines
     * how many days of the testing period are left. Finally the licenses are checked for
     * a valid Shopware PE license. If this license exists, any previously determined
     * testing period is overwritten (with 100) to allow using the test version binary
     * e.g. after upgrading to Shopware PE during the testing period.
     *
     * @return array|null
     */
    public function pluginTestStatus()
    {
        $validUntil = $this->pluginTestValidUntil();
        if (!$validUntil) {
            return null;
        }

        // Check the config for a saved date of a previous test version
        $key = '6LGkFabCs9RkErB9d';
        $installId = $this->get('viison_common.hidden_config_storage')->getConfigValue(self::CONFIG_ELEMENT_NAME_INSTALL_ID);
        if (!empty($installId)) {
            // 'Decrypt' the value and use it as the validity date
            try {
                $validUntil = new DateTime(base64_decode($installId) ^ $key);
            } catch (\Exception $e) { // phpcs:ignore Generic.CodeAnalysis.EmptyStatement
                // Saved, encoded date string is invalid. We just ignore this for now
                // and use the date of this binary instead.
            }
        } else {
            // 'Encrypt' the validity date of this binary and save it in the config
            $value = base64_encode($validUntil->format('Y-m-d') ^ $key);
            $this->get('viison_common.hidden_config_storage')->setConfigValue(
                self::CONFIG_ELEMENT_NAME_INSTALL_ID,
                'text',
                $value
            );
        }

        // Define the plugin status
        $diff = $validUntil->diff(new DateTime());
        $status = [
            'daysLeft' => (($diff->invert === 0) ? -1 : 1) * $diff->days,
        ];

        // Try to find a Shopware license key
        $licenseString = $this->get('db')->fetchOne(
            'SELECT license
            FROM s_core_licenses
            WHERE active = 1
            AND module = \'SwagCommercial\''
        );
        if (!$licenseString) {
            return $status;
        }

        try {
            // Evaluate the license string
            $host = $this->get('models')->getRepository(Shop::class)->getActiveDefault()->getHost();
            $request = new LicenseUnpackRequest($licenseString, $host);
            $licenseData = $this->get('shopware_core.local_license_unpack_service')->evaluateLicense($request);
            if ($licenseData) {
                // Shop has a valid Shopware PE license, hence overwrite the test validity
                $status['daysLeft'] = 100;
            }
        } catch (\Exception $e) { // phpcs:ignore Generic.CodeAnalysis.EmptyStatement
            // Ignore exception
        }

        return $status;
    }

    /**
     * Adds email template for returns
     */
    private function addReturnEmailTemplates()
    {
        $mails = [
            [
                'mailTemplateName' => ReturnShipmentMailingService::RETURN_RECEIVED_NOTICE_EMAIL_TEMPLATE,
                'subject' => 'Retoure zu Bestellung #{$orderNumber} eingegangen',
                'templateFile' => 'return_received_mail.tpl',
            ],
            [
                'mailTemplateName' => ReturnShipmentMailingService::RETURN_COMPLETED_NOTICE_EMAIL_TEMPLATE,
                'subject' => 'Retoure zu Bestellung #{$orderNumber} abgeschlossen',
                'templateFile' => 'return_completed_mail.tpl',
            ],
        ];

        /** @var ModelManager $modelManager */
        $modelManager = $this->get('models');
        foreach ($mails as $mailData) {
            $mail = $modelManager->getRepository(Mail::class)->findOneBy([
                'name' => $mailData['mailTemplateName'],
            ]);

            if (!$mail) {
                // Create a new mail template
                $mail = new Mail();
                $mail->setName($mailData['mailTemplateName']);
                $mail->setFromMail('{config name=mail}');
                $mail->setFromName('{config name=shopName}');
                $mail->setSubject($mailData['subject']);
                $mail->setMailtype(1);
                $modelManager->persist($mail);
            }

            // Add the plain email content
            $content = file_get_contents($this->Path() . 'Views/mails/' . $mailData['templateFile']);
            if ($content !== false) {
                $mail->setContent($content);
            }

            // Save the mail
            $modelManager->flush($mail);
        }
    }

    /**
     * Execute the barcode label migration from ViisonPickwareCommon to ViisonPickwareERP
     */
    private function migrateOrCreateBarcodeLabelPrinting()
    {
        $this->removeForeignKeyConstraintsFromBarcodeLabelTables();
        $this->createOrRenameArticleDetailBarcodeLabelsTable();
        $this->createOrRenameBinLocationBarcodeLabelsTable();
        $this->createOrRenameBarcodeLabelPresetTable();
        $this->createOrRenameBarcodeLabelPresetBlockTable();
        $this->addForeignKeyConstraintsToBarcodeLabelTables();

        $this->createDefaultBarcodeLabelPresets();
    }

    /**
     * Removes the foreign key constraints for the barcode label migration.
     */
    private function removeForeignKeyConstraintsFromBarcodeLabelTables()
    {
        $sqlHelper = new SQLHelper($this->get('db'));
        $sqlHelper->dropForeignKeyConstraintIfExists(
            'pickware_erp_barcode_label_preset_blocks',
            'presetId',
            'pickware_erp_barcode_label_presets',
            'id'
        );
        $sqlHelper->dropForeignKeyConstraintIfExists(
            'pickware_erp_bin_location_barcode_labels',
            'binLocationId',
            'pickware_erp_bin_locations',
            'id'
        );
        $sqlHelper->dropForeignKeyConstraintIfExists(
            'pickware_erp_article_detail_barcode_labels',
            'articleDetailId',
            's_articles_details',
            'id'
        );
    }

    /**
     * Ensures the 'pickware_erp_barcode_label_presets' table exists.
     */
    private function createOrRenameBarcodeLabelPresetTable()
    {
        $sqlHelper = new SQLHelper($this->get('db'));
        if (!$sqlHelper->doesTableExist('pickware_erp_barcode_label_presets')
            && !$sqlHelper->doesTableExist('s_plugin_pickware_barcode_label_preset')
        ) {
            $this->get('db')->query(
                'CREATE TABLE IF NOT EXISTS `pickware_erp_barcode_label_presets` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `type` varchar(255) COLLATE utf8_unicode_ci NOT NULL,
                    `name` varchar(255) COLLATE utf8_unicode_ci NOT NULL,
                    `paperLayoutIdentifier` varchar(255) COLLATE utf8_unicode_ci NOT NULL,
                    `templateIdentifier` varchar(255) COLLATE utf8_unicode_ci NOT NULL,
                    `paperHeight` double DEFAULT NULL,
                    `paperWidth` double DEFAULT NULL,
                    `paddingTop` double NOT NULL,
                    `paddingRight` double NOT NULL,
                    `paddingBottom` double NOT NULL,
                    `paddingLeft` double NOT NULL,
                    `comment` longtext COLLATE utf8_unicode_ci DEFAULT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB AUTO_INCREMENT=8 DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
            );
        } else {
            $sqlHelper->ensureTableIsRenamed(
                's_plugin_pickware_barcode_label_preset',
                'pickware_erp_barcode_label_presets'
            );
        }
    }

    /**
     * Ensures the 'pickware_erp_barcode_label_preset_blocks' table exists.
     */
    private function createOrRenameBarcodeLabelPresetBlockTable()
    {
        $sqlHelper = new SQLHelper($this->get('db'));
        if (!$sqlHelper->doesTableExist('pickware_erp_barcode_label_preset_blocks')
            && !$sqlHelper->doesTableExist('s_plugin_pickware_barcode_label_preset_block')
        ) {
            $this->get('db')->query(
                'CREATE TABLE IF NOT EXISTS `pickware_erp_barcode_label_preset_blocks` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `presetId` int(11) NOT NULL,
                    `name` varchar(255) COLLATE utf8_unicode_ci NOT NULL,
                    `value` longtext COLLATE utf8_unicode_ci NOT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB AUTO_INCREMENT=26 DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
            );
        } else {
            $sqlHelper->ensureTableIsRenamed(
                's_plugin_pickware_barcode_label_preset_block',
                'pickware_erp_barcode_label_preset_blocks'
            );
        }
    }

    /**
     * Ensures the 'pickware_erp_article_detail_barcode_labels' table exists.
     */
    private function createOrRenameArticleDetailBarcodeLabelsTable()
    {
        $sqlHelper = new SQLHelper($this->get('db'));
        if (!$sqlHelper->doesTableExist('pickware_erp_article_detail_barcode_labels')
            && !$sqlHelper->doesTableExist('s_plugin_pickware_article_barcode_labels')
        ) {
            $this->get('db')->query(
                'CREATE TABLE IF NOT EXISTS `pickware_erp_article_detail_barcode_labels` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `articleDetailId` int(11) NOT NULL,
                    `quantity` int(11) NOT NULL,
                    `added` datetime NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `IDX_ARTICLE_BARCODE_LABELS` (`articleDetailId`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
            );
        } else {
            $sqlHelper->ensureTableIsRenamed(
                's_plugin_pickware_article_barcode_labels',
                'pickware_erp_article_detail_barcode_labels'
            );
        }
    }

    /**
     * Ensures the 'pickware_erp_bin_location_barcode_labels' table exists.
     */
    private function createOrRenameBinLocationBarcodeLabelsTable()
    {
        $sqlHelper = new SQLHelper($this->get('db'));
        if (!$sqlHelper->doesTableExist('pickware_erp_bin_location_barcode_labels')
            && !$sqlHelper->doesTableExist('s_plugin_pickware_bin_location_barcode_labels')
        ) {
            $this->get('db')->query(
                'CREATE TABLE IF NOT EXISTS `pickware_erp_bin_location_barcode_labels` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `binLocationId` int(11) NOT NULL,
                    `quantity` int(11) NOT NULL,
                    `added` datetime NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `IDX_7ECAFBC0EC8281FB` (`binLocationId`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;'
            );
        } else {
            $sqlHelper->ensureTableIsRenamed(
                's_plugin_pickware_bin_location_barcode_labels',
                'pickware_erp_bin_location_barcode_labels'
            );
        }
    }

    /**
     * Adds all necessary foreign key constraints for the barcode label tables.
     */
    private function addForeignKeyConstraintsToBarcodeLabelTables()
    {
        $sqlHelper = new SQLHelper($this->get('db'));
        $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
            'pickware_erp_barcode_label_preset_blocks',
            'presetId',
            'pickware_erp_barcode_label_presets',
            'id',
            'CASCADE'
        );
        $sqlHelper->cleanUpAndEnsureCascadingForeignKeyConstraint(
            'pickware_erp_bin_location_barcode_labels',
            'binLocationId',
            'pickware_erp_bin_locations',
            'id'
        );

        $sqlHelper->cleanUpForRestrictingOrCascadingForeignKeyConstraint(
            'pickware_erp_article_detail_barcode_labels',
            'articleDetailId',
            's_articles_details',
            'id'
        );
        $this->get('db')->query(
            'ALTER TABLE pickware_erp_article_detail_barcode_labels
            MODIFY articleDetailId INT(11) unsigned NOT NULL'
        );
        $sqlHelper->ensureForeignKeyConstraint(
            'pickware_erp_article_detail_barcode_labels',
            'articleDetailId',
            's_articles_details',
            'id',
            'CASCADE'
        );
    }

    /**
     * Creates standard presets for the label configurator.
     */
    private function createDefaultBarcodeLabelPresets()
    {
        $templatesUsedInPreviousVersions = [
            '3x6' => [
                'labelTemplate' => 'avery_6171_3x6',
                'labelWidth' => null,
                'labelHeight' => null,
                'labelPadding' => 3,
            ],
            '3x7-ebl' => [
                'labelTemplate' => 'ebl_060x030_3x7',
                'labelWidth' => null,
                'labelHeight' => null,
                'labelPadding' => 3,
            ],
            '4x10-ebl' => [
                'labelTemplate' => 'ebl_048x025_4x10',
                'labelWidth' => null,
                'labelHeight' => null,
                'labelPadding' => 1,
            ],
            '4x12' => [
                'labelTemplate' => 'avery_4736_4x12',
                'labelWidth' => null,
                'labelHeight' => null,
                'labelPadding' => 1,
            ],
            '5x16' => [
                'labelTemplate' => 'avery_4732_5x16',
                'labelWidth' => null,
                'labelHeight' => null,
                'labelPadding' => 1,
            ],
            '1x1_38x23' => [
                'labelTemplate' => 'custom',
                'labelWidth' => 38,
                'labelHeight' => 23,
                'labelPadding' => 1,
            ],
            '1x1_51x19' => [
                'labelTemplate' => 'custom',
                'labelWidth' => 51,
                'labelHeight' => 19,
                'labelPadding' => 2,
            ],
            '1x1_54x25' => [
                'labelTemplate' => 'custom',
                'labelWidth' => 54,
                'labelHeight' => 25,
                'labelPadding' => 2,
            ],
            '1x1_57x32' => [
                'labelTemplate' => 'custom',
                'labelWidth' => 57,
                'labelHeight' => 32,
                'labelPadding' => 2,
            ],
            '1x1_60x30' => [
                'labelTemplate' => 'custom',
                'labelWidth' => 60,
                'labelHeight' => 30,
                'labelPadding' => 2,
            ],
            '1x1_60x40' => [
                'labelTemplate' => 'custom',
                'labelWidth' => 60,
                'labelHeight' => 40,
                'labelPadding' => 2,
            ],
            '1x1_70x38' => [
                'labelTemplate' => 'custom',
                'labelWidth' => 70,
                'labelHeight' => 38,
                'labelPadding' => 2,
            ],
            '1x1_120x60' => [
                'labelTemplate' => 'custom',
                'labelWidth' => 120,
                'labelHeight' => 60,
                'labelPadding' => 3,
            ],
        ];

        $presetsSqlInsert = [];
        $presetBlocksSqlInsert = [];
        $presetTranslations = [];

        $previousBinLocationPreset = $this->get('models')->getRepository(BarcodeLabelPreset::class)->findOneBy([
            'type' => BinLocationBarcodeLabelType::IDENTIFIER,
        ]);
        if (!$previousBinLocationPreset) {
            $presetsSqlInsert = [
                '(1, \'article\', \'Barcode unten\', :labelTemplate, \'three_rows_two_columns_one_column_barcode\', :labelHeight, :labelWidth, :labelPadding, :labelPadding, :labelPadding, :labelPadding)',
                '(2, \'article\', \'Barcode mittig\', :labelTemplate, \'three_rows_two_columns_barcode_one_column\', :labelHeight, :labelWidth, :labelPadding, :labelPadding, :labelPadding, :labelPadding)',
                '(3, \'article\', \'Barcode oben\', :labelTemplate, \'three_rows_barcode_two_columns_one_column\', :labelHeight, :labelWidth, :labelPadding, :labelPadding, :labelPadding, :labelPadding)',
                '(4, \'article\', \'Mit Artikelbild\', :labelTemplate, \'three_rows_one_column_two_columns_barcode\', :labelHeight, :labelWidth, :labelPadding, :labelPadding, :labelPadding, :labelPadding)',
                '(5, \'article\', \'Mit Herstellerlogo\', :labelTemplate, \'three_rows_two_columns_one_column_barcode\', :labelHeight, :labelWidth, :labelPadding, :labelPadding, :labelPadding, :labelPadding)',
                '(6, \'bin_location\', \'Lagerplatz\', :labelTemplate, \'three_rows_one_column_barcode_one_column\', :labelHeight, :labelWidth, :labelPadding, :labelPadding, :labelPadding, :labelPadding)',
                '(7, \'picking_box\', \'Kommissionierkiste\', :labelTemplate, \'two_rows_barcode_one_column\', :labelHeight, :labelWidth, :labelPadding, :labelPadding, :labelPadding, :labelPadding)',
            ];

            $presetBlocksSqlInsert = [
                '(1, \'1\', \'<p>{$articleNumber}</p>\')',
                '(1, \'2\', \'<p style="text-align: right;">{$articlePrice}</p>\')',
                '(1, \'3\', \'<p>{$articleName}</p>\')',
                '(1, \'barcode\', \'{$articleNumberBarcode}\')',
                '(2, \'1\', \'<p>{$articleNumber}</p>\')',
                '(2, \'2\', \'<p style="text-align: right;">{$articlePrice}</p>\')',
                '(2, \'3\', \'<p>{$articleName}</p>\')',
                '(2, \'barcode\', \'{$articleNumberBarcode}\')',
                '(3, \'1\', \'<p>{$articleNumber}</p>\')',
                '(3, \'2\', \'<p style="text-align: right;">{$articlePrice}</p>\')',
                '(3, \'3\', \'<p>{$articleName}</p>\')',
                '(3, \'barcode\', \'{$articleNumberBarcode}\')',
                '(4, \'1\', \'<p>{$articleName}</p>\')',
                '(4, \'2\', \'<p>{$articleImage}</p>\')',
                '(4, \'3\', \'<p style="text-align: right;">{$articlePrice}</p>\')',
                '(4, \'barcode\', \'{$articleNumberBarcode}\')',
                '(5, \'1\', \'<p>{$articleNumber}</p>\')',
                '(5, \'2\', \'<p style="text-align: right;">{$articleManufacturerLogo}</p>\')',
                '(5, \'3\', \'<p>{$articleName}</p>\')',
                '(5, \'barcode\', \'{$articleNumberBarcode}\')',
                '(6, \'1\', \'<p>Lagerplatz</p>\')',
                '(6, \'2\', \'<p>{$binLocationCode}</p>\')',
                '(6, \'barcode\', \'{$barcode}\')',
                '(7, \'1\', \'<p>(Kiste {$pickingBoxNumber})</p>\')',
                '(7, \'barcode\', \'{$barcode}\')',
            ];

            $presetTranslations = [
                1 => [
                    'name' => 'Barcode bottom',
                ],
                2 => [
                    'name' => 'Barcode middle',
                ],
                3 => [
                    'name' => 'Barcode top',
                ],
                4 => [
                    'name' => 'With article image',
                ],
                5 => [
                    'name' => 'With manufacturer logo',
                ],
                6 => [
                    'name' => 'Bin location',
                ],
                7 => [
                    'name' => 'Picking box',
                ],
            ];
        }

        $hiddenConfigStorage = $this->get('viison_common.hidden_config_storage');
        $barcodeLessPresetWasInstalledByPickwareCommon = $hiddenConfigStorage->getConfigValue('viisonPickwareCommonIsBarcodeLessBarcodeLabelPresetInstalled') === true;
        $barcodeLessPresetWasInstalledByPickwareErp = $hiddenConfigStorage->getConfigValue('pickwareErpBarcodeLessBarcodeLabelPresetIsInstalled') === true;
        $installBarcodeLessPreset = !$barcodeLessPresetWasInstalledByPickwareCommon && !$barcodeLessPresetWasInstalledByPickwareErp;
        if ($installBarcodeLessPreset) {
            $presetsSqlInsert[] = '(NULL, \'article\', \'Ohne Barcode\', :labelTemplate, \'two_rows_two_columns_two_columns\', :labelHeight, :labelWidth, :labelPadding, :labelPadding, :labelPadding, :labelPadding)';
        }

        if (count($presetsSqlInsert) === 0) {
            return;
        }

        // Use avery 3x6 layout as default template
        $labelTemplate = 'avery_6171_3x6';
        $labelWidth = null;
        $labelHeight = null;
        $labelPadding = 5;

        // Migrate old default template
        $defaultTemplate = $this->get('models')->getRepository(Element::class)->findOneBy([
            'name' => 'viisonPickwareCommonBarcodeLabelPrintingDefaultTemplate',
        ]);
        if ($defaultTemplate && $defaultTemplate->getValue()) {
            if (isset($templatesUsedInPreviousVersions[$defaultTemplate->getValue()])) {
                $labelConfig = $templatesUsedInPreviousVersions[$defaultTemplate->getValue()];
                $labelTemplate = $labelConfig['labelTemplate'];
                $labelWidth = $labelConfig['labelWidth'];
                $labelHeight = $labelConfig['labelHeight'];
                $labelPadding = $labelConfig['labelPadding'];
            }
        }

        // Create preconfigured barcode label presets
        $this->get('db')->query(
            'INSERT IGNORE INTO `pickware_erp_barcode_label_presets`
                (`id`, `type`, `name`, `paperLayoutIdentifier`, `templateIdentifier`, `paperHeight`, `paperWidth`, `paddingTop`, `paddingRight`, `paddingBottom`, `paddingLeft`)
            VALUES ' . implode(",\n", $presetsSqlInsert),
            [
                'labelHeight' => $labelHeight,
                'labelPadding' => $labelPadding,
                'labelTemplate' => $labelTemplate,
                'labelWidth' => $labelWidth,
            ]
        );

        $barcodelessPresetId = null;
        if ($installBarcodeLessPreset) {
            $barcodelessPresetId = $this->get('db')->lastInsertId();

            $presetBlocksSqlInsert[] = '(:barcodelessPresetId, \'1\', \'<p>{$articleName}</p>\\n<p><span style="font-size: x-small;">{$articleNumber}</span></p>\')';
            $presetBlocksSqlInsert[] = '(:barcodelessPresetId, \'2\', \'<p style="text-align: right;">{$articlePrice}</p>\')';
            $presetBlocksSqlInsert[] = '(:barcodelessPresetId, \'3\', \'\')';
            $presetBlocksSqlInsert[] = '(:barcodelessPresetId, \'4\', \'<p style="text-align: right;">{$articleImage}</p>\');';

            $presetTranslations[$id] = [
                'name' => 'Without barcode',
            ];
        }

        // Add all preset blocks
        $this->get('db')->query(
            'INSERT IGNORE INTO `pickware_erp_barcode_label_preset_blocks`
                (`presetId`, `name`, `value`)
            VALUES
                ' . implode(", \n", $presetBlocksSqlInsert),
            [
                'barcodelessPresetId' => $barcodelessPresetId,
            ]
        );

        $this->installPresetTranslations($presetTranslations);

        $hiddenConfigStorage->removeConfigValue('viisonPickwareCommonBarcodeLabelPrintingDefaultTemplate');
        $hiddenConfigStorage->setConfigValue(
            'pickwareErpBarcodeLessBarcodeLabelPresetIsInstalled',
            'boolean',
            true
        );
    }

    /**
     * Inserts the standard preset translations if they do not exist yet.
     */
    private function installPresetTranslations($presetNameTranslations)
    {
        $translations = $this->get('db')->query(
            'SELECT * FROM s_core_translations WHERE `objecttype` = \'viison_pickware_erp_barcode_label_preset_name\''
        );

        if (count($translations->fetchAll()) != 0) {
            return;
        }

        $this->get('db')->query(
            'INSERT INTO `s_core_translations`
                (`objecttype`, `objectdata`, `objectkey`, `objectlanguage`, `dirty`)
            VALUES
                (\'viison_pickware_erp_barcode_label_preset_name\', \'' . serialize($presetNameTranslations) . '\', 1, \'2\', 1);'
        );
    }

    /**
     * Don't change or remove this method!
     *
     * @return DateTime|null
     */
    private function pluginTestValidUntil()
    {
        return null;
    }
}
