<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\Article;

use Exception;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\BarcodeLabel\ArticleBarcodeLabel;
use Shopware\CustomModels\ViisonPickwareERP\BarcodeLabel\Repository as BarcodeLabelRepository;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Attribute\Article as ArticleAttribute;
use Shopware\Models\Customer\Group as CustomerGroup;
use Shopware\Models\Shop\Shop;
use Shopware\Plugins\ViisonCommon\Components\ImageService;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\AbstractBarcodeLabelType;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeLabelField;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeLabelFieldTranslation;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeRenderer;
use Shopware\Plugins\ViisonPickwareERP\Components\PluginConfig\PluginConfig;
use Shopware_Components_Snippet_Manager;

class ArticleBarcodeLabelType extends AbstractBarcodeLabelType
{
    use BarcodeLabelFieldTranslation;

    /**
     * Barcode label type identifier.
     */
    const IDENTIFIER = 'article';

    /**
     * The snippet namespace containing the translations of all barcode label field descriptions.
     */
    const SNIPPETS_NAMESPACE = 'backend/viison_pickware_erp_barcode_label_presets/barcode_label_fields';

    /**
     * List of attributes which should not be provided as barcode label variables.
     */
    const ARTICLE_DETAIL_ATTRIBUTE_BLACK_LIST = [
        'articleDetailId',
        'articleId',
        'id',
        'pickwareIncomingStock',
        'pickwarePhysicalStockForSale',
        'pickwareStockInitializationTime',
        'pickwareStockInitialized',
        'pickwareStockManagementDisabled',
        'viisonCoupon',
        'viisonCouponDocumentTypeId',
        'viisonCouponId',
        'viisonDropShipperMail',
        'viisonMoss',
        'viisonOtherServiceParagraphOneOrTwo',
        'viisonSetarticleActive',
    ];

    /**
     * @var ModelManager
     */
    private $entityManager;

    /**
     * @var BarcodeRenderer
     */
    private $barcodeRenderer;

    /**
     * @var ImageService
     */
    private $imageService;

    /**
     * @param ModelManager $entityManager
     * @param PluginConfig $pluginConfig
     * @param Shopware_Components_Snippet_Manager $snippetManager
     * @param BarcodeRenderer $barcodeRenderer
     * @param ImageService $imageService
     */
    public function __construct(
        $entityManager,
        PluginConfig $pluginConfig,
        $snippetManager,
        BarcodeRenderer $barcodeRenderer,
        ImageService $imageService
    ) {
        parent::__construct(self::IDENTIFIER, true, $pluginConfig->getBarcodeLabelTemplateDirectories());

        $this->entityManager = $entityManager;
        $this->setNamespace($snippetManager, self::SNIPPETS_NAMESPACE);
        $this->barcodeRenderer = $barcodeRenderer;
        $this->imageService = $imageService;
    }

    /**
     * @inheritDoc
     */
    public function createItemProvider(array $params, array $filter, array $sort)
    {
        /** @var BarcodeLabelRepository $repository */
        $repository = $this->entityManager->getRepository(ArticleBarcodeLabel::class);
        $barcodeLabels = $repository->getArticleBarcodeLabelListQuery(null, null, $sort, $filter)->getResult();

        if (count($barcodeLabels) == 0) {
            throw new \InvalidArgumentException('No barcode labels found.');
        }

        $customerGroupId = intval($params['customerGroup']);
        $customerGroup = null;
        if ($customerGroupId) {
            $customerGroup = $this->entityManager->find(CustomerGroup::class, $customerGroupId);
        }
        if ($customerGroup === null) {
            $customerGroup = $this->entityManager->getRepository(Shop::class)->getDefault()->getCustomerGroup();
        }
        if ($customerGroup === null) {
            throw new \InvalidArgumentException(
                sprintf('Customer group with id %d not found.', $customerGroupId)
            );
        }

        $shopId = intval($params['shop']);
        $shop = null;
        if ($shopId) {
            $shop = $this->entityManager->find(Shop::class, $shopId);
        }
        if ($shop === null) {
            $shop = $this->entityManager->getRepository(Shop::class)->getDefault();
        }

        return new ArticleItemProvider(
            $this->entityManager,
            $this->barcodeRenderer,
            $this->imageService,
            $this,
            $barcodeLabels,
            $shop,
            $customerGroup
        );
    }

    /**
     * @inheritdoc
     *
     * @deprecated Use `createItemProvider()` instead.
     */
    public function createItemProviderForFilter(array $params, array $filter)
    {
        return $this->createItemProvider($params, $filter, []);
    }

    /**
     * @inheritdoc
     */
    public function createItemProviderForPreview(array $params)
    {
        $articleDetail = $this->entityManager->getRepository(ArticleDetail::class)->findOneBy([
            'number' => $params['identifier'],
        ]);
        if ($articleDetail === null) {
            throw new \InvalidArgumentException(sprintf(
                'Article with articleNumber "%s" not found.',
                $params['articleNumber']
            ));
        }

        $barcodeLabels = [new ArticleBarcodeLabel($articleDetail, 1)];
        $shop = $this->entityManager->getRepository(Shop::class)->getActiveDefault();
        $customerGroup = $shop->getCustomerGroup();

        return new ArticleItemProvider(
            $this->entityManager,
            $this->barcodeRenderer,
            $this->imageService,
            $this,
            $barcodeLabels,
            $shop,
            $customerGroup
        );
    }

    /**
     * @inheritdoc
     */
    public function getFields()
    {
        $fields = [
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$today}',
                '{$today}',
                $this->getFieldDescription('field/article/today')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_BARCODE,
                '{$articleEANBarcodeWithNumbers}',
                '<img src="{$cellItem.articleEANBarcodeWithNumbers}" class="is--barcode is--1x1-aspect-ratio-barcode" />',
                $this->getFieldDescription('field/article/article_ean_barcode_with_numbers')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_BARCODE,
                '{$articleEANBarcodeWithoutNumbers}',
                '<img src="{$cellItem.articleEANBarcodeWithoutNumbers}" class="is--barcode is--stretchable-barcode" />',
                $this->getFieldDescription('field/article/article_ean_barcode_without_numbers')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_BARCODE,
                '{$articleNumberBarcode}',
                '<img class="is--barcode is--stretchable-barcode" src="{$cellItem.articleNumberBarcode}" />',
                $this->getFieldDescription('field/article/article_number_barcode')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleName}',
                '{$cellItem.articleName}',
                $this->getFieldDescription('field/article/article_name')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleVariantText}',
                '{$cellItem.articleVariantText}',
                $this->getFieldDescription('field/article/article_variant_text')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleNumber}',
                '{$cellItem.articleNumber}',
                $this->getFieldDescription('field/article/article_number')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleEAN}',
                '{$cellItem.articleEAN}',
                $this->getFieldDescription('field/article/article_ean')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleImage}',
                '<img src="{$cellItem.articleImage}" />',
                $this->getFieldDescription('field/article/article_image')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleMetaTitle}',
                '{$cellItem.articleMetaTitle}',
                $this->getFieldDescription('field/article/article_meta_title')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleDescription}',
                '{$cellItem.articleDescription}',
                $this->getFieldDescription('field/article/article_description')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleKeywords}',
                '{$cellItem.articleKeywords}',
                $this->getFieldDescription('field/article/article_keywords')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleReleaseDate}',
                '{$cellItem.articleReleaseDate}',
                $this->getFieldDescription('field/article/article_release_date')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articlePrice}',
                '{$cellItem.articlePrice}',
                $this->getFieldDescription('field/article/article_price')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articlePseudoPrice}',
                '{$cellItem.articlePseudoPrice}',
                $this->getFieldDescription('field/article/article_pseudo_price')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleBasePrice}',
                '{$cellItem.articleBasePrice}',
                $this->getFieldDescription('field/article/article_base_price')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articlePurchasePrice}',
                '{$cellItem.articlePurchasePrice}',
                $this->getFieldDescription('field/article/article_purchase_price')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleTax}',
                '{$cellItem.articleTax}',
                $this->getFieldDescription('field/article/article_tax')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleMinStock}',
                '{$cellItem.articleMinStock}',
                $this->getFieldDescription('field/article/article_min_stock')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articlePackUnit}',
                '{$cellItem.articlePackUnit}',
                $this->getFieldDescription('field/article/article_pack_unit')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articlePurchaseUnit}',
                '{$cellItem.articlePurchaseUnit|rtrim:\'0\'|rtrim:\'.\'}',
                $this->getFieldDescription('field/article/article_purchase_unit')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleMaxPurchaseSize}',
                '{$cellItem.articleMaxPurchaseSize}',
                $this->getFieldDescription('field/article/article_max_purchase_size')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleMinPurchaseSize}',
                '{$cellItem.articleMinPurchaseSize}',
                $this->getFieldDescription('field/article/article_min_purchase_size')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleUnit}',
                '{$cellItem.articleUnit}',
                $this->getFieldDescription('field/article/article_unit')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleWidth}',
                '{$cellItem.articleWidth|rtrim:\'0\'|rtrim:\'.\'}',
                $this->getFieldDescription('field/article/article_width')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleHeight}',
                '{$cellItem.articleHeight|rtrim:\'0\'|rtrim:\'.\'}',
                $this->getFieldDescription('field/article/article_height')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleLength}',
                '{$cellItem.articleLength|rtrim:\'0\'|rtrim:\'.\'}',
                $this->getFieldDescription('field/article/article_length')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleWeight}',
                '{$cellItem.articleWeight|rtrim:\'0\'|rtrim:\'.\'}',
                $this->getFieldDescription('field/article/article_weight')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleDefaultSupplierName}',
                '{$cellItem.articleDefaultSupplierName}',
                $this->getFieldDescription('field/article/article_supplier_name')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleDefaultSupplierNumber}',
                '{$cellItem.articleDefaultSupplierNumber}',
                $this->getFieldDescription('field/article/article_supplier_number')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleDefaultSupplierArticleNumber}',
                '{$cellItem.articleDefaultSupplierArticleNumber}',
                $this->getFieldDescription('field/article/article_supplier_article_number')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleDefaultSupplierPurchasePrice}',
                '{$cellItem.articleDefaultSupplierPurchasePrice}',
                $this->getFieldDescription('field/article/article_supplier_purchase_price')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleManufacturerName}',
                '{$cellItem.articleManufacturerName}',
                $this->getFieldDescription('field/article/article_manufacturer_name')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleManufacturerLogo}',
                '<img src="{$cellItem.articleManufacturerLogo}" />',
                $this->getFieldDescription('field/article/article_manufacturer_logo')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleManufacturerWebsite}',
                '{$cellItem.articleManufacturerWebsite}',
                $this->getFieldDescription('field/article/article_manufacturer_website')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$shopName}',
                '{$shopName}',
                $this->getFieldDescription('field/article/shop_name')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$shopDomain}',
                '{$shopDomain}',
                $this->getFieldDescription('field/article/shop_domain')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$shopURL}',
                '{$shopURL}',
                $this->getFieldDescription('field/article/shop_url')
            ),
        ];

        $articleAttributesMetaData = $this->entityManager->getClassMetadata(ArticleAttribute::class);
        foreach ($articleAttributesMetaData->getFieldNames() as $articleAttributesFieldName) {
            if (in_array($articleAttributesFieldName, self::ARTICLE_DETAIL_ATTRIBUTE_BLACK_LIST)) {
                continue;
            }

            $fields[] = new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$articleAttributes[\'' . $articleAttributesFieldName . '\']}',
                '{$cellItem.articleAttributes[\'' . $articleAttributesFieldName . '\']}',
                $this->getFieldDescription('field/article/article_attribute', $articleAttributesFieldName)
            );
        }

        return $fields;
    }

    /**
     * @inheritdoc
     */
    public function enqueueForPrinting($identifier, $quantity = 1)
    {
        $articleDetail = $this->entityManager->getRepository(ArticleDetail::class)->findOneBy([
            'number' => $identifier,
        ]);
        if (!$articleDetail) {
            throw new Exception(
                sprintf('Article detail with number "%s" not found.', $identifier)
            );
        }

        $barcodeLabel = $this->entityManager->getRepository(ArticleBarcodeLabel::class)->findOneBy([
            'articleDetail' => $articleDetail,
        ]);
        if (!$barcodeLabel) {
            $barcodeLabel = new ArticleBarcodeLabel($articleDetail, $quantity);
            $this->entityManager->persist($barcodeLabel);
        } else {
            $barcodeLabel->addQuantity($quantity);
        }

        $this->entityManager->flush($barcodeLabel);
    }
}
