<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\Article;

use DateTime;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\BarcodeLabel\ArticleBarcodeLabel;
use Shopware\CustomModels\ViisonPickwareERP\Supplier\ArticleDetailSupplierMapping;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Customer\Group as CustomerGroup;
use Shopware\Models\Shop\Shop;
use Shopware\Plugins\ViisonCommon\Classes\Util\Currency as CurrencyUtil;
use Shopware\Plugins\ViisonCommon\Classes\Util\Localization;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Components\ImageService;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\AbstractBarcodeLabelType;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeLabelGrouping;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeLabelItemProvider;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeRenderer;

class ArticleItemProvider implements BarcodeLabelItemProvider
{
    use BarcodeLabelGrouping;

    /**
     * Default customer group of shopware.
     */
    const DEFAULT_CUSTOMER_GROUP_KEY = 'EK';

    /**
     * @var ModelManager
     */
    private $entityManager;

    /**
     * @var AbstractBarcodeLabelType
     */
    private $type;

    /**
     * @var BarcodeRenderer
     */
    private $barcodeRenderer;

    /**
     * @var ImageService
     */
    private $imageService;

    /**
     * @var ArticleBarcodeLabel[]
     */
    private $labels;

    /**
     * @var Shop
     */
    private $shop;

    /**
     * @var CustomerGroup
     */
    private $customerGroup;

    /**
     * ArticleItemProvider constructor.
     * @param ModelManager $entityManager
     * @param BarcodeRenderer $barcodeRenderer
     * @param ImageService $imageService
     * @param AbstractBarcodeLabelType $type
     * @param ArticleBarcodeLabel[] $labels
     * @param Shop $shop
     * @param CustomerGroup $customerGroup
     */
    public function __construct(
        $entityManager,
        BarcodeRenderer $barcodeRenderer,
        ImageService $imageService,
        AbstractBarcodeLabelType $type,
        array $labels,
        Shop $shop,
        CustomerGroup $customerGroup
    ) {
        $this->entityManager = $entityManager;
        $this->barcodeRenderer = $barcodeRenderer;
        $this->imageService = $imageService;
        $this->type = $type;
        $this->labels = $labels;
        $this->shop = $shop;
        $this->customerGroup = $customerGroup;
    }

    /**
     * @inheritdoc
     */
    public function getContext()
    {
        return [
            'shopName' => $this->shop->getName(),
            'shopDomain' => $this->shop->getHost(),
            'shopURL' => $this->shop->getHost() . $this->shop->getBasePath(),
            'today' => (new DateTime())->format(
                Localization::getDateFormat($this->shop->getLocale()->getLocale())
            ),
        ];
    }

    /**
     * @inheritdoc
     */
    public function getItems()
    {
        $labelsByDetailId = $this->groupBarcodeLabelsByIdentifier($this->labels);

        $articleDetailIds = array_keys($labelsByDetailId);
        $articleDetails = $this->fetchArticleDetailData($articleDetailIds);
        $articleImages = $this->fetchArticleImages($articleDetailIds);
        $additionalTexts = ViisonCommonUtil::getVariantAdditionalTexts($articleDetailIds);
        $articleDefaultSuppliers = $this->fetchArticleSuppliers($articleDetailIds);

        // Iterate the detail ids to keep their order
        $articleDetailsData = [];
        foreach ($articleDetailIds as $articleDetailId) {
            $articleDetail = $articleDetails[$articleDetailId];
            if (!$articleDetail) {
                continue;
            }
            $articleDetailData = $this->getArticleDetailData(
                $articleDetail,
                $additionalTexts,
                $articleImages[$articleDetailId] ? $articleImages[$articleDetailId] : null,
                $articleDefaultSuppliers[$articleDetailId]
            );

            $articleBarcodeLabel = $labelsByDetailId[$articleDetail['id']];
            for ($i = 0; $i < $articleBarcodeLabel->getQuantity(); $i++) {
                $articleDetailsData[] = $articleDetailData;
            }
        }

        return $articleDetailsData;
    }

    /**
     * Fetches the article data tree by their article details ids.
     *
     * @param int[] $articleDetailIds
     * @return array
     */
    private function fetchArticleDetailData(array $articleDetailIds)
    {
        $builder = $this->entityManager->createQueryBuilder();
        $builder
            ->select(
                'articleDetail',
                'article',
                'supplier',
                'image',
                'price',
                'articleAttribute',
                'imageMedia',
                'tax',
                'options',
                'unit'
            )
            ->from(ArticleDetail::class, 'articleDetail', 'articleDetail.id')
            ->leftJoin('articleDetail.article', 'article')
            ->leftJoin('article.supplier', 'supplier')
            ->leftJoin('article.images', 'image', 'WITH', 'image.main = 1')
            ->leftJoin('articleDetail.prices', 'price')
            ->leftJoin('articleDetail.attribute', 'articleAttribute')
            ->leftJoin('image.media', 'imageMedia')
            ->leftJoin('article.tax', 'tax')
            ->leftJoin('article.configuratorSet', 'options')
            ->leftJoin('articleDetail.unit', 'unit')
            ->where('articleDetail.id IN (:articleDetailIds)')
            ->setParameter(
                'articleDetailIds',
                $articleDetailIds,
                \Doctrine\DBAL\Connection::PARAM_INT_ARRAY
            );

        return $builder->getQuery()->getArrayResult();
    }

    /**
     * Fetches the images for the articles.
     *
     * @param int[] $articleDetailIds
     * @return string
     */
    private function fetchArticleImages(array $articleDetailIds)
    {
        $images = $this->imageService->getVariantImages($articleDetailIds);

        return array_map(function ($articleImages) {
            if (count($articleImages) < 1) {
                return null;
            }

            return $this->imageService->findImageUrlForSize($articleImages[0], 250, 250);
        }, $images);
    }

    /**
     * Fetches the article suppliers to the given article detail ids.
     *
     * @param int[] $articleDetailIds
     * @return array
     */
    private function fetchArticleSuppliers(array $articleDetailIds)
    {
        $builder = $this->entityManager->createQueryBuilder();
        $builder
            ->select(
                'supplierMapping',
                'supplier'
            )
            ->from(ArticleDetailSupplierMapping::class, 'supplierMapping')
            ->leftJoin('supplierMapping.supplier', 'supplier')
            ->where('supplierMapping.articleDetailId IN (:articleDetailIds)')
            ->andWhere('supplierMapping.defaultSupplier = true')
            ->setParameter(
                'articleDetailIds',
                $articleDetailIds,
                \Doctrine\DBAL\Connection::PARAM_INT_ARRAY
            );

        $supplierMappings = $builder->getQuery()->getArrayResult();
        $supplier = [];
        foreach ($supplierMappings as $mapping) {
            $mapping['supplier']['supplierArticleNumber'] = $mapping['supplierArticleNumber'];
            $mapping['supplier']['purchasePrice'] = $mapping['purchasePrice'];
            $supplier[$mapping['articleDetailId']] = $mapping['supplier'];
        }

        return $supplier;
    }

    /**
     * Collects and returns for a given article all article specific informations,
     * which are accessible inside barcode label templates.
     *
     * @param array $articleDetail
     * @param array $additionalTexts
     * @param string $articleImageUrl
     * @param array $supplier
     * @return array
     */
    private function getArticleDetailData(
        array $articleDetail,
        array $additionalTexts,
        $articleImageUrl,
        $supplier
    ) {
        $article = $articleDetail['article'];
        $manufacturer = $article['supplier'];
        $tax = $articleDetail['article']['tax'];
        $prices = $this->getArticleDetailPrices($articleDetail);
        $supplier = $supplier ?: [];
        $eanNumber = $this->getFirstEanFromString($articleDetail['ean']);

        $imageDataUri = '';
        if ($articleImageUrl) {
            $imageDataUri = $this->imageService->getImageAsDataUri($articleImageUrl);
        } elseif ($article['images']) {
            $imageDataUri = $this->imageService->getImageAsDataUri($article['images'][0]['media']['path']);
        }

        $dateFormat = Localization::getDateFormat($this->shop->getLocale()->getLocale());

        return [
            'articleAttributes' => $articleDetail['attribute'],
            'articleBasePrice' => $prices['basePrice'],
            'articleDefaultSupplierArticleNumber' => $supplier['supplierArticleNumber'],
            'articleDefaultSupplierName' => $supplier['name'],
            'articleDefaultSupplierNumber' => $articleDetail['supplierNumber'],
            'articleDefaultSupplierPurchasePrice' => $supplier['purchasePrice'],
            'articleDescription' => $article['description'],
            'articleDetailId' => $articleDetail['id'],
            'articleEAN' => $articleDetail['ean'],
            'articleEANBarcodeType' => $eanNumber ? $this->barcodeRenderer->getBarcodeTypeByEanLength($eanNumber) : '',
            'articleEANBarcodeWithNumbers' => $eanNumber ? $this->barcodeRenderer->createEanBarcode($eanNumber) : '',
            'articleEANBarcodeWithoutNumbers' => $eanNumber ? $this->barcodeRenderer->createEanBarcode($eanNumber, false) : '',
            'articleHeight' => $articleDetail['height'],
            'articleId' => $article['id'],
            'articleImage' => $imageDataUri,
            'articleKeywords' => $article['keywords'],
            'articleLength' => $articleDetail['len'],
            'articleManufacturerDescription' => $manufacturer['description'],
            'articleManufacturerLogo' => $this->imageService->getImageAsDataUri($manufacturer['image']),
            'articleManufacturerName' => $manufacturer['name'],
            'articleManufacturerWebsite' => $manufacturer['link'],
            'articleMaxPurchaseSize' => $articleDetail['maxPurchase'],
            'articleMetaTitle' => $article['metaTitle'],
            'articleMinPurchaseSize' => $articleDetail['minPurchase'],
            'articleMinStock' => $articleDetail['stockMin'],
            'articleName' => $article['name'],
            'articleNumber' => $articleDetail['number'],
            'articleNumberBarcode' => $this->barcodeRenderer->createCode128Barcode($articleDetail['number']),
            'articlePackUnit' => $articleDetail['packUnit'],
            'articlePrice' => $prices['price'],
            'articlePseudoPrice' => $prices['pseudoPrice'],
            'articlePurchasePrice' => $articleDetail['purchasePrice'],
            'articlePurchaseUnit' => $articleDetail['purchaseUnit'],
            'articleReleaseDate' => $articleDetail['releaseDate'] ? $articleDetail['releaseDate']->format($dateFormat) : '',
            'articleTax' => $tax['name'],
            'articleUnit' => $articleDetail['unit'] ? $articleDetail['unit']['name'] : '',
            'articleVariantText' => $additionalTexts[$articleDetail['id']],
            'articleWeight' => $articleDetail['weight'],
            'articleWidth' => $articleDetail['width'],
        ];
    }

    /**
     * Returns the sales price and the base price of a given article detail.
     *
     * @param array $articleDetail
     * @return array
     * @throws \Exception
     */
    private function getArticleDetailPrices(array $articleDetail)
    {
        $baseUnit = $articleDetail['unit']['unit'];
        $currency = $this->shop->getCurrency();
        $pricesPerCustomerGroup = $articleDetail['prices'];
        $purchaseUnit = $articleDetail['purchaseUnit'];
        $referenceUnit = $articleDetail['referenceUnit'];
        $tax = floatval($articleDetail['article']['tax']['tax']);

        $priceData = $this->findCustomerGroupPrice($pricesPerCustomerGroup, $this->customerGroup->getKey());
        if ($priceData === null) {
            // Use price of default customer group as fallback
            $priceData = $this->findCustomerGroupPrice($pricesPerCustomerGroup, self::DEFAULT_CUSTOMER_GROUP_KEY);
        }
        if ($priceData === null) {
            throw new \Exception(sprintf('Sales price for item "%s" not found.', $articleDetail['number']));
        }

        $grossPrice = $this->getGrossPrice($priceData['price'], $tax) * $currency->getFactor();
        $grossPseudoPrice = $priceData['pseudoPrice'] ? $this->getGrossPrice($priceData['pseudoPrice'], $tax) * $currency->getFactor() : null;

        $basePrice = null;
        if ($purchaseUnit && $referenceUnit && $baseUnit) {
            $basePrice = round($grossPrice / $purchaseUnit * $referenceUnit, 2) * $currency->getFactor();
        }

        $basePriceString = '-';
        if ($basePrice) {
            $referenceUnitString = ($referenceUnit !== 1) ? floatval($referenceUnit) : '';
            $basePriceString = CurrencyUtil::getFormattedPriceStringByCurrency($basePrice, $currency) . '/' . $referenceUnitString . $baseUnit;
        }

        return [
            'price' => CurrencyUtil::getFormattedPriceStringByCurrency($grossPrice, $currency),
            'basePrice' => $basePriceString,
            'pseudoPrice' => $grossPseudoPrice ? CurrencyUtil::getFormattedPriceStringByCurrency($grossPseudoPrice, $currency) : '',
        ];
    }

    /**
     * Calculates the gross prices from a tax rate and a net price.
     *
     * @param float $netPrice
     * @param float $tax
     * @return float
     */
    private function getGrossPrice($netPrice, $tax)
    {
        return round($netPrice * (100 + $tax) / 100, 2);
    }

    /**
     * Searches inside all prices of an article detail and returns the price for a given customer group.
     * If the requested price does not exist, null is returned.
     *
     * @param $articleDetailPricesPerCustomerGroup
     * @param $customerGroupKey
     * @return array|null
     */
    private function findCustomerGroupPrice($articleDetailPricesPerCustomerGroup, $customerGroupKey)
    {
        foreach ($articleDetailPricesPerCustomerGroup as $priceData) {
            if ($customerGroupKey === $priceData['customerGroupKey']
                && $priceData['from'] === 1
            ) {
                return $priceData;
            }
        }

        return null;
    }

    /**
     * Extracts the first valid ean number.
     *
     * @param string $value
     * @return string
     */
    private function getFirstEanFromString($value)
    {
        $eanNumbers = [];
        $numberOfMatches = preg_match_all('/\\d+/', $value, $eanNumbers);

        if ($numberOfMatches === false || $numberOfMatches < 1) {
            return '';
        }

        foreach ($eanNumbers[0] as $ean) {
            $eanLength = mb_strlen($ean);

            if ($eanLength === 8 || $eanLength === 12 || $eanLength === 13) {
                return $ean;
            }
        }

        return '';
    }

    /**
     * @inheritdoc
     */
    public function getType()
    {
        return $this->type;
    }
}
