<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel;

use Doctrine\Common\Collections\ArrayCollection;
use Exception;
use Enlight_Event_EventManager;
use Shopware\CustomModels\ViisonPickwareERP\BarcodeLabel\BarcodeLabelPreset;

class BarcodeLabelFacadeService implements BarcodeLabelFacade
{
    /**
     * Event to be fired to collect all installed barcode label types.
     */
    const COLLECT_SUPPORTED_BARCODE_LABEL_TYPES = 'Shopware_Plugins_ViisonPickwareERP_CollectSupportedBarcodeLabelTypes';

    /**
     * @var Enlight_Event_EventManager;
     */
    private $eventManager;

    /**
     * @var BarcodeLabelTemplates
     */
    private $barcodeLabelTemplates;

    /**
     * @var BarcodeLabelPresets
     */
    private $barcodeLabelPresets;

    /**
     * @var BarcodeLabelRenderer
     */
    private $barcodeLabelRenderer;

    /**
     * @var AbstractBarcodeLabelType[]
     */
    private $supportedBarcodeLabelTypes;

    /**
     * @param Enlight_Event_EventManager $eventManager
     * @param BarcodeLabelTemplates $barcodeLabelTemplates
     * @param BarcodeLabelPresets $barcodeLabelPresets
     * @param BarcodeLabelRenderer $barcodeLabelRenderer
     */
    public function __construct(
        $eventManager,
        BarcodeLabelTemplates $barcodeLabelTemplates,
        BarcodeLabelPresets $barcodeLabelPresets,
        BarcodeLabelRenderer $barcodeLabelRenderer
    ) {
        $this->eventManager = $eventManager;
        $this->barcodeLabelTemplates = $barcodeLabelTemplates;
        $this->barcodeLabelPresets = $barcodeLabelPresets;
        $this->barcodeLabelRenderer = $barcodeLabelRenderer;

        $this->loadSupportedBarcodeLabelTypes();
    }

    /**
     * @inheritdoc
     */
    public function getLabelTemplates($barcodeLabelTypeIdentifier)
    {
        return $this->barcodeLabelTemplates->getTemplates(
            $this->getBarcodeLabelTypeByName($barcodeLabelTypeIdentifier)
        );
    }

    /**
     * @inheritdoc
     */
    public function getPaperLayouts()
    {
        return BarcodeLabelGridPaperLayout::getLabelPresetGridPaperLayouts();
    }

    /**
     * Determines all barcode label types that are installed.
     */
    private function loadSupportedBarcodeLabelTypes()
    {
        $collectedBarcodeLabelTypes = $this->eventManager->collect(self::COLLECT_SUPPORTED_BARCODE_LABEL_TYPES, new ArrayCollection());

        $this->supportedBarcodeLabelTypes = [];
        foreach ($collectedBarcodeLabelTypes as $supportedType) {
            $this->supportedBarcodeLabelTypes[$supportedType->getType()] = $supportedType;
        }
    }

    /**
     * @inheritdoc
     */
    public function getBarcodeLabelTypeByName($typeIdentifier)
    {
        if (isset($this->supportedBarcodeLabelTypes[$typeIdentifier])) {
            return $this->supportedBarcodeLabelTypes[$typeIdentifier];
        }

        return null;
    }

    /**
     * @inheritdoc
     */
    public function renderBarcodeLabels(
        BarcodeLabelItemProvider $itemProvider,
        BarcodeLabelPreset $preset,
        $offsetX = null,
        $offsetY = null
    ) {
        return $this->barcodeLabelRenderer->render(
            $itemProvider,
            $preset,
            $offsetX,
            $offsetY
        );
    }

    /**
     * @inheritdoc
     */
    public function updatePresetWithValues(BarcodeLabelPreset $preset, array $values)
    {
        $preset->updateWithValues($values);

        if (!$this->getBarcodeLabelTypeByName($preset->getType())) {
            throw new Exception('Unknown barcode label type!');
        }

        if ($this->getBarcodeLabelTypeByName($preset->getType())->areBlocksEditable() === true) {
            $this->barcodeLabelPresets->updatePresetBlocksWithValues(
                $this->getBarcodeLabelTypeByName($preset->getType()),
                $preset,
                $values
            );
        }
    }
}
