<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel;

use InvalidArgumentException;
use Shopware\CustomModels\ViisonPickwareERP\BarcodeLabel\BarcodeLabelPreset;
use Shopware\Plugins\ViisonCommon\Classes\Document\GridPaperLayout;

class BarcodeLabelGridPaperLayout extends GridPaperLayout
{
    /**
     * @param BarcodeLabelPreset $preset
     * @return GridPaperLayout
     */
    public static function createLabelGridPaperLayoutForPreset(BarcodeLabelPreset $preset)
    {
        return self::createLabelGridPaperLayout(
            $preset->getPaperLayoutIdentifier(),
            $preset->getPaperWidth(),
            $preset->getPaperHeight()
        );
    }

    /**
     * Creates and returns the paper layout for a $layoutIdentifier, $paperWidth and $paperHeight. To easily model
     * single label paper layouts, a custom grid paper layout that consists of only one cell is created, if the barcode
     * label preset refers to a non-standard grid paper layout.
     *
     * @param string $layoutIdentifier
     * @param float|null $paperWidth
     * @param float|null $paperHeight
     * @return GridPaperLayout
     * @throws InvalidArgumentException if a custom layout should be created, but $paperWidth or $paperHeight are zero.
     */
    public static function createLabelGridPaperLayout($layoutIdentifier, $paperWidth, $paperHeight)
    {
        try {
            return parent::createDefaultGridPaperLayout($layoutIdentifier);
        } catch (InvalidArgumentException $exception) { // phpcs:ignore Generic.CodeAnalysis.EmptyStatement
        }

        if ($paperWidth === 0 || $paperHeight === 0) {
            throw new InvalidArgumentException(sprintf(
                'The identifier "%s" is not a valid paper layout identifier.',
                $layoutIdentifier
            ));
        }

        return static::createCustomGridPaperLayout($paperWidth, $paperHeight);
    }

    /**
     * @param float $width
     * @param float $height
     * @return BarcodeLabelGridPaperLayout
     */
    private static function createCustomGridPaperLayout($width, $height)
    {
        $layout = new self();
        $layout->identifier = 'custom';
        $layout->cellWidthInMillimeters = $width;
        $layout->cellHeightInMillimeters = $height;
        $layout->pageWidthInMillimeters = $width;
        $layout->pageHeightInMillimeters = $height;
        $layout->columnAmount = 1;
        $layout->rowAmount = 1;
        $layout->horizontalCellSpacingInMillimeters = 0;
        $layout->verticalCellSpacingInMillimeters = 0;
        $layout->pageMarginTopInMillimeters = 0;
        $layout->pageMarginLeftInMillimeters = 0;
        $layout->snippetNamespace = 'backend/viison_pickware_erp_barcode_label_presets/main';
        $layout->snippetName = 'paper_layout/custom/name';

        return $layout;
    }

    /**
     * Returns all grid paper layouts, which are available for barcode label printing.
     *
     * @return GridPaperLayout[]
     */
    public static function getLabelPresetGridPaperLayouts()
    {
        $gridLayouts = parent::getDefaultGridPaperLayouts();
        $gridLayouts[] = static::createCustomGridPaperLayout(0, 0);

        return $gridLayouts;
    }
}
