<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel;

use Shopware\CustomModels\ViisonPickwareERP\BarcodeLabel\BarcodeLabelPreset;
use Shopware\Plugins\ViisonCommon\Classes\Document\GridPaperLayout;
use Shopware\Plugins\ViisonCommon\Classes\Document\RenderedDocument\RenderedDocument;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\FileSystemExceptions\DirectoryNotWritableException;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util;
use Shopware\Plugins\ViisonCommon\Components\DocumentRenderingContextFactoryService;
use Shopware\Plugins\ViisonPickwareERP\Components\PluginConfig\PluginConfig;

class BarcodeLabelRendererService implements BarcodeLabelRenderer
{

    /**
     * Amount of barcodes to be printed (at least) within one batch step.
     */
    const NUMBER_OF_LABELS_PER_BATCH_STEP = 50;

    /**
     * @var DocumentRenderingContextFactoryService
     */
    private $documentRenderingContextFactory;

    /**
     * @var BarcodeLabelTemplatesService
     */
    private $barcodeLabelTemplates;

    /**
     * @var PluginConfig
     */
    private $pluginConfig;

    /**
     * BarcodeLabelRenderer constructor.
     *
     * @param BarcodeLabelTemplates $barcodeLabelTemplates
     * @param DocumentRenderingContextFactoryService $documentRenderingContextFactoryService
     * @param PluginConfig $pluginConfig
     */
    public function __construct(
        BarcodeLabelTemplates $barcodeLabelTemplates,
        DocumentRenderingContextFactoryService $documentRenderingContextFactoryService,
        PluginConfig $pluginConfig
    ) {
        $this->barcodeLabelTemplates = $barcodeLabelTemplates;
        $this->documentRenderingContextFactory = $documentRenderingContextFactoryService;
        $this->pluginConfig = $pluginConfig;
    }

    /**
     * @inheritdoc
     */
    public function render(
        BarcodeLabelItemProvider $itemProvider,
        BarcodeLabelPreset $preset,
        $offsetX = null,
        $offsetY = null
    ) {
        $paperLayout = BarcodeLabelGridPaperLayout::createLabelGridPaperLayoutForPreset($preset);
        $batchAmount = $this->getNumberOfLabelsPerBatchStep($paperLayout);

        $itemsData = $itemProvider->getItems();
        $additionalData = $itemProvider->getContext();
        $totalLabelAmount = count($itemsData);
        $numberOfPrintedLabels = 0;

        $pdfDocumentContents = [];
        while ($totalLabelAmount > $numberOfPrintedLabels) {
            $offsetInLabelAmount = $offsetY * $paperLayout->columnAmount + $offsetX;
            $labelAmountInThisBatch = $batchAmount - $offsetInLabelAmount;
            $labelBatchData = array_slice($itemsData, $numberOfPrintedLabels, $labelAmountInThisBatch);

            $document = $this->renderBarcodeLabels($itemProvider->getType(), $preset, $labelBatchData, $additionalData, $offsetX, $offsetY);
            $pdfDocumentContents[] = $document->getPdf();

            $offsetY = 0;
            $offsetX = 0;
            $numberOfPrintedLabels += $labelAmountInThisBatch;
        }

        return new RenderedDocument('', Util::mergePdfDocumentContents(
            $pdfDocumentContents,
            $paperLayout->pageWidthInMillimeters < $paperLayout->pageHeightInMillimeters ? 'P' : 'L'
        ));
    }

    /**
     * Calculates the correct batch size.
     *
     * @param GridPaperLayout $paperLayout
     * @return int
     */
    private function getNumberOfLabelsPerBatchStep(GridPaperLayout $paperLayout)
    {
        $labelsPerPage = $paperLayout->rowAmount * $paperLayout->columnAmount;
        $minPageAmountPerBatch = (int) (self::NUMBER_OF_LABELS_PER_BATCH_STEP / $labelsPerPage);

        return ($minPageAmountPerBatch + 1) * $labelsPerPage;
    }

    /**
     * Renders a single batch of barcode labels to pdf.
     *
     * @param AbstractBarcodeLabelType $barcodeLabelType
     * @param BarcodeLabelPreset $preset
     * @param array $labelData
     * @param array $additionalData
     * @param int $offsetX
     * @param int $offsetY
     * @return RenderedDocument
     */
    private function renderBarcodeLabels(
        AbstractBarcodeLabelType $barcodeLabelType,
        BarcodeLabelPreset $preset,
        array $labelData,
        array $additionalData,
        $offsetX,
        $offsetY
    ) {
        $template = $this->barcodeLabelTemplates->getTemplate($barcodeLabelType, $preset->getTemplateIdentifier());
        $renderingContext = $this->documentRenderingContextFactory->createGridLayoutRenderingContext(
            $this->getTemplateString($barcodeLabelType->getFields(), $preset, $template),
            BarcodeLabelGridPaperLayout::createLabelGridPaperLayoutForPreset($preset),
            DocumentRenderingContextFactoryService::RENDERER_DOMPDF
        );

        $renderingContext->addTemplateDir($this->pluginConfig->getDefaultBarcodeLabelTemplateDirectory());
        $renderingContext->addTemplateDir(dirname($template->file));

        $renderingContext->setItems($labelData);
        $renderingContext->setTemplateVars([
            'paddingInMillimeter' => [
                'top' => $preset->getPaddingTop(),
                'right' => $preset->getPaddingRight(),
                'bottom' => $preset->getPaddingBottom(),
                'left' => $preset->getPaddingLeft(),
            ],
        ]);
        $renderingContext->assignTemplateVar($additionalData);
        $renderingContext->setOffsetX($offsetX);
        $renderingContext->setOffsetY($offsetY);

        return $renderingContext->renderDocument();
    }

    /**
     * Creates and returns the smarty template string to be rendered, which handels
     * the injection of the preset blocks content by suitable template extensions.
     * Additionally it escapes all smarty tags which are no template variables.
     *
     * @param BarcodeLabelField[] $templateFields
     * @param BarcodeLabelPreset $preset
     * @param BarcodeLabelTemplate $labelTemplate
     * @return string
     */
    private function getTemplateString(array $templateFields, BarcodeLabelPreset $preset, BarcodeLabelTemplate $labelTemplate)
    {
        $presetBlocks = $preset->getBlocks();
        $templateStringElements = [];
        // Every barcode label template extends a base template "barcode_label.tpl", which
        // holds custom CSS definitions. Remark: Due to our usage of the smarty engine
        // (template string) we can not define this extension inside the templates itself.
        $templateStringElements[] = 'extends:' . $this->pluginConfig->getDefaultBarcodeLabelTemplateDirectory() . '/barcode_label.tpl';
        $templateStringElements[] = $labelTemplate->file;

        // Apply fields transformation
        foreach ($presetBlocks as $presetBlock) {
            $blockName = $presetBlock->getName();
            $blockValue = preg_replace('/({|})/', '{literal}$1{/literal}', $presetBlock->getValue());

            foreach ($templateFields as $field) {
                $blockValue = str_replace(
                    preg_replace('/({|})/', '{literal}$1{/literal}', $field->variable),
                    $field->templateVariable,
                    $blockValue
                );
            }

            $blockValue = preg_replace(
                '/{literal}{{\\/literal}(media\\spath=[\'|"][^\'\\"\\{\\}%?:|*<>]+[\'|"]){literal}}{\\/literal}/u',
                '{$1}',
                $blockValue
            );

            $smartyBlockDefinition = '{block name="' . BarcodeLabelTemplate::TEMPLATE_BLOCK_NAME_PREFIX . $blockName . '"}' . $blockValue . '{/block}';
            $templateStringElements[] = 'eval:base64:' . base64_encode($smartyBlockDefinition);
        }

        return implode('|', $templateStringElements);
    }
}
