<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel;

use InvalidArgumentException;

class BarcodeLabelTemplate
{
    /**
     * Prefix used to identify customizable smarty blocks in barcode label
     * templates.
     */
    const TEMPLATE_BLOCK_NAME_PREFIX = 'barcode-label-content-box--';

    /**
     * @var string
     */
    public $identifier;

    /**
     * @var string
     */
    public $name;

    /**
     * @var string
     */
    public $file;

    /**
     * @var string
     */
    public $imageFile;

    /**
     * @var BarcodeLabelTemplateBlock[]
     */
    public $blocks;

    /**
     * @var bool
     */
    public $isDefaultTemplate;

    /**
     * @param string $identifier
     * @param string $name
     * @param string $file
     * @param bool $isDefaultTemplate
     * @param string $imageFile
     * @throws InvalidArgumentException
     */
    public function __construct($identifier, $name, $file, $isDefaultTemplate, $imageFile = '')
    {
        if (!file_exists($file)) {
            throw new InvalidArgumentException(sprintf(
                'Barcode Label template file does not exist ("%s").',
                $file
            ));
        }

        $this->identifier = $identifier;
        $this->name = $name;
        $this->file = $file;
        $this->isDefaultTemplate = $isDefaultTemplate;
        $this->imageFile = $imageFile;
        $this->blocks = self::getBlocks($file);
    }

    /**
     * Extracts and returns all customizable content blocks of a given barcode label file.
     *
     * @param string $templateFile
     * @return BarcodeLabelTemplateBlock[]
     */
    private static function getBlocks($templateFile)
    {
        $regex = '/{block.+name="' . self::TEMPLATE_BLOCK_NAME_PREFIX . '((?:[^"]+))"/';
        preg_match_all($regex, file_get_contents($templateFile), $blockMatches);
        $blocks = array_map(
            function ($blockName) {
                return new BarcodeLabelTemplateBlock($blockName);
            },
            $blockMatches[1]
        );

        return $blocks;
    }

    /**
     * Converts the object to an assoc array.
     *
     * @return array
     */
    public function toArray()
    {
        return [
            'identifier' => $this->identifier,
            'name' => $this->name,
            'file' => $this->file,
            'isDefaultTemplate' => $this->isDefaultTemplate,
            'imageFile' => $this->imageFile,
            'blocks' => array_map(function ($block) {
                return $block->toArray();
            }, $this->blocks),
        ];
    }
}
