<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel;

use FilesystemIterator;
use InvalidArgumentException;

class BarcodeLabelTemplatesService implements BarcodeLabelTemplates
{
    /**
     * Files that may not be listed as template files.
     */
    const BARCODE_LABLE_TEMPLATES_BLACKLIST = [
        '/Views/documents/barcode_label_templates/barcode_label.tpl',
    ];

    /**
     * @var string
     */
    private $fallbackImagePath;

    /**
     * @var BarcodeLabelTemplate[]
     */
    private $templates = [];

    /**
     * @var string
     */
    private $rootPath;

    /**
     * @param string $rootPath
     * @param string $fallbackImagePath
     */
    public function __construct($rootPath, $fallbackImagePath)
    {
        $this->rootPath = $rootPath;
        $this->fallbackImagePath = $fallbackImagePath;
    }

    /**
     * @inheritdoc
     */
    public function getTemplate(AbstractBarcodeLabelType $type, $identifier)
    {
        $templates = $this->getTemplates($type);
        if (!isset($templates[$identifier])) {
            throw new InvalidArgumentException(sprintf(
                'The identifier "%s" is not a valid barcode label template identifier.',
                $identifier
            ));
        }

        return $templates[$identifier];
    }

    /**
     * @inheritdoc
     */
    public function getTemplates(AbstractBarcodeLabelType $type)
    {
        $templates = [];

        foreach ($type->getBarcodeLabelTemplateDirectories() as $path) {
            if (!isset($this->templates[$path])) {
                $this->getTemplatesInFolder($path);
            }

            $templates = array_merge($templates, $this->templates[$path]);
        }

        return $templates;
    }

    /**
     * Returns the list of all available custom barcode label templates.
     *
     * @param string $folderPath
     */
    private function getTemplatesInFolder($folderPath)
    {
        $labelTemplates = [];

        if (is_dir($folderPath)) {
            $templateFilesInPath = new FilesystemIterator($folderPath);

            /** @var \SplFileInfo $templateFile */
            foreach ($templateFilesInPath as $templateFile) {
                if ($templateFile->getExtension() === 'tpl' && !self::isFileBlacklisted($templateFile->getPathname())) {
                    $templateName = $templateFile->getBasename('.tpl');

                    // Determine the preview image path
                    $previewImagePath = $templateFile->getPath() . '/' . $templateName . '.png';
                    if (!file_exists($previewImagePath)) {
                        $previewImagePath = $this->fallbackImagePath;
                    }
                    $resolvedPreviewImagePath = trim(mb_substr($previewImagePath, mb_strlen($this->rootPath)), '/');

                    $labelTemplate = new BarcodeLabelTemplate(
                        $templateName,
                        $templateName,
                        $templateFile->getPathname(),
                        false,
                        $resolvedPreviewImagePath
                    );
                    $labelTemplates[$labelTemplate->identifier] = $labelTemplate;
                }
            }
        }

        $this->templates[$folderPath] = $labelTemplates;
    }

    /**
     * Checks if the file is blacklisted.
     *
     * @param string $fullPath
     * @return bool
     */
    private static function isFileBlacklisted($fullPath)
    {
        foreach (self::BARCODE_LABLE_TEMPLATES_BLACKLIST as $blacklistedFile) {
            if (mb_strpos($fullPath, $blacklistedFile) !== false) {
                return true;
            }
        }

        return false;
    }
}
