<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel;

use DOMDocument;
use DOMElement;
use Zend\Barcode\Barcode;
use Zend_Config;

class BarcodeRendererService implements BarcodeRenderer
{
    /**
     * Barcode type used when generating barcodes from the shopware product number.
     */
    const BARCODE_TYPE_CODE_128 = 'code128';

    /**
     * Barcode types to be used for ean/upc barcodes.
     */
    const BARCODE_TYPE_EAN_13 = 'ean13';
    const BARCODE_TYPE_EAN_8 = 'ean8';
    const BARCODE_TYPE_UPC_A = 'upca';

    /**
     * @inheritdoc
     */
    public function createEanBarcode($value, $drawText = true)
    {
        switch ($this->getBarcodeTypeByEanLength($value)) {
            case self::BARCODE_TYPE_EAN_13:
                return $this->createEan13Barcode($value, $drawText);
            case self::BARCODE_TYPE_UPC_A:
                return $this->createUpcABarcode($value, $drawText);
            case self::BARCODE_TYPE_EAN_8:
                return $this->createEan8Barcode($value, $drawText);
            default:
                return null;
        }
    }

    /**
     * @inheritdoc
     */
    public function getBarcodeTypeByEanLength($ean)
    {
        switch (mb_strlen($ean)) {
            case 13:
                return self::BARCODE_TYPE_EAN_13;
            case 12:
                return self::BARCODE_TYPE_UPC_A;
            case 8:
                return self::BARCODE_TYPE_EAN_8;
            default:
                return null;
        }
    }

    /**
     * Creates an ean 13 barcode data string.
     *
     * @param string $value
     * @param boolean $drawText
     * @return string
     */
    private function createEan13Barcode($value, $drawText)
    {
        // The barHeight of 94 leads to the barcode aspect ratio being almost exactly 1x1.
        // It therefore simplifies the math to calculate the correct positioning of the barcode
        // to align it with the bottom of its container.
        $barcodeImage = $this->createBarcode($this->removeChecksumFromEan($value), self::BARCODE_TYPE_EAN_13, $drawText, 94);
        $barcodeImage = $this->removeRightEan13QuietZone($barcodeImage);
        if (!$drawText) {
            $this->removeLeftEan13QuietZone($barcodeImage);
        }

        return $this->convertBarcodeToDataString($barcodeImage);
    }

    /**
     * Adjusts view box to remove left quiet zone.
     *
     * @param DOMDocument $barcodeImage
     * @return DOMDocument
     */
    private function removeLeftEan13QuietZone(DOMDocument $barcodeImage)
    {
        /** @var DOMElement $svg */
        $svg = $barcodeImage->getElementsByTagName('svg')[0];
        $svg->setAttribute('width', $svg->getAttribute('width') - 40);
        $svg->setAttribute('viewBox', '40 0 ' . $svg->getAttribute('width') . ' 376');
        $svg->setAttribute('preserveAspectRatio', 'none');

        return $barcodeImage;
    }

    /**
     * Removes the right quiet zone.
     *
     * @param DOMDocument $barcodeImage
     * @return DOMDocument
     */
    private function removeRightEan13QuietZone(DOMDocument $barcodeImage)
    {
        /** @var DOMElement $svg */
        $svg = $barcodeImage->getElementsByTagName('svg')[0];
        $svg->setAttribute('width', $svg->getAttribute('width') - 40);

        return $barcodeImage;
    }

    /**
     * Creates an ean 8 barcode data string.
     *
     * @param string $value
     * @param boolean $drawText
     * @return string
     */
    private function createEan8Barcode($value, $drawText)
    {
        // The barHeight of 56 leads to the barcode aspect ratio being almost exactly 1x1.
        // It therefore simplifies the math to calculate the correct positioning of the barcode
        // to align it with the bottom of its container.
        return $this->convertBarcodeToDataString($this->createBarcode($this->removeChecksumFromEan($value), self::BARCODE_TYPE_EAN_8, $drawText, 56));
    }

    /**
     * Creates an UPC-A barcode data string.
     *
     * @param string $value
     * @param boolean $drawText
     * @return string
     */
    private function createUpcABarcode($value, $drawText)
    {
        // The barHeight of 103 leads to the barcode aspect ratio being almost exactly 1x1.
        // It therefore simplifies the math to calculate the correct positioning of the barcode
        // to align it with the bottom of its container.
        return $this->convertBarcodeToDataString($this->createBarcode($this->removeChecksumFromEan($value), self::BARCODE_TYPE_UPC_A, $drawText, 103));
    }

    /**
     * @inheritdoc
     */
    public function createCode128Barcode($value)
    {
        return $this->convertBarcodeToDataString($this->createBarcode($value, self::BARCODE_TYPE_CODE_128));
    }

    /**
     * Removes the checksum character from an ean.
     *
     * @param string $ean ean number
     * @return string
     */
    private function removeChecksumFromEan($ean)
    {
        return mb_substr($ean, 0, -1);
    }

    /**
     * Creates a svg barcode image for a given value und returns it as data uri.
     *
     * @param string $value
     * @param string $barcodeType
     * @param bool $drawText
     * @param int $barHeight
     * @return DOMDocument
     * @throws \Zend\Barcode\Exception\ExceptionInterface
     */
    private function createBarcode($value, $barcodeType, $drawText = false, $barHeight = 25)
    {
        // Draw a new barcode image using the Zend_Barcode class of the custom Library
        $config = new Zend_Config([
            'barcode' => $barcodeType,
            'barcodeParams' => [
                'text' => $value,
                'factor' => 4,
                'barHeight' => $barHeight,
                'withQuietZones' => false,
                'drawText' => $drawText,
            ],
            'renderer' => 'svg',
        ]);

        return Barcode::factory($config)->draw();
    }

    /**
     * Creates a data string from a barcode.
     *
     * @param DOMDocument $barcodeImage
     * @return string
     */
    private function convertBarcodeToDataString(DOMDocument $barcodeImage)
    {
        $xml = $barcodeImage->saveXML();
        // The title tag is not supported by dompdf, it is not necessary, so remove it
        $xml = preg_replace('%<title>.*</title>%', '', $xml);

        // In old zend-barcode versions not including the PR https://github.com/zendframework/zend-barcode/pull/24, the
        // rgb color definitions are invalid. They contain whitespace characters between the rgb values, which is not
        // allowed according to the SVG grammar (https://www.w3.org/TR/SVG/types.html#DataTypeColor).
        $xml = preg_replace('%rgb\\((\\d+), (\\d+), (\\d+)\\)%', 'rgb(\\1,\\2,\\3)', $xml);

        return 'data:image/svg+xml;base64,' . base64_encode($xml);
    }
}
