<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BinLocation;

use Exception;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\BarcodeLabel\BinLocationBarcodeLabel;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\AbstractBarcodeLabelType;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeLabelField;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeLabelFieldTranslation;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeRenderer;
use Shopware\Plugins\ViisonPickwareERP\Components\PluginConfig\PluginConfig;
use Shopware_Components_Snippet_Manager;

class BinLocationBarcodeLabelType extends AbstractBarcodeLabelType
{
    use BarcodeLabelFieldTranslation;

    /**
     * Identifier for bin location barcode labels.
     */
    const IDENTIFIER = 'bin_location';

    /**
     * The snippet namespace containing the translations of all barcode label field descriptions.
     */
    const SNIPPETS_NAMESPACE = 'backend/viison_pickware_erp_barcode_label_presets/barcode_label_fields';

    /**
     * @var ModelManager
     */
    private $entityManager;

    /**
     * @var BarcodeRenderer
     */
    private $barcodeRenderer;

    /**
     * @param ModelManager $entityManager
     * @param PluginConfig $pluginConfig
     * @param Shopware_Components_Snippet_Manager $snippetManager
     * @param BarcodeRenderer $barcodeRenderer
     */
    public function __construct(
        $entityManager,
        PluginConfig $pluginConfig,
        $snippetManager,
        BarcodeRenderer $barcodeRenderer
    ) {
        parent::__construct(self::IDENTIFIER, false, $pluginConfig->getBarcodeLabelTemplateDirectories());

        $this->entityManager = $entityManager;
        $this->barcodeRenderer = $barcodeRenderer;
        $this->setNamespace($snippetManager, self::SNIPPETS_NAMESPACE);
    }

    /**
     * @inheritdoc
     */
    public function createItemProvider(array $params, array $filter, array $sort)
    {
        $repository = $this->entityManager->getRepository(BinLocationBarcodeLabel::class);
        $barcodeLabels = $repository->getBinLocationBarcodeLabelListQuery(null, null, $sort, $filter)->getResult();

        if (count($barcodeLabels) == 0) {
            throw new \InvalidArgumentException('No bin location barcode labels found.');
        }

        return new BinLocationItemProvider($this->barcodeRenderer, $this, $barcodeLabels);
    }

    /**
     * @inheritdoc
     *
     * @deprecated Use `createItemProvider()` instead.
     */
    public function createItemProviderForFilter(array $params, array $filter)
    {
        return $this->createItemProvider($params, $filter, []);
    }

    /**
     * @inheritdoc
     */
    public function createItemProviderForPreview(array $params)
    {
        throw new Exception('Preview is not supported for bin location barcode labels.');
    }

    /**
     * @inheritdoc
     */
    public function getFields()
    {
        return [
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$binLocationCode}',
                '{$cellItem.binLocationCode}',
                $this->getFieldDescription('field/bin_location/bin_location_code')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$warehouseCode}',
                '{$cellItem.warehouseCode}',
                $this->getFieldDescription('field/bin_location/warehouse_code')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_CONTENT,
                '{$warehouseName}',
                '{$cellItem.warehouseName}',
                $this->getFieldDescription('field/bin_location/warehouse_name')
            ),
            new BarcodeLabelField(
                BarcodeLabelField::TYPE_BARCODE,
                '{$barcode}',
                '<img class="is--barcode is--stretchable-barcode" src="{$cellItem.barcode}" />',
                $this->getFieldDescription('field/bin_location/barcode')
            ),
        ];
    }

    /**
     * @inheritdoc
     */
    public function enqueueForPrinting($identifier, $quantity = 1)
    {
        $binLocation = $this->entityManager->find(BinLocation::class, $identifier);
        if (!$binLocation) {
            throw new \Exception(
                sprintf('Bin location with ID "%d" not found.', $identifier)
            );
        }

        $barcodeLabel = $this->entityManager->getRepository(BinLocationBarcodeLabel::class)->findOneBy([
            'binLocation' => $binLocation,
        ]);
        if (!$barcodeLabel) {
            // Create new label entity
            $barcodeLabel = new BinLocationBarcodeLabel($binLocation);
            $this->entityManager->persist($barcodeLabel);
        } else {
            // Add the quantity to the label
            $barcodeLabel->addQuantity($quantity);
        }
        $this->entityManager->flush($barcodeLabel);
    }
}
