<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\Cancellation;

use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentItem;
use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\LocalizableThrowable;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\MessageLocalization;

class CancellationException extends \Exception implements LocalizableThrowable
{
    use MessageLocalization;

    /**
     * @param string $message
     * @param string $snippetName
     * @param array $snippetArguments
     * @param \Exception|null $previousException
     */
    public function __construct($message, $snippetName, array $snippetArguments = [], \Exception $previousException = null)
    {
        $message = vsprintf($message, $snippetArguments);
        parent::__construct($message, 0, $previousException);
        $this->snippetNamespace = 'viison_pickware_erp/components/cancellation/cancellation_exception';
        $this->snippetName = $snippetName;
        $this->snippetArguments = $snippetArguments;
    }

    /**
     * @param OrderDetail $orderDetail
     * @param int $requestedQuantity
     * @param int $maxQuantity
     * @return CancellationException
     */
    public static function cancellationOfUnshippedNotAllowed(OrderDetail $orderDetail, $requestedQuantity, $maxQuantity)
    {
        $message = 'A cancellation of %d unshipped items of the order position "%s" of order "%s" is not allowed because ' .
            'it would lead to an amount of cancelled unshipped items that is either less than 0 or greater than %d ' .
            '(number of currently unshipped items).';

        return new self($message, 'cancellation_of_unshipped_not_allowed', [
            $requestedQuantity,
            $orderDetail->getArticleNumber() ?: $orderDetail->getArticleName(),
            $orderDetail->getOrder()->getNumber(),
            $maxQuantity,
        ]);
    }

    /**
     * @param ReturnShipmentItem $returnShipmentItem
     * @param int $requestedQuantity
     * @param int $maxQuantity
     * @return CancellationException
     */
    public static function cancellationOfReturnedNotAllowed(ReturnShipmentItem $returnShipmentItem, $requestedQuantity, $maxQuantity)
    {
        $message = 'A cancellation of %d returned items of the return shipment position "%s" of return shipment %s is ' .
            'not allowed because it would lead to an amount of cancelled returned items that is less lighter than ' .
            '0 or greater than %d (number of currently returned items in this return shipment item).';

        return new self($message, 'cancellation_of_returned_not_allowed', [
            $requestedQuantity,
            $returnShipmentItem->getOrderDetail()->getArticleNumber() ?: $returnShipmentItem->getOrderDetail()->getArticleName(),
            $returnShipmentItem->getReturnShipment()->getNumber(),
            $maxQuantity,
        ]);
    }
}
