<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\Cancellation;

use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentItem;
use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Models\Order\Order;

interface OrderCanceler
{
    /**
     * @param Order $order
     */
    public function cancelShippingCostsOfOrder(Order $order);

    /**
     * Performs a cancellation for an order detail item that is remaining to ship (neither shipped, nor returned).
     *
     * @param OrderDetail $orderDetail
     * @param int $quantity Positive integer if you want to create a cancellation, negative if you want to revert one
     * @throws CancellationException
     */
    public function cancelRemainingQuantityToShipOfOrderDetail(OrderDetail $orderDetail, $quantity);

    /**
     * Performs a cancellation for a return shipment item.
     *
     * @param ReturnShipmentItem $returnShipmentItem
     * @param int $quantity Positive integer if you want to create a cancellation, negative if you want to revert one
     * @throws CancellationException
     */
    public function cancelReturnedQuantityOfReturnShipmentItem(ReturnShipmentItem $returnShipmentItem, $quantity);

    /**
     * @param OrderDetail $orderDetail
     * @return int
     */
    public function getMaxCancelUnshipped(OrderDetail $orderDetail);

    /**
     * @param ReturnShipmentItem $returnShipmentItem
     * @return int
     */
    public function getMaxCancelReturned(ReturnShipmentItem $returnShipmentItem);

    /**
     * @param OrderDetail $orderDetail
     * @param int $quantity
     * @return bool
     */
    public function isCancelUnshippedAllowed(OrderDetail $orderDetail, $quantity);

    /**
     * @param ReturnShipmentItem $returnShipmentItem
     * @param int $quantity
     * @return bool
     */
    public function isCancelReturnedAllowed(ReturnShipmentItem $returnShipmentItem, $quantity);

    /**
     * @param OrderDetail $orderDetail
     * @param int $quantity
     * @throws CancellationException
     */
    public function assertCancelUnshippedAllowed(OrderDetail $orderDetail, $quantity);

    /**
     * @param ReturnShipmentItem $returnShipmentItem
     * @param int $quantity
     * @throws CancellationException
     */
    public function assertCancelReturnedAllowed(ReturnShipmentItem $returnShipmentItem, $quantity);
}
