<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\Cancellation;

use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentItem;
use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Models\Order\Order;
use Shopware\Plugins\ViisonCommon\Classes\Util\OrderDetailUtil;
use Shopware\Plugins\ViisonPickwareERP\Components\OrderDetailQuantityCalculator\OrderDetailQuantityCalculator;

class OrderCancelerService implements OrderCanceler, \Enlight_Hook
{
    /**
     * @var ModelManager
     */
    private $entityManager;

    /**
     * @var OrderDetailQuantityCalculator
     */
    private $orderDetailQuantityCalculator;

    /**
     * @param ModelManager $entityManager
     * @param OrderDetailQuantityCalculator $orderDetailQuantityCalculator
     */
    public function __construct(ModelManager $entityManager, OrderDetailQuantityCalculator $orderDetailQuantityCalculator)
    {
        $this->entityManager = $entityManager;
        $this->orderDetailQuantityCalculator = $orderDetailQuantityCalculator;
    }

    /**
     * @inheritdoc
     */
    public function cancelShippingCostsOfOrder(Order $order)
    {
        $order->setInvoiceShipping(0.0);
        $order->setInvoiceShippingNet(0.0);

        $order->calculateInvoiceAmount();
        $this->entityManager->flush($order);
    }

    /**
     * @inheritdoc
     */
    public function cancelRemainingQuantityToShipOfOrderDetail(OrderDetail $orderDetail, $quantity)
    {
        if ($quantity === 0) {
            return;
        }

        $this->assertCancelUnshippedAllowed($orderDetail, $quantity);

        $orderDetail->setQuantity($orderDetail->getQuantity() - $quantity);
        $orderDetailAttribute = $orderDetail->getAttribute();
        $orderDetailAttribute->setPickwareCanceledQuantity($orderDetailAttribute->getPickwareCanceledQuantity() + $quantity);
        $orderDetail->getOrder()->calculateInvoiceAmount();
        $this->entityManager->flush([$orderDetail, $orderDetailAttribute, $orderDetail->getOrder()]);
        // I did not find out completely why we need to flush the article detail here, but otherwise the instock of it
        // would not be updated
        $articleDetail = OrderDetailUtil::getArticleDetailForOrderDetail($orderDetail);
        if ($articleDetail) {
            $this->entityManager->flush($articleDetail);
        }
    }

    /**
     * @inheritdoc
     */
    public function cancelReturnedQuantityOfReturnShipmentItem(ReturnShipmentItem $returnShipmentItem, $quantity)
    {
        if ($quantity === 0) {
            return;
        }

        $this->assertCancelReturnedAllowed($returnShipmentItem, $quantity);

        $orderDetail = $returnShipmentItem->getOrderDetail();
        $orderDetail->setQuantity($orderDetail->getQuantity() - $quantity);
        $orderDetailAttribute = $orderDetail->getAttribute();
        $orderDetailAttribute->setPickwareCanceledQuantity($orderDetailAttribute->getPickwareCanceledQuantity() + $quantity);
        $returnShipmentItem->setCancelledQuantity($returnShipmentItem->getCancelledQuantity() + $quantity);
        $orderDetail->getOrder()->calculateInvoiceAmount();
        $this->entityManager->flush([
            $orderDetail,
            $orderDetailAttribute,
            $orderDetail->getOrder(),
            $returnShipmentItem,
        ]);

        // When creating the cancelation for a returnShipment, we do not want to change the inStock of the corresponding
        // ArticleDetail. This has already been done when the ReturnShipment was created. Since Shopware always
        // adjusts the inStock of an ArticleDetail when the quantity of an OrderDetail is changed, we need to revert
        // this automatic change of the inStock.
        $articleDetail = OrderDetailUtil::getArticleDetailForOrderDetail($orderDetail);
        if ($articleDetail) {
            $articleDetail->setInStock($articleDetail->getInStock() - $quantity);
            $this->entityManager->flush($articleDetail);
        }
    }

    /**
     * @inheritdoc
     */
    public function getMaxCancelUnshipped(OrderDetail $orderDetail)
    {
        return $this->orderDetailQuantityCalculator->calculateRemainingQuantityToShip($orderDetail) + $this->orderDetailQuantityCalculator->calculateCancelledRemainingQuantityToShip($orderDetail);
    }

    /**
     * @inheritdoc
     */
    public function getMaxCancelReturned(ReturnShipmentItem $returnShipmentItem)
    {
        return $returnShipmentItem->getReturnedQuantity();
    }

    /**
     * @inheritdoc
     */
    public function isCancelUnshippedAllowed(OrderDetail $orderDetail, $quantity)
    {
        $max = $this->getMaxCancelUnshipped($orderDetail);
        $current = $this->orderDetailQuantityCalculator->calculateCancelledRemainingQuantityToShip($orderDetail);
        $new = $current + $quantity;

        return 0 <= $new && $new <= $max;
    }

    /**
     * @inheritdoc
     */
    public function isCancelReturnedAllowed(ReturnShipmentItem $returnShipmentItem, $quantity)
    {
        $max = $this->getMaxCancelReturned($returnShipmentItem);
        $current = $returnShipmentItem->getCancelledQuantity();
        $new = $current + $quantity;

        return 0 <= $new && $new <= $max;
    }

    /**
     * @inheritdoc
     */
    public function assertCancelUnshippedAllowed(OrderDetail $orderDetail, $quantity)
    {
        if (!$this->isCancelUnshippedAllowed($orderDetail, $quantity)) {
            throw CancellationException::cancellationOfUnshippedNotAllowed(
                $orderDetail,
                $quantity,
                $this->getMaxCancelUnshipped($orderDetail)
            );
        }
    }

    /**
     * @inheritdoc
     */
    public function assertCancelReturnedAllowed(ReturnShipmentItem $returnShipmentItem, $quantity)
    {
        if (!$this->isCancelReturnedAllowed($returnShipmentItem, $quantity)) {
            throw CancellationException::cancellationOfReturnedNotAllowed(
                $returnShipmentItem,
                $quantity,
                $this->getMaxCancelReturned($returnShipmentItem)
            );
        }
    }
}
