<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\DerivedPropertyUpdater;

use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Article\Detail as ArticleDetail;

interface DerivedPropertyUpdater
{
    /**
     * Updates the minimum stocks of all article details in the database to the sum of their minimum stocks in all
     * warehouses, whose stock is available for sale, or `null`, if the article detail's stock is not managed.
     */
    public function recalculateAllArticleDetailMinimumOnlineStocks();

    /**
     * Updates the minimum stocks of the passed `$articleDetail` to the sum of its minimum stocks in all warehouses,
     * whose stock is available for sale, if the `$articleDetail`'s stock is managed. Otherwise it is set to `null`.
     *
     * Note: The implementation of this method must be safe to be called during a Doctrine lifecycle hook. That is,
     * flushing any changes to the database is not allowed!
     *
     * @param ArticleDetail $articleDetail
     */
    public function recalculateMinimumOnlineStockForArticleDetail(ArticleDetail $articleDetail);

    /**
     * Updates the total physical stock for the passed `$articleDetail`.
     *
     * @param ArticleDetail $articleDetail
     */
    public function recalculateTotalPhysicalStockForSaleForArticleDetail(ArticleDetail $articleDetail);

    /**
     * Updates the total physical stock for the passed `$articleDetail` in the passed `$warehouse`
     * (i.e. `WarehouseArticleDetailStockCounts`).
     *
     * @param ArticleDetail $articleDetail
     * @param Warehouse $warehouse
     */
    public function recalculatePhysicalStockForSaleForArticleDetailInWarehouse(
        ArticleDetail $articleDetail,
        Warehouse $warehouse
    );

    /**
     * Updates the total reserved stock for the passed `$articleDetail` that is cached for the passed `$warehouse` (i.e.
     * `WarehouseArticleDetailStockCounts`).
     *
     * Note: The implementation of this method must be safe to be called during a Doctrine lifecycle hook. That is,
     * flushing any changes to the database is not allowed!
     *
     * @param ArticleDetail $articleDetail
     * @param Warehouse $warehouse
     */
    public function recalculateReservedStockForArticleDetailInWarehouse(
        ArticleDetail $articleDetail,
        Warehouse $warehouse
    );

    /**
     * Updates the incoming stock (i.e. open quantity of incomplete supplier orders) for the passed
     * `$articleDetail` in all warehouses.
     *
     * Note: The implementation of this method must be safe to be called during a Doctrine lifecycle hook. That is,
     * flushing any changes to the database is not allowed!
     *
     * @param ArticleDetail $articleDetail
     */
    public function recalculateIncomingStockForArticleDetail(ArticleDetail $articleDetail);
}
