<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\DerivedPropertyUpdater;

use Doctrine\ORM\QueryBuilder;
use Enlight_Hook;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\WarehouseArticleDetailStockCount;
use Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderItem;
use Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderItemStatus;
use Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderStatus;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\ArticleDetailBinLocationMapping;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Attribute\Article as ArticleAttribute;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\ArticleDetailStockInfoProvider;
use Zend_Db_Adapter_Pdo_Abstract;

class DerivedPropertyUpdaterService implements DerivedPropertyUpdater, Enlight_Hook
{
    /**
     * @var ModelManager
     */
    protected $entityManager;

    /**
     * @var Zend_Db_Adapter_Pdo_Abstract
     */
    protected $database;

    /**
     * @var ArticleDetailStockInfoProvider
     */
    protected $articleDetailStockInfoProvider;

    /**
     * @param ModelManager $entityManager
     * @param Zend_Db_Adapter_Pdo_Abstract $database
     * @param ArticleDetailStockInfoProvider $articleDetailStockInfoProvider
     */
    public function __construct(
        $entityManager,
        $database,
        ArticleDetailStockInfoProvider $articleDetailStockInfoProvider
    ) {
        $this->entityManager = $entityManager;
        $this->database = $database;
        $this->articleDetailStockInfoProvider = $articleDetailStockInfoProvider;
    }

    /**
     * @inheritdoc
     */
    public function recalculateAllArticleDetailMinimumOnlineStocks()
    {
        // Set 'stockMin' to the sum of minimum stocks of all available warehouses, for article details whose stock
        // is managed
        $this->database->query(
            'UPDATE `s_articles_details` AS `articleDetails`
            LEFT JOIN (
                SELECT
                    SUM(`warehouseArticleDetailConfig`.`minimumStock`) AS `minimumStock`,
                    `warehouseArticleDetailConfig`.`articleDetailId` AS `articleDetailId`
                FROM `pickware_erp_warehouse_article_detail_configurations` AS `warehouseArticleDetailConfig`
                LEFT JOIN `pickware_erp_warehouses` AS `warehouses`
                    ON `warehouses`.`id` = `warehouseArticleDetailConfig`.`warehouseId`
                WHERE `warehouses`.`stockAvailableForSale` = 1
                GROUP BY `warehouseArticleDetailConfig`.`articleDetailId`
            ) AS `calculatedStock`
                ON `calculatedStock`.`articleDetailId` = `articleDetails`.`id`
            LEFT JOIN `s_articles_attributes` AS `articleAttributes`
                ON `articleAttributes`.`articledetailsID` = `articleDetails`.`id`
            SET `articleDetails`.`stockmin` = `calculatedStock`.`minimumStock`
            WHERE `articleAttributes`.`pickware_stock_management_disabled` = 0'
        );

        // Set 'stockMin' to `null`, for article details whose stock is not managed
        $this->database->query(
            'UPDATE `s_articles_details` AS `articleDetails`
            LEFT JOIN `s_articles_attributes` AS `articleAttributes`
                ON `articleAttributes`.`articledetailsID` = `articleDetails`.`id`
            SET `articleDetails`.`stockmin` = NULL
            WHERE `articleAttributes`.`pickware_stock_management_disabled` = 1'
        );
    }

    /**
     * @inheritdoc
     */
    public function recalculateMinimumOnlineStockForArticleDetail(ArticleDetail $articleDetail)
    {
        if ($this->articleDetailStockInfoProvider->isStockManaged($articleDetail)) {
            $newStockMin = $this->database->fetchOne(
                'SELECT SUM(`warehouseArticleDetailConfig`.`minimumStock`)
                FROM `pickware_erp_warehouse_article_detail_configurations` AS `warehouseArticleDetailConfig`
                LEFT JOIN `pickware_erp_warehouses` AS `warehouses`
                    ON `warehouses`.`id` = `warehouseArticleDetailConfig`.`warehouseId`
                WHERE
                    `warehouseArticleDetailConfig`.`articleDetailId` = :articleDetailId
                    AND `warehouses`.`stockAvailableForSale` = 1',
                [
                    'articleDetailId' => $articleDetail->getId(),
                ]
            );
            $newStockMin = intval($newStockMin);
        } else {
            $newStockMin = null;
        }

        // Since this method might be called during a Doctrine entity lifecycle hook, we cannot flush the changes to the
        // database. However, we still like to save the new value in the database, which is why we do it via an SQL
        // query. That said, we still update the `$articleDetail` to keep the local representation in sync with
        // the database.
        $articleDetail->setStockMin($newStockMin);
        $this->database->query(
            'UPDATE `s_articles_details`
            SET `stockmin` = :newStockMin
            WHERE `id` = :articleDetailId',
            [
                'articleDetailId' => $articleDetail->getId(),
                'newStockMin' => $newStockMin,
            ]
        );
    }

    /**
     * @inheritdoc
     */
    public function recalculateTotalPhysicalStockForSaleForArticleDetail(ArticleDetail $articleDetail)
    {
        if (!$this->articleDetailStockInfoProvider->isStockManaged($articleDetail)) {
            return;
        }

        $attribute = $this->ensureArticleDetailHasAttribute($articleDetail);
        $attribute->setPickwarePhysicalStockForSale(
            $this->articleDetailStockInfoProvider->getTotalPhysicalStockForSale($articleDetail)
        );
        $this->entityManager->flush($attribute);
    }

    /**
     * @inheritdoc
     */
    public function recalculatePhysicalStockForSaleForArticleDetailInWarehouse(
        ArticleDetail $articleDetail,
        Warehouse $warehouse
    ) {
        if (!$this->articleDetailStockInfoProvider->isStockManaged($articleDetail)) {
            return;
        }

        $warehouseStockCounts = $this->ensureArticleDetailHasStockCountsInWarehouse($articleDetail, $warehouse);
        $warehouseStockCounts->setStock(
            $this->articleDetailStockInfoProvider->getStockInWarehouse($articleDetail, $warehouse)
        );
        $this->entityManager->flush($warehouseStockCounts);
    }

    /**
     * @inheritdoc
     */
    public function recalculateReservedStockForArticleDetailInWarehouse(
        ArticleDetail $articleDetail,
        Warehouse $warehouse
    ) {
        if (!$this->articleDetailStockInfoProvider->isStockManaged($articleDetail)) {
            return;
        }

        // Find all bin location mappings
        $builder = $this->entityManager->createQueryBuilder();
        $builder
            ->select('articleDetailBinLocationMapping')
            ->from(ArticleDetailBinLocationMapping::class, 'articleDetailBinLocationMapping')
            ->join('articleDetailBinLocationMapping.binLocation', 'binLocation')
            ->where('articleDetailBinLocationMapping.articleDetail = :articleDetail')
            ->andWhere('binLocation.warehouse = :warehouse')
            ->setParameters([
                'articleDetail' => $articleDetail,
                'warehouse' => $warehouse,
            ]);
        $binLocationMappings = $builder->getQuery()->getResult();

        // Update the cached reservedStock per bin location mapping and calculate the total reserved stock
        $totalReservedStock = 0;
        foreach ($binLocationMappings as $binLocationMapping) {
            // Make sure to have the latest data of the bin location mapping, because for some reason the query might
            // return cached, outdated mappings
            $this->entityManager->refresh($binLocationMapping);

            // Calculate the total reserved stock of the mapping
            $reservedStock = array_reduce(
                $binLocationMapping->getStockReservations()->toArray(),
                function ($carry, $stockReservation) {
                    return $carry + $stockReservation->getQuantity();
                },
                0
            );

            // Since this method might be called during a Doctrine entity lifecycle hook, we cannot flush the changes to
            // the database. However, we still like to save the new value in the database, which is why we do it via an
            // SQL query. That said, we still update the mapping's reserved stockto keep the local representation in
            // sync with the database.
            $this->database->query(
                'UPDATE `pickware_erp_article_detail_bin_location_mappings`
                SET `reservedStock` = :reservedStock
                WHERE `id` = :id',
                [
                    'id' => $binLocationMapping->getId(),
                    'reservedStock' => $reservedStock,
                ]
            );
            $binLocationMapping->setReservedStock($reservedStock);
            $totalReservedStock += $reservedStock;
        }

        // Since this method might be called during a Doctrine entity lifecycle hook, we cannot flush the changes to the
        // database. However, we still like to save the new value in the database, which is why we do it via an SQL
        // query. That said, we still update the `$articleDetail`'s warehouse stock counts to keep the local
        // representation in sync with the database.
        $this->database->query(
            'UPDATE `pickware_erp_warehouse_article_detail_stock_counts`
            SET `reservedStock` = :totalReservedStock
            WHERE
                `articleDetailId` = :articleDetailId
                AND `warehouseId` = :warehouseId',
            [
                'articleDetailId' => $articleDetail->getId(),
                'totalReservedStock' => $totalReservedStock,
                'warehouseId' => $warehouse->getId(),
            ]
        );
        $warehouseStockCounts = $this->ensureArticleDetailHasStockCountsInWarehouse($articleDetail, $warehouse);
        $warehouseStockCounts->setReservedStock($totalReservedStock);
    }

    /**
     * @inheritdoc
     */
    public function recalculateIncomingStockForArticleDetail(ArticleDetail $articleDetail)
    {
        // Since this method might be called during a Doctrine entity lifecycle hook, we cannot flush the changes to the
        // database. However, we still like to save the new value in the database, which is why we do it via an SQL
        // query. That said, we still update the `$articleDetail`'s warehouse stock counts to keep the local
        // representation in sync with the database.
        $this->database->query(
            'UPDATE `pickware_erp_warehouse_article_detail_stock_counts` AS `stockCounts`
            LEFT JOIN (
                SELECT
                    `supplierOrders`.`warehouseId`,
                    `supplierOrderItems`.`articleDetailId`,
                    GREATEST(
                        SUM(`supplierOrderItems`.`orderedQuantity`) - SUM(`supplierOrderItems`.`deliveredQuantity`),
                        0
                    ) AS `incomingStock`
                FROM `pickware_erp_supplier_order_items` AS `supplierOrderItems`
                INNER JOIN `pickware_erp_supplier_orders` AS `supplierOrders`
                    ON `supplierOrderItems`.`supplierOrderId` = `supplierOrders`.`id`
                WHERE
                    `supplierOrders`.`statusId` NOT IN (4, 5)
                    AND `supplierOrderItems`.`statusId` NOT IN (4, 5)
                    AND `supplierOrderItems`.`articleDetailId` = :articleDetailId
                GROUP BY `supplierOrders`.`warehouseId`, `supplierOrderItems`.`articleDetailId`
            ) AS `supplierOrderIncomingStocks`
                ON `supplierOrderIncomingStocks`.`warehouseId` = `stockCounts`.`warehouseId`
                AND `supplierOrderIncomingStocks`.`articleDetailId` = `stockCounts`.`articleDetailId`
            SET `stockCounts`.`incomingStock` = IFNULL(`supplierOrderIncomingStocks`.`incomingStock`, 0)
            WHERE `stockCounts`.`articleDetailId` = :articleDetailId',
            [
                'articleDetailId' => $articleDetail->getId(),
            ]
        );

        // Retrieve updated incoming stocks per warehouse and calculate the total incoming stock for this article detail
        $incomingStocksPerWarehouse = $this->database->fetchPairs(
            'SELECT
                `warehouseId`,
                `incomingStock`
            FROM `pickware_erp_warehouse_article_detail_stock_counts`
            WHERE `articleDetailId` = :articleDetailId',
            [
                'articleDetailId' => $articleDetail->getId(),
            ]
        );
        $warehouses = $this->entityManager->getRepository(Warehouse::class)->findAll();

        array_map(function (Warehouse $warehouse) use ($articleDetail, $incomingStocksPerWarehouse) {
            $incomingStock = intval($incomingStocksPerWarehouse[$warehouse->getId()]);
            $warehouseStockCounts = $this->ensureArticleDetailHasStockCountsInWarehouse($articleDetail, $warehouse);
            $warehouseStockCounts->setIncomingStock($incomingStock);
        }, $warehouses);

        $totalIncomingStock = array_sum(array_values($incomingStocksPerWarehouse));
        $this->database->query(
            'UPDATE `s_articles_attributes`
            SET `pickware_incoming_stock` = :totalIncomingStock
            WHERE `articledetailsID` = :articleDetailId',
            [
                'articleDetailId' => $articleDetail->getId(),
                'totalIncomingStock' => $totalIncomingStock,
            ]
        );
        $attribute = $this->ensureArticleDetailHasAttribute($articleDetail);
        $attribute->setPickwareIncomingStock($totalIncomingStock);
    }

    /**
     * @param ArticleDetail $articleDetail
     * @return ArticleAttribute
     */
    protected function ensureArticleDetailHasAttribute(ArticleDetail $articleDetail)
    {
        if (!$articleDetail->getAttribute()) {
            $attribute = new ArticleAttribute();
            $articleDetail->setAttribute($attribute);
            $attribute->setArticleDetail($articleDetail);
            $this->entityManager->persist($attribute);
        }

        return $articleDetail->getAttribute();
    }

    /**
     * @param ArticleDetail $articleDetail
     * @param Warehouse $warehouse
     * @return WarehouseArticleDetailStockCount
     */
    protected function ensureArticleDetailHasStockCountsInWarehouse(ArticleDetail $articleDetail, Warehouse $warehouse)
    {
        $articleDetailStockCounts = $this->entityManager->getRepository(WarehouseArticleDetailStockCount::class)->findOneBy([
            'articleDetail' => $articleDetail,
            'warehouse' => $warehouse,
        ]);
        if (!$articleDetailStockCounts) {
            // Create a new stock counts entity
            $articleDetailStockCounts = new WarehouseArticleDetailStockCount($warehouse, $articleDetail);
            $this->entityManager->persist($articleDetailStockCounts);
        }

        return $articleDetailStockCounts;
    }

    /**
     * Note: By substracting the sum of the minium of `deliveredQuantity` and `orderQuantity` from the sum of
     * `deliveredQuantity` we ensure that the calculated incoming stock will not be negative.
     *
     * @param ArticleDetail $articleDetail
     * @return QueryBuilder
     */
    protected function createIncomingStockCalculationQueryBuilder(ArticleDetail $articleDetail)
    {
        return $this->entityManager->createQueryBuilder()
            ->select(
                'SUM(supplierOrderItems.orderedQuantity) - SUM(
                    CASE WHEN (supplierOrderItems.deliveredQuantity < supplierOrderItems.orderedQuantity)
                        THEN supplierOrderItems.deliveredQuantity
                        ELSE supplierOrderItems.orderedQuantity
                    END
                )'
            )
            ->from(SupplierOrderItem::class, 'supplierOrderItems')
            ->innerJoin('supplierOrderItems.supplierOrder', 'supplierOrders')
            ->where('supplierOrderItems.articleDetail = :articleDetail')
            ->andWhere('supplierOrders.statusId != :orderStatusCanceled')
            ->andWhere('supplierOrders.statusId != :orderStatusCompletelyRecevied')
            ->andWhere('supplierOrderItems.statusId != :itemStatusCanceled')
            ->andWhere('supplierOrderItems.statusId != :itemStatusCompletelyRecevied')
            ->setParameters([
                'articleDetail' => $articleDetail,
                'orderStatusCanceled' => SupplierOrderStatus::VALUE_CANCELED,
                'orderStatusCompletelyRecevied' => SupplierOrderStatus::VALUE_COMPLETELY_RECEIVED,
                'itemStatusCanceled' => SupplierOrderItemStatus::VALUE_CANCELED,
                'itemStatusCompletelyRecevied' => SupplierOrderItemStatus::VALUE_COMPLETELY_RECEIVED,
            ]);
    }
}
