<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\Document;

use Shopware\Components\Model\ModelManager;
use Shopware\Models\Document\Document as DocumentType;
use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Models\Order\Document\Document as OrderDocument;
use Shopware\Models\Order\Order;
use Shopware\Models\Tax\Tax;
use Shopware\Plugins\ViisonCommon\Classes\Util\Document;

class CancellationInvoiceContent
{
    const DATE_FORMAT = 'd.m.Y';

    /**
     * @var string
     */
    private $invoiceNumber;

    /**
     * @var \DateTime
     */
    private $date;

    /**
     * @var string
     */
    private $comment = '';

    /**
     * @var array
     */
    private $positionsForOrderDetails = [];

    /**
     * @var array
     */
    private $positionsWithoutOrderDetail = [];

    /**
     * @var float
     */
    private $shippingCosts = 0.0;

    /**
     * @var float
     */
    private $shippingCostsNet = 0.0;

    /**
     * @var float
     */
    private $shippingCostsTaxRate = 0.0;

    /**
     * @var Order
     */
    private $order;

    /**
     * @param Order $order
     * @param string $invoiceNumber
     */
    public function __construct(Order $order, $invoiceNumber)
    {
        $this->order = $order;
        $this->invoiceNumber = $invoiceNumber;
        $this->date = new \DateTime();
    }

    /**
     * @param OrderDocumentCreation $orderDocumentCreationService
     * @param ModelManager $entityManager
     * @return OrderDocument
     */
    public function createDocument(OrderDocumentCreation $orderDocumentCreationService, $entityManager)
    {
        return $orderDocumentCreationService->createOrderCancellationDocument(
            $this->order,
            $this->getCancellationDocumentType($entityManager),
            $this->createPositionsArray(),
            $this->invoiceNumber,
            $this->comment,
            $this->date->format(self::DATE_FORMAT),
            false
        );
    }

    /**
     * Outputs a preview of a cancellation document to the client.
     *
     * Script execution is ended after output.
     *
     * @param OrderDocumentCreation $orderDocumentCreationService
     * @param ModelManager $entityManager
     */
    public function showDocumentPreview(OrderDocumentCreation $orderDocumentCreationService, $entityManager)
    {
        $orderDocumentCreationService->showOrderCancellationDocumentPreview(
            $this->order,
            $this->getCancellationDocumentType($entityManager),
            $this->createPositionsArray(),
            $this->invoiceNumber,
            $this->comment,
            $this->date->format(self::DATE_FORMAT),
            false
        );
    }

    /**
     * @return array
     */
    private function createPositionsArray()
    {
        $positions = [];
        foreach ($this->positionsForOrderDetails as $positionForOrderDetail) {
            if ($positionForOrderDetail['quantity'] === 0) {
                continue;
            }
            $positions[] = [
                'id' => $positionForOrderDetail['orderDetail']->getId(),
                'quantity' => $positionForOrderDetail['quantity'],
            ];
        }
        foreach ($this->positionsWithoutOrderDetail as $positionWithoutOrderDetail) {
            if ($positionWithoutOrderDetail['quantity'] === 0) {
                continue;
            }
            $position = [
                'name' => $positionWithoutOrderDetail['name'],
                'articleordernumber' => $positionWithoutOrderDetail['articleNumber'],
                'price' => $positionWithoutOrderDetail['price'],
                'quantity' => $positionWithoutOrderDetail['quantity'],
            ];
            if ($positionWithoutOrderDetail['tax'] instanceof Tax) {
                $position['taxId'] = $positionWithoutOrderDetail['tax']->getId();
                $position['taxID'] = $position['taxId'];
            } elseif ($positionWithoutOrderDetail['tax'] !== null) {
                $position['tax_rate'] = $positionWithoutOrderDetail['tax'];
            }
            $positions[] = $position;
        }
        if ($this->shippingCosts !== 0.0 || $this->shippingCostsNet !== 0.0) {
            // The shipping costs are added as dummy item. The order has already been set to zero shipping costs, that's
            // why they would never appear on the cancellation invoice.
            $taxRate = round($this->shippingCostsTaxRate, 2);

            // If the order is a net order AND the shipping cost taxes are 0.0 (with a tax rate of 0.0) the generation
            // of the invoice document would result in calling
            // \Shopware\Components\Cart\NetRounding\RoundLineAfterQuantity::round
            // with null as second parameter here:
            // https://github.com/shopware/shopware/blob/900f252c0f21b13a5767e8114e957a52ab631d7b/engine/Shopware/Models/Document/Order.php#L559
            // Since we cannot provide a tax (as object) or a taxID here, the following work-around is used:
            if ($taxRate === 0.0 && $this->order->getNet()) {
                $taxRate = 1e-100;
                // Almost zero tax rate to NOT pass the the empty-check here
                // https://github.com/shopware/shopware/blob/900f252c0f21b13a5767e8114e957a52ab631d7b/engine/Shopware/Models/Document/Order.php#L531
                // This will lead to the method \Shopware\Components\Cart\NetRounding\RoundLineAfterQuantity::round
                // being called with a valid float as second argument but the result of the addition is still 100 here:
                // https://github.com/shopware/shopware/blob/900f252c0f21b13a5767e8114e957a52ab631d7b/engine/Shopware/Components/Cart/NetRounding/RoundLineAfterQuantity.php#L31
                // Therefore passing 1e-100 as $tax is exactly like passing 0.0.
            }
            $positions[] = [
                // Shopware will translate the name, see https://github.com/shopware/shopware/blob/948cb23870f5fbd821f5e15643ac45c27aaaccbd/themes/Frontend/Bare/documents/index.tpl#L215
                'name' => 'Versandkosten',
                'tax_rate' => $taxRate,
                'price' => $this->order->getNet() ? $this->shippingCostsNet : $this->shippingCosts,
                'quantity' => 1,
            ];
        }

        return $positions;
    }

    /**
     * @param ModelManager $entityManager
     * @return DocumentType
     */
    private function getCancellationDocumentType(ModelManager $entityManager)
    {
        /** @var DocumentType $documentType */
        $documentType = $entityManager->find(DocumentType::class, Document::DOCUMENT_TYPE_ID_CANCELLATION);

        return $documentType;
    }

    /**
     * @param float $shippingCosts
     * @param float $shippingCostsNet
     * @param float $taxRate
     */
    public function setShippingCosts($shippingCosts, $shippingCostsNet, $taxRate)
    {
        // floatval is used because in createPositionsArray an equal comparison is used
        $this->shippingCosts = floatval($shippingCosts);
        $this->shippingCostsNet = floatval($shippingCostsNet);
        $this->shippingCostsTaxRate = floatval($taxRate);
    }

    /**
     * @param string $invoiceNumber
     */
    public function setInvoiceNumber($invoiceNumber)
    {
        $this->invoiceNumber = $invoiceNumber;
    }

    /**
     * @param \DateTime $date
     */
    public function setDate(\DateTime $date)
    {
        $this->date = $date;
    }

    /**
     * @param string $comment
     */
    public function setComment($comment)
    {
        $this->comment = $comment;
    }

    /**
     * @param OrderDetail $orderDetail
     * @param int $quantity
     */
    public function addPositionForOrderDetail(OrderDetail $orderDetail, $quantity)
    {
        if (!array_key_exists($orderDetail->getId(), $this->positionsForOrderDetails)) {
            $this->positionsForOrderDetails[$orderDetail->getId()] = [
                'orderDetail' => $orderDetail,
                'quantity' => 0,
            ];
        }
        $this->positionsForOrderDetails[$orderDetail->getId()]['quantity'] += $quantity;
    }

    /**
     * @param string $name
     * @param float $price
     * @param int $quantity
     * @param Tax|int|null $tax
     * @param string $articleNumber
     */
    public function addPositionWithoutOrderDetail($name, $price, $quantity, $tax = null, $articleNumber = '')
    {
        if ($tax !== null && !($tax instanceof Tax) && !is_int($tax)) {
            throw new \InvalidArgumentException(sprintf(
                'Parameter $tax of method %s must be either instance of %s, integer or null',
                __METHOD__,
                Tax::class
            ));
        }

        $this->positionsWithoutOrderDetail[] = [
            'articleNumber' => $articleNumber,
            'name' => $name,
            'price' => $price,
            'quantity' => $quantity,
            'tax' => $tax,
        ];
    }

    /**
     * @return bool true, iff the generated invoice would be completely empty
     */
    public function isEmpty()
    {
        return (
            count($this->positionsForOrderDetails) === 0
            && count($this->positionsWithoutOrderDetail) === 0
            && $this->shippingCosts === 0.0
            && $this->shippingCostsNet === 0.0
        );
    }
}
