<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\Document;

use Enlight_Hook;
use Shopware\Components\Model\ModelManager;
use Shopware\Models\Mail\Mail;
use Shopware\Models\Order\Document\Document;
use Shopware\Models\Shop\Shop;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\FileSystemExceptions\FileNotReadableException;
use Shopware\Plugins\ViisonCommon\Classes\Util\Document as ViisonCommonDocumentUtil;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Components\FileStorage\FileStorage;
use Shopware_Components_TemplateMail;

class DocumentMailingService implements DocumentMailing, Enlight_Hook
{
    /**
     * @var ModelManager
     */
    protected $entityManager;

    /**
     * @var Shopware_Components_TemplateMail
     */
    protected $templateMailFactory;

    /**
     * @var FileStorage
     */
    private $documentFileStorage;

    /**
     * @param ModelManager $entityManager
     * @param Shopware_Components_TemplateMail $templateMailFactory
     * @param FileStorage $documentFileStorage
     */
    public function __construct($entityManager, $templateMailFactory, FileStorage $documentFileStorage)
    {
        $this->entityManager = $entityManager;
        $this->templateMailFactory = $templateMailFactory;
        $this->documentFileStorage = $documentFileStorage;
    }

    /**
     * @inheritdoc
     * @throws DocumentMailingException
     */
    public function createDocumentMailFromTemplate(
        $templateName,
        array $templateContext,
        Document $document,
        Shop $shop = null,
        array $mailConfig = []
    ) {
        // Try to load the template
        $template = $this->entityManager->getRepository(Mail::class)->findOneBy([
            'name' => $templateName,
        ]);
        if (!$template) {
            throw DocumentMailingException::mailTemplateNotFound($templateName);
        }

        // Make sure to have a shop
        if (!$shop) {
            $shop = $document->getOrder()->getLanguageSubShop();
        }

        // Prepare the tempalte context
        $defaultTemplateContext = [
            'shopId' => $shop->getId(),
        ];
        $combinedTemplateContext = array_merge($defaultTemplateContext, $templateContext);

        // Create a new email using the template and context
        $mail = $this->templateMailFactory->createMail(
            $templateName,
            $combinedTemplateContext,
            $shop,
            $mailConfig
        );
        $mail->clearRecipients();

        try {
            ViisonCommonDocumentUtil::attachDocumentToMail($document, $mail);
        } catch (FileNotReadableException $e) {
            throw DocumentMailingException::documentFileNotReadable($e->getPath());
        }

        return $mail;
    }
}
