<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\Document;

use Enlight_Hook;
use Shopware\Models\Order\Document\Document;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonPickwareERP\Components\PluginConfig\PluginConfig;
use Shopware_Plugins_Core_ViisonPickwareERP_Bootstrap;

class InvoiceArchiveService implements InvoiceArchive, Enlight_Hook
{
    /**
     * @var DocumentMailing
     */
    protected $documentMailingService;

    /**
     * @var string[]
     */
    protected $archiveEmailAddresses;

    /**
     * @param PluginConfig $pluginConfig
     * @param DocumentMailing $documentMailingService
     */
    public function __construct(PluginConfig $pluginConfig, DocumentMailing $documentMailingService)
    {
        $this->documentMailingService = $documentMailingService;
        $this->archiveEmailAddresses = $pluginConfig->getInvoiceArchiveEmailAddresses();
    }

    /**
     * @inheritdoc
     */
    public function sendDocumentToInvoiceArchive(Document $document)
    {
        if (count($this->archiveEmailAddresses) === 0) {
            return;
        }

        // Create a new email from template
        $order = $document->getOrder();
        $shop = $order->getLanguageSubShop();
        $subjectOrderString = ViisonCommonUtil::getSnippetFromLocale(
            'viison_pickware_erp/components/document/invoice_archive_service',
            'email/subject/order_document_infix',
            $shop
        );
        $templateContext = [
            'documentPrefix' => ViisonCommonUtil::getTranslatedDocumentName($document, false),
            'documentIdentifier' => ViisonCommonUtil::getDocumentIdentifier($document),
            'subjectOrderString' => $subjectOrderString,
            'orderNumber' => $order->getNumber(),
            'shopName' => $shop->getName(),
        ];

        // Send the email to each archive email address individually
        foreach ($this->archiveEmailAddresses as $emailAddress) {
            // Note: We must create a email for each email address, because `Zend_Mime_Part`, which wraps the document
            // attachment, consumes the document file handle when sending the email for the first time.
            $mail = $this->documentMailingService->createDocumentMailFromTemplate(
                Shopware_Plugins_Core_ViisonPickwareERP_Bootstrap::DOCUMENT_MAILER_MAIL_TEMPLATE_NAME . '_archive',
                $templateContext,
                $document,
                $shop
            );
            $mail->clearRecipients();
            $mail->addTo(trim($emailAddress));
            $mail->send();
        }
    }
}
