<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\Document;

use Shopware\Models\Document\Document as DocumentType;
use Shopware\Models\Order\Document\Document;
use Shopware\Models\Order\Order;

interface OrderDocumentCreation
{
    /**
     * Creates and returns a new cancellation document of given `$documentType` for the given `$order`. The created
     * document will only contain the positions contained in `$positions`.
     *
     * This method supports creating multiple documents of the same type for a single order.
     *
     * @param Order $order
     * @param DocumentType $documentType
     * @param array $positions
     * @param string $invoiceNumber
     * @param string $comment
     * @param string|null $dateString A formatted date string for the invoice. If null, date of today is used.
     * @param boolean $cancelShippingCost
     * @return Document
     */
    public function createOrderCancellationDocument(
        Order $order,
        DocumentType $documentType,
        array $positions,
        $invoiceNumber,
        $comment,
        $dateString,
        $cancelShippingCost
    );

    /**
     * Creates and returns a new document of given `$documentType` for the given `$order`.
     *
     * @param Order $order
     * @param DocumentType $documentType
     * @param array $documentConfig
     * @return Document
     */
    public function createOrderDocument(Order $order, DocumentType $documentType, array $documentConfig = []);

    /**
     * Outputs a preview of a cancellation document to the client.
     *
     * Script execution is ended after output.
     *
     * @param Order $order
     * @param DocumentType $documentType
     * @param array $positions
     * @param string $invoiceNumber
     * @param string $comment
     * @param string|null $dateString A formatted date string for the invoice. If null, date of today is used.
     * @param boolean $cancelShippingCost
     */
    public function showOrderCancellationDocumentPreview(
        Order $order,
        DocumentType $documentType,
        array $positions,
        $invoiceNumber,
        $comment,
        $dateString,
        $cancelShippingCost
    );
}
