<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\Document;

use Enlight_Event_EventManager;
use Enlight_Hook;
use ReflectionObject;
use Shopware\Components\Model\ModelManager;
use Shopware\Models\Document\Document as DocumentType;
use Shopware\Models\Order\Document\Document;
use Shopware\Models\Order\Order;
use Shopware\Models\Tax\Tax;
use Shopware_Components_Document;
use Shopware_Models_Document_Order;
use Shopware\Plugins\ViisonCommon\Components\DocumentComponentListenerService;

class OrderDocumentCreationService implements OrderDocumentCreation, Enlight_Hook
{
    const DEFAULT_DOCUMENT_CONFIG = [
        '_renderer' => 'pdf',
    ];

    /**
     * @var ModelManager
     */
    protected $entityManager;

    /**
     * @var Enlight_Event_EventManager
     */
    protected $eventManager;

    /**
     * @var DocumentComponentListenerService
     */
    protected $documentComponentListenerService;

    /**
     * @param ModelManager $entityManager
     * @param Enlight_Event_EventManager $eventManager
     * @param DocumentComponentListenerService $documentComponentListenerService
     */
    public function __construct(
        $entityManager,
        $eventManager,
        DocumentComponentListenerService $documentComponentListenerService
    ) {
        $this->entityManager = $entityManager;
        $this->eventManager = $eventManager;
        $this->documentComponentListenerService = $documentComponentListenerService;
    }

    /**
     * @inheritdoc
     */
    public function createOrderCancellationDocument(
        Order $order,
        DocumentType $documentType,
        array $positions,
        $invoiceNumber,
        $comment,
        $dateString,
        $cancelShippingCost
    ) {
        $this->setPositionsForNextCancellationDocument($positions);

        // Register temporary hooks that allow to create more than one document of the same type for the same order
        $this->documentComponentListenerService->allowNextDocumentToHaveTypeOfExistingDocument();

        // Create and render the document
        $document = $this->createOrderDocument(
            $order,
            $documentType,
            [
                'bid' => $invoiceNumber,
                'docComment' => $comment,
                'date' => $dateString,
                'shippingCostsAsPosition' => ($documentType->getId() === 4) ? $cancelShippingCost : !$cancelShippingCost,
            ]
        );

        return $document;
    }

    /**
     * @inheritdoc
     */
    public function showOrderCancellationDocumentPreview(
        Order $order,
        DocumentType $documentType,
        array $positions,
        $invoiceNumber,
        $comment,
        $dateString,
        $cancelShippingCost
    ) {
        $this->setPositionsForNextCancellationDocument($positions);

        // Create and render the document
        $this->createOrderDocument(
            $order,
            $documentType,
            [
                'invoiceNumber' => $invoiceNumber,
                'bid' => 12345,
                'docComment' => $comment,
                'date' => $dateString,
                'shippingCostsAsPosition' => ($documentType->getId() === 4) ? $cancelShippingCost : !$cancelShippingCost,
                '_preview' => true,
            ]
        );
    }

    /**
     * Sets the positions that should appear on the next cancellation document.
     *
     * @param array $positions
     */
    private function setPositionsForNextCancellationDocument(array $positions)
    {
        // Separate the items in to 'real' items, i.e. backed by an order detail, and dummy items
        $positionIndex = [];
        $dummyPositions = [];
        $defaultTax = $this->entityManager->getRepository(Tax::class)->findOneBy([], ['tax' => 'DESC']);
        $defaultTaxRate = ($defaultTax) ? $defaultTax->getTax() : 0.0;
        foreach ($positions as $position) {
            if (!empty($position['id'])) {
                $positionIndex[$position['id']] = $position;
                continue;
            }

            // Determine the tax rate, using the highest possible tax rate as fallback, if none was set
            if (!isset($position['tax_rate'])) {
                $taxRate = $defaultTaxRate;
                if (isset($position['taxId'])) {
                    $tax = $this->entityManager->find(Tax::class, $position['taxId']);
                    if ($tax) {
                        $taxRate = $tax->getTax();
                    }
                }
                $position['tax_rate'] = floatval($taxRate); // Cast value to remove "unnecessary" fractions: 7.00 -> 7
            }

            $dummyPositions[] = $position;
        }

        // Create and add a temporary hook that filters the position on the document to show only the ones passed to
        // this method
        $this->documentComponentListenerService->filterPositionsOnNextDocument(
            function (Shopware_Models_Document_Order $documentModel, array $positions) use ($positionIndex, $dummyPositions) {
                $filteredPositions = [];
                foreach ($positions as $position) {
                    if (!isset($positionIndex[$position['id']])) {
                        continue;
                    }

                    // Override the position's quantity with the one passed to the enclosing method
                    $position['quantity'] = $positionIndex[$position['id']]['quantity'];
                    $filteredPositions[] = $position;
                }

                return array_merge($filteredPositions, $dummyPositions);
            }
        );
    }

    /**
     * @inheritdoc
     */
    public function createOrderDocument(Order $order, DocumentType $documentType, array $documentConfig = [])
    {
        // Determine the document config
        $documentConfig = array_merge(
            self::DEFAULT_DOCUMENT_CONFIG,
            ['shippingCostsAsPosition' => ($documentType->getId() !== 2)],
            $documentConfig
        );

        // Create and render the document (this loads all order data)
        $documentComponent = Shopware_Components_Document::initDocument(
            $order->getId(),
            $documentType->getId(),
            $documentConfig
        );
        $documentComponent->render();

        // Use reflection to get the ID of the created document
        $documentCommentReflection = new ReflectionObject($documentComponent);
        $documentRowIdProperty = $documentCommentReflection->getProperty('_documentRowID');
        $documentRowIdProperty->setAccessible(true);
        $documentId = $documentRowIdProperty->getValue($documentComponent);

        return $this->entityManager->find(Document::class, $documentId);
    }
}
