<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\Document;

use Enlight_Hook;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Order\Order;
use Shopware\Plugins\ViisonCommon\Classes\ArrayAccessorWrapper;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\BinLocationStockChange;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\StockChangeListFactory;
use Shopware_Components_Snippet_Manager;

class PickListDocumentCreationService implements PickListDocumentCreation, Enlight_Hook
{
    /**
     * @var ModelManager
     */
    protected $entityManager;

    /**
     * @var Shopware_Components_Snippet_Manager
     */
    protected $snippetManager;

    /**
     * @var StockChangeListFactory
     */
    protected $stockChangeListFactory;

    /**
     * @param ModelManager $entityManager
     * @param Shopware_Components_Snippet_Manager $entityManager
     * @param StockChangeListFactory $stockChangeListFactory
     */
    public function __construct(
        $entityManager,
        $snippetManager,
        StockChangeListFactory $stockChangeListFactory
    ) {
        $this->entityManager = $entityManager;
        $this->snippetManager = $snippetManager;
        $this->stockChangeListFactory = $stockChangeListFactory;
    }

    /**
     * @inheritdoc
     */
    public function createPickListDocumentTemplateVariables(array $templateVariables, Warehouse $warehouse)
    {
        // Add warehouse
        $templateVariables['warehouse'] = $warehouse;

        // Add shop
        /** @var Order $order */
        $order = $this->entityManager->find(Order::class, intval($templateVariables['Order']['_id']));
        $templateVariables['Shop'] = $order->getLanguageSubShop();

        // Add picking instructions as positions
        $pickingInstructions = $this->getSortedPickingInstructions($templateVariables, $warehouse);
        $snippetNamespace = $this->snippetManager->getNamespace('backend/viison_pickware_erp_pick_list/document');
        $templateVariables['Order']['_positions'] = array_map(function ($pickingInstruction) use ($snippetNamespace) {
            $pickListPosition = $pickingInstruction['position'];
            /** @var BinLocationStockChange $stockChange */
            $stockChange = $pickingInstruction['stockChange'];
            $pickListPosition['quantity'] = -1 * $stockChange->getStockChange();
            if ($stockChange->getBinLocation()->isNullBinLocation()) {
                $pickListPosition['binLocation'] = $snippetNamespace->get('null_bin_location');
            } else {
                $pickListPosition['binLocation'] = $stockChange->getBinLocation()->getCode();
            }

            $pickListPosition['meta'] = Shopware()->Modules()->Articles()->sGetPromotionById('fix', 0, $pickListPosition['articleordernumber']);

            return $pickListPosition;
        }, $pickingInstructions);

        // Since the number of positions changed, we need to rearrange the document pages (re-chunk positions)
        $pageBreak = intval($templateVariables['Document']['pagebreak']);
        $pages = array_chunk($templateVariables['Order']['_positions'], $pageBreak, true);
        $templateVariables['Pages'] = $pages;

        return $templateVariables;
    }

    /**
     * @inheritdoc
     */
    public function createPickListDocumentTemplatePreviewVariables(\Shopware_Components_Document $document)
    {
        $templateVariables = $document->_view->getTemplateVars();

        // Create new template variables
        $shop = [
            'name' => 'Demo Shop',
        ];
        $warehouse = [
            'name' => 'Main Warehouse',
            'code' => 'MW',
        ];

        // Modify existing data as it is needed in the template
        $templateVariables['Order']['_user']['billing'] = $templateVariables['Order']['_billing'];
        $templateVariables['Order']['_user']['shipping'] = $templateVariables['Order']['_shipping'];

        // Modify the pages (positions) by rearranging the positions and adding bin location information
        $page = $templateVariables['Pages'][0];
        $page[2] = $page[1];
        $page[3] = $page[1];
        $page[4] = $page[1];
        $page[1] = $page[0];
        $page[0]['binLocation'] = 'A-1-1';
        $page[0]['quantity'] = 25;
        $page[1]['binLocation'] = 'A-1-2';
        $page[1]['quantity'] = 10;
        $page[2]['binLocation'] = 'A-1-3';
        $page[2]['quantity'] = 4;
        $page[3]['binLocation'] = 'B-2-1';
        $page[3]['quantity'] = 9;
        $page[4]['binLocation'] = 'C-1-3';
        $page[4]['quantity'] = 7;
        $templateVariables['Pages'][0] = $page;

        // Prepare template variables with their respective name in the document template
        return [
            'Shop' => new ArrayAccessorWrapper($shop),
            'warehouse' => new ArrayAccessorWrapper($warehouse),
            'Order' => $templateVariables['Order'],
            'User' => $templateVariables['Order']['_user'],
            'Pages' => $templateVariables['Pages'],
        ];
    }

    /**
     * Fetches all stock changes for the order positions of the given template variables and sorts them by bin location.
     *
     * @param array $templateVars
     * @param Warehouse $warehouse
     * @return array
     * @throws PickListDocumentException Iff no picking instructions can be created for the given `$templateVars` (i.e.
     *         the pick list is empty).
     */
    protected function getSortedPickingInstructions(array $templateVars, Warehouse $warehouse)
    {
        // Fetch stock changes for each order position (an order position stock change may be divided into multiple bin
        // locations) bin location to the document. Reminder: stock changes are negative here.
        $articleDetailsByNumber = $this->getArticleDetailsByOrderNumberFromPositions(
            $templateVars['Order']['_positions']
        );
        $pickingInstructions = [];
        foreach ($templateVars['Order']['_positions'] as $position) {
            $positionIsNotRelevant = (
                !isset($articleDetailsByNumber[$position['articleordernumber']])
                || intval($position['quantity']) <= 0
                || intval($position['modus']) >= 2
                || $position['modus'] === false
            );
            if ($positionIsNotRelevant) {
                continue;
            }

            $stockChangeList = $this->stockChangeListFactory->createStockChangeList(
                $warehouse,
                $articleDetailsByNumber[$position['articleordernumber']],
                -1 * intval($position['quantity'])
            );
            foreach ($stockChangeList->getBinLocationStockChanges() as $stockChange) {
                $pickingInstructions[] = [
                    'position' => $position,
                    'stockChange' => $stockChange,
                ];
            }
        }

        if (count($pickingInstructions) === 0) {
            throw PickListDocumentException::noPickableItemsInOrder(
                $templateVars['Order']['_order']['ordernumber'],
                $templateVars['warehouse']
            );
        }

        return $this->sortPickingInstructions($pickingInstructions);
    }

    /**
     * @param array[] $positions
     * @return ArticleDetail[]
     */
    protected function getArticleDetailsByOrderNumberFromPositions(array $positions)
    {
        $articleOrderNumbers = array_map(
            function (array $position) {
                return $position['articleordernumber'];
            },
            $positions
        );

        $articleDetails = $this->entityManager->getRepository(ArticleDetail::class)->findBy([
            'number' => $articleOrderNumbers,
        ]);

        $articleDetailsByNumber = [];
        foreach ($articleDetails as $articleDetail) {
            $articleDetailsByNumber[$articleDetail->getNumber()] = $articleDetail;
        }

        return $articleDetailsByNumber;
    }

    /**
     * Sorts stock changes by their bin location: null bin location to the top, sort by bin location code
     * otherwise and by article number if positions share the same bin location.
     *
     * @param array $pickingInstructions
     * @return array
     */
    protected function sortPickingInstructions(array $pickingInstructions)
    {
        usort(
            $pickingInstructions,
            function ($lhs, $rhs) {
                /** @var BinLocation $lhsBinLocation */
                $lhsBinLocation = $lhs['stockChange']->getBinLocation();
                /** @var BinLocation $rhsBinLocation */
                $rhsBinLocation = $rhs['stockChange']->getBinLocation();

                // Sort default bin location to the top of the list
                if ($lhsBinLocation->isNullBinLocation() && !$rhsBinLocation->isNullBinLocation()) {
                    return -1;
                }
                if (!$lhsBinLocation->isNullBinLocation() && $rhsBinLocation->isNullBinLocation()) {
                    return 1;
                }

                // Sort by bin location code
                if ($lhsBinLocation->getCode() !== $rhsBinLocation->getCode()) {
                    return strcmp(
                        preg_replace('/[^a-zA-Z0-9]+/', '', $lhsBinLocation->getCode()),
                        preg_replace('/[^a-zA-Z0-9]+/', '', $rhsBinLocation->getCode())
                    );
                }

                // Both stock changes share the same bin location (including null bin location), sort by article number
                return strcmp(
                    preg_replace('/[^a-zA-Z0-9]+/', '', $lhs['position']['articleordernumber']),
                    preg_replace('/[^a-zA-Z0-9]+/', '', $rhs['position']['articleordernumber'])
                );
            }
        );

        return $pickingInstructions;
    }
}
