<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\OrderDetailQuantityCalculator;

use Shopware\Models\Order\Detail as OrderDetail;

interface OrderDetailQuantityCalculator
{
    /**
     * Returns the total ordered quantity, containing cancelled items.
     *
     * @param OrderDetail $orderDetail
     * @return int
     */
    public function calculateOrderedQuantity(OrderDetail $orderDetail);

    /**
     * Returns number of items that are remaining to ship.
     *
     * Aka: "Left for shipping", "open", "free".
     *
     * Parameter $orderDetail does not have a type-hint because the proxy generation is SW 5.2 is broken. It would
     * generate a proxy classes without type-hints what leads to methods being incompatible with their parents. Since
     * we need to hook this method in ShopwarePickwareMobile the type-hint is dropped.
     *
     * @param OrderDetail $orderDetail
     * @return int
     */
    public function calculateRemainingQuantityToShip($orderDetail);

    /**
     * Returns total quantity of returned items.
     *
     * @param OrderDetail $orderDetail
     * @return int
     */
    public function calculateReturnedQuantity(OrderDetail $orderDetail);

    /**
     * Returns number of items that have been cancelled after they have been returned.
     *
     * @param OrderDetail $orderDetail
     * @return int
     */
    public function calculateCancelledQuantityOfReturnShipmentItems(OrderDetail $orderDetail);

    /**
     * Returns number of cancelled items that were cancelled before ever shipped
     *
     * @param OrderDetail $orderDetail
     * @return int
     */
    public function calculateCancelledRemainingQuantityToShip(OrderDetail $orderDetail);

    /**
     * Returns number of items shipped to the customer but not returned yet.
     *
     * This is the number of items the customer currently has at its own place.
     *
     * @param OrderDetail $orderDetail
     * @return int
     */
    public function calculateShippedAndNotReturnedQuantity(OrderDetail $orderDetail);
}
