<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\OrderDetailQuantityCalculator;

use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentItem;
use Shopware\Models\Order\Detail as OrderDetail;

class OrderDetailQuantityCalculatorService implements OrderDetailQuantityCalculator, \Enlight_Hook
{
    /**
     * @var ModelManager
     */
    private $entityManager;

    /**
     * @param ModelManager $entityManager
     */
    public function __construct($entityManager)
    {
        $this->entityManager = $entityManager;
    }

    /**
     * @inheritdoc
     */
    public function calculateOrderedQuantity(OrderDetail $orderDetail)
    {
        return $orderDetail->getQuantity() + $orderDetail->getAttribute()->getPickwareCanceledQuantity();
    }

    /**
     * @inheritdoc
     */
    public function calculateRemainingQuantityToShip($orderDetail)
    {
        return $orderDetail->getQuantity() + $this->calculateCancelledQuantityOfReturnShipmentItems($orderDetail) - $orderDetail->getShipped();
    }

    /**
     * @inheritdoc
     */
    public function calculateReturnedQuantity(OrderDetail $orderDetail)
    {
        $returnShipmentItems = $this->entityManager->getRepository(ReturnShipmentItem::class)->findBy([
            'orderDetail' => $orderDetail,
        ]);

        return array_sum(array_map(function (ReturnShipmentItem $returnShipmentItem) {
            return $returnShipmentItem->getReturnedQuantity();
        }, $returnShipmentItems));
    }

    /**
     * @inheritdoc
     */
    public function calculateCancelledQuantityOfReturnShipmentItems(OrderDetail $orderDetail)
    {
        $returnShipmentItems = $this->entityManager->getRepository(ReturnShipmentItem::class)->findBy([
            'orderDetail' => $orderDetail,
        ]);

        return array_sum(array_map(function (ReturnShipmentItem $returnShipmentItem) {
            return $returnShipmentItem->getCancelledQuantity();
        }, $returnShipmentItems));
    }

    /**
     * @inheritdoc
     */
    public function calculateCancelledRemainingQuantityToShip(OrderDetail $orderDetail)
    {
        return $orderDetail->getAttribute()->getPickwareCanceledQuantity() - $this->calculateCancelledQuantityOfReturnShipmentItems($orderDetail);
    }

    /**
     * @inheritdoc
     */
    public function calculateShippedAndNotReturnedQuantity(OrderDetail $orderDetail)
    {
        return $orderDetail->getShipped() - $this->calculateReturnedQuantity($orderDetail);
    }
}
