<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\OrderDetailQuantityValidator;

use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Plugins\ViisonPickwareERP\Components\OrderDetailQuantityCalculator\OrderDetailQuantityCalculator;

class OrderDetailQuantityValidatorService implements OrderDetailQuantityValidator, \Enlight_Hook
{
    /**
     * @var OrderDetailQuantityCalculator
     */
    private $orderDetailQuantityCalculator;

    /**
     * @param OrderDetailQuantityCalculator $orderDetailQuantityCalculator
     */
    public function __construct(OrderDetailQuantityCalculator $orderDetailQuantityCalculator)
    {
        $this->orderDetailQuantityCalculator = $orderDetailQuantityCalculator;
    }

    /**
     * {@inheritdoc}
     */
    public function isQuantityAndShippedQuantityCombinationValid($orderDetail, $quantity, $shippedQuantity)
    {
        // Both values must not be lower than 0

        // Constraints for quantity:
        //   - Quantity has no upper limit.
        //   - There cannot be less items ordered than shipped. Therefore following must always be true:
        //         (original quantity) >= shipped
        //     <=> quantity + cancelled >= shipped
        //     <=> shipped <= quantity + cancelled

        // Constraints for shipped:
        //   - Shipped can never be lower than all returned articles
        //   - Shipped can never be higher than (original quantity) - (unshipped cancelled) Therefore the following must
        //     be true:
        //       shipped >= returned
        //     And one of this equivalent conditions:
        //           shipped <= (original quantity) - (unshipped cancelled)
        //       <=> shipped <= quantity - cancelled - (unshipped cancelled)
        //       <=> shipped <= quantity - cancelled - (cancelled - (returned cancelled))
        //       <=> shipped <= quantity + (returned cancelled)

        // The condition
        //   shipped <= quantity + (returned cancelled)
        // is stronger than
        //   shipped <= quantity + cancelled
        // because cancelled = (returned cancelled) + (unshipped cancelled)
        // therefore we do not need the last one

        $returnedQuantity = $this->orderDetailQuantityCalculator->calculateReturnedQuantity($orderDetail);
        $cancelledQuantityOfReturnShipmentItems = $this->orderDetailQuantityCalculator->calculateCancelledQuantityOfReturnShipmentItems($orderDetail);

        return $shippedQuantity >= 0
            && $quantity >= 0
            && $shippedQuantity <= $quantity + $cancelledQuantityOfReturnShipmentItems
            && $shippedQuantity >= $returnedQuantity;
    }

    /**
     * @inheritdoc
     */
    public function validateQuantityAndShippedQuantityCombination(OrderDetail $orderDetail, $quantity, $shippedQuantity)
    {
        if (!$this->isQuantityAndShippedQuantityCombinationValid($orderDetail, $quantity, $shippedQuantity)) {
            throw OrderDetailQuantityViolationException::violationForExistingOrderDetail($orderDetail, $quantity, $shippedQuantity);
        }
    }

    /**
     * @inheritdoc
     */
    public function validateQuantityAndShippedQuantityCombinationForOrderDetailCreation($quantity, $shippedQuantity)
    {
        if (!$this->isQuantityAndShippedQuantityCombinationValidForOrderDetailCreation($quantity, $shippedQuantity)) {
            throw OrderDetailQuantityViolationException::violationForNewOrderDetail($quantity, $shippedQuantity);
        }
    }

    /**
     * @inheritdoc
     */
    public function isQuantityAndShippedQuantityCombinationValidForOrderDetailCreation($quantity, $shippedQuantity)
    {
        return $shippedQuantity >= 0
            && $quantity >= 0
            && $quantity >= $shippedQuantity;
    }
}
