<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\OrderDetailQuantityValidator;

use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\LocalizableThrowable;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\MessageLocalization;

class OrderDetailQuantityViolationException extends \Exception implements LocalizableThrowable
{
    use MessageLocalization;

    /**
     * @param string $message
     * @param string $snippetName
     * @param array $snippetArguments
     */
    public function __construct($message, $snippetName, array $snippetArguments = [])
    {
        $this->snippetName = $snippetName;
        $this->snippetNamespace  = 'viison_pickware_erp/components/order_detail_quantity_protector/order_detail_quantity_violation_exception';
        $this->snippetArguments = $snippetArguments;

        parent::__construct(vsprintf($message, $snippetArguments));
    }

    /**
     * @param OrderDetail $orderDetail
     * @param int $quantity
     * @param int $shipped
     * @return OrderDetailQuantityViolationException
     */
    public static function violationForExistingOrderDetail(OrderDetail $orderDetail, $quantity, $shipped)
    {
        $message = 'The combination of (%d, %d) for quantity and shipped is not allowed for ' .
            'the order detail (item number: "%s", order number: "%s"). It would lead to either more items being ' .
            'shipped than ordered and not cancelled, more items being returned than shipped or at least one of the ' .
            'values being negative.';
        $snippetArguments = [
            $quantity,
            $shipped,
            $orderDetail->getArticleNumber(),
            $orderDetail->getOrder()->getNumber(),
        ];

        return new self(
            $message,
            'violation_for_existing_order_detail',
            $snippetArguments
        );
    }

    /**
     * @param int $quantity
     * @param int $shipped
     * @return OrderDetailQuantityViolationException
     */
    public static function violationForNewOrderDetail($quantity, $shipped)
    {
        $message = 'The combination of (%d, %d) for quantity and shipped is not allowed for ' .
            'the newly created order detail. It would lead to either more items being ' .
            'shipped than ordered or at least one of the values being negative.';
        $snippetArguments = [
            $quantity,
            $shipped,
        ];

        return new self(
            $message,
            'violation_for_new_order_detail',
            $snippetArguments
        );
    }
}
