<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\PluginConfig;

use Enlight_Config;
use Enlight_Hook;
use Shopware\Components\Model\ModelManager;
use Shopware\Models\Media\Album;
use Shopware\Plugins\ViisonCommon\Classes\Util\Document as DocumentUtil;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Components\HiddenConfigStorageService;
use Shopware_Plugins_Core_ViisonPickwareERP_Bootstrap as ErpBootstrap;

class PluginConfigService implements PluginConfig, Enlight_Hook
{
    /**
     * The name of the config element, which stores the id of the default barcode label preset,
     * which is the preset, that was last used for printing barcode labels using the barcode
     * label printing dialog.
     */
    const LAST_PRINTED_PRESET_IDS = 'viisonPickwareCommonLastPrintedPresetIds';

    /**
     * Folder for default barcode label templates.
     */
    const DEFAULT_TEMPLATE_PATH = '/../../Views/documents/barcode_label_templates/';

    /**
     * Folder for custom barcode label templates.
     */
    const CUSTOM_TEMPLATE_PATH = '/themes/Frontend/Bare/documents/viison_pickware_barcode_label_templates/';

    /**
     * Fallback preview image file, which is used when no template specific image
     * file can be found.
     */
    const FALLBACK_PREVIEW_IMAGE = '/Views/_resources/viison_pickware_common/img/no-preview-available.png';

    /**
     * Name of the media album used for the photos of a return shipment item.
     */
    const RETURN_SHIPMENT_MEDIA_ALBUM_NAME = 'Pickware Retouren';

    /**
     * @var Enlight_Config
     */
    protected $pluginConfig;

    /**
     * @var ModelManager
     */
    protected $entityManager;

    /**
     * @var HiddenConfigStorageService
     */
    private $hiddenConfigStorageService;

    /**
     * @var string
     */
    private $rootPath;

    /**
     * @param Enlight_Config $pluginConfig
     * @param ModelManager $entityManager
     * @param HiddenConfigStorageService $hiddenConfigStorageService
     * @param string $rootPath
     */
    public function __construct(
        Enlight_Config $pluginConfig,
        $entityManager,
        HiddenConfigStorageService $hiddenConfigStorageService,
        $rootPath
    ) {
        $this->pluginConfig = $pluginConfig;
        $this->entityManager = $entityManager;
        $this->hiddenConfigStorageService = $hiddenConfigStorageService;
        $this->rootPath = $rootPath;
    }

    /**
     * @inheritdoc
     */
    public function getPickListDocumentType()
    {
        return DocumentUtil::getDocumentType(
            ErpBootstrap::PICK_LIST_DOCUMENT_TYPE_KEY,
            ErpBootstrap::PICK_LIST_DOCUMENT_TYPE_NAME
        );
    }

    /**
     * @inheritdoc
     */
    public function getSupplierOrderDocumentTye()
    {
        return DocumentUtil::getDocumentType(
            ErpBootstrap::SUPPLIER_ORDER_DOCUMENT_TEMPLATE_KEY,
            ErpBootstrap::SUPPLIER_ORDER_DOCUMENT_TEMPLATE_NAME
        );
    }

    /**
     * @inheritdoc
     */
    public function getInvoiceArchiveEmailAddresses()
    {
         $emailAddresses = ViisonCommonUtil::safeExplode(',', $this->pluginConfig->get('invoiceArchiveEmailAddresses'));
         $emailAddresses = array_values(array_filter($emailAddresses));

         return $emailAddresses;
    }

    /**
     * @inheritdoc
     */
    public function getDisplayAboutPickwareErpWindow()
    {
        return $this->hiddenConfigStorageService->getConfigValue('pickwareErpDisplayAboutWindow');
    }

    /**
     * @inheritdoc
     */
    public function setDisplayAboutPickwareErpWindow($newValue)
    {
        $this->hiddenConfigStorageService->setConfigValue(
            'pickwareErpDisplayAboutWindow',
            'boolean',
            $newValue
        );
    }

    /**
     * @inheritdoc
     */
    public function getLastPrintedPresetIds()
    {
        $lastPrintedPresets = $this->hiddenConfigStorageService->getConfigValue(static::LAST_PRINTED_PRESET_IDS);

        if (!$lastPrintedPresets) {
            return [];
        }

        return json_decode($lastPrintedPresets, true);
    }

    /**
     * @inheritdoc
     */
    public function setLastPrintedPresetId($typeIdentifier, $presetId)
    {
        $lastPrintedPresets = $this->getLastPrintedPresetIds();
        $lastPrintedPresets[$typeIdentifier] = $presetId;

        $configValue = json_encode($lastPrintedPresets);
        $this->hiddenConfigStorageService->setConfigValue(static::LAST_PRINTED_PRESET_IDS, 'string', $configValue);
    }

    /**
     * @inheritdoc
     */
    public function clearLastPrintedPresetIds()
    {
        $this->hiddenConfigStorageService->removeConfigValue(static::LAST_PRINTED_PRESET_IDS);
    }

    /**
     * @inheritdoc
     */
    public function getBarcodeLabelTemplateDirectories()
    {
        return [
            $this->getDefaultBarcodeLabelTemplateDirectory(),
            $this->getCustomBarcodeLabelTemplateDirectory(),
        ];
    }

    /**
     * @inheritdoc
     */
    public function getDefaultBarcodeLabelTemplateDirectory()
    {
        return __DIR__ . static::DEFAULT_TEMPLATE_PATH;
    }

    /**
     * inheritdoc
     */
    public function getCustomBarcodeLabelTemplateDirectory()
    {
        return $this->rootPath . static::CUSTOM_TEMPLATE_PATH;
    }

    /**
     * @inheritdoc
     */
    public function getPreviewFallbackImagePath()
    {
        return __DIR__ . static::FALLBACK_PREVIEW_IMAGE;
    }

    /**
     * @inheritdoc
     */
    public function getReturnShipmentMediaAlbum()
    {
        /** @var Album|null $album */
        $album = $this->entityManager->getRepository(Album::class)->findOneBy([
            'name' => self::RETURN_SHIPMENT_MEDIA_ALBUM_NAME,
        ]);

        return $album;
    }

    /**
     * @inheritdoc
     */
    public function getShowCurrentStockWhenStocktaking()
    {
        return (bool) $this->pluginConfig->get('showCurrentStockWhenStocktaking');
    }
}
