<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\RestApi;

use DateTime;
use Shopware\CustomModels\ViisonPickwareERP\RestApi\RestApiIdempotentOperation;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\ValidationExceptions\AbstractValidationException;
use Shopware\Plugins\ViisonCommon\Components\ParameterValidator;
use Shopware\Plugins\ViisonPickwareERP\Components\RestApi\Idempotency\RecoveryPointResult;
use Shopware\Plugins\ViisonPickwareERP\Components\RestApi\Idempotency\ResponseResult;
use Shopware\Plugins\ViisonPickwareERP\Components\RestApi\Idempotency\RestApiIdempotencyService;
use Shopware_Controllers_Api_Rest;

abstract class AbstractRestApiBatchController extends Shopware_Controllers_Api_Rest
{
    const MAX_BATCH_SIZE = 1000;

    public function init()
    {
        $request = $this->Request();

        // Redirect the request to the correct controller action and set the parameter idempotencyKey
        // Example: PUT /api/stock/change/{idempotencyKey}
        // This request is redirected to putChangeAction. The redirection to the correct controller is done by
        // the ApiRouterService.
        $action = $request->getParams()['action'];
        $method = $request->getMethod();
        $request->setActionName(mb_strtolower($method) . ucfirst($action));

        // Save the idempotency key from the URL as request parameter
        $resourceUrl = $request->getPathInfo();
        $urlParts = explode('/', ltrim($resourceUrl, '/'), 5);
        $idempotencyKey = $urlParts[3];
        $request->setParam('idempotencyKey', $idempotencyKey);

        // Set this parameters to null as Shopware's default URL parsing fills them with pointless data, e.g. id would
        // be the action name.
        $request->setParam('id', null);
        $request->setParam('subId', null);
    }

    /**
     * Idempotently processes datasets for a batch operation sent via HttpRequest to a controller action.
     *
     * The body of the request needs to be a JSON array. Each element of the array defines one dataset for the
     * operation. E.g.
     *     [
     *         {                  \
     *             orderId: 1,    |--> an operation dataset
     *             statusId: 2    |
     *         },                 /
     *         ...
     *     ]
     *
     * First, each dataset array is passed to the callback $validator. This callback needs to verify that the dataset
     * has a valid format that can be processed by the operator. If the dataset does not have a valid format, the
     * callback should throw an AbstractValidationException on a validation error. If the validation has failed, the
     * request is considered as finished and will not executed any operations.
     *
     * After that the passed callback $operator is called for each dataset. Its task is to execute the actual business
     * logic for a dataset. This callback should return an array informing about the success of the operation. If
     * successfully, it should return ['success' => true], otherwise it should return
     * ['success' => false, 'message' => 'A descriptive error message about what went wrong'].
     *
     * The controller action result will be a merge of all operation result, that will look like this:
     *     [
     *         'idempotencyKey' => idempotency key from the request,
     *         'success' => true, iff all operations results were successful,
     *         'results' => all returned operation results as array,
     *     ];
     *
     * The whole batch operation is executed idempotently. If the request fails and is sent again by the client, this
     * method will continue the work. No operation dataset will be processed twice.
     *
     * @param callable $validator A callable with signature `function(mixed $dataset, int $datasetKey): void`
     * @param callable $operator A callable with signature `function(mixed $dataset, int $datasetKey): array`
     */
    protected function executeBatchAction(callable $validator, callable $operator)
    {
        $idempotencyKey = $this->Request()->getParam('idempotencyKey');

        /** @var RestApiIdempotencyService $idempotencyService */
        $idempotencyService = $this->get('pickware.erp.rest_api_idempotency_service');
        $idempotentOperation = $idempotencyService->createIdempotentOperation($this->Request(), $idempotencyKey);

        $datasets = $this->Request()->getPost();

        $operationResults = null;
        $validationPhase = function () use ($validator, $idempotentOperation, $datasets) {
            try {
                ParameterValidator::assertIsArray($datasets, '');
                if (count($datasets) > self::MAX_BATCH_SIZE) {
                    return new ResponseResult(
                        413,
                        [
                            'idempotencyKey' => $idempotentOperation->getIdempotencyKey(),
                            'success' => false,
                            'message' => sprintf('Maximum batch size of %d exceeded.', self::MAX_BATCH_SIZE),
                        ]
                    );
                }
                foreach ($datasets as $datasetKey => $dataset) {
                    $validator($dataset, $datasetKey);
                }
            } catch (AbstractValidationException $e) {
                return new ResponseResult(
                    400,
                    [
                        'idempotencyKey' => $idempotentOperation->getIdempotencyKey(),
                        'success' => false,
                        'message' => $e->getMessage(),
                    ]
                );
            }

            // Start with the first dataset (that has key 0)
            return new RecoveryPointResult(0);
        };

        // Define a generic phase for processing a single dataset
        $processingPhase = function () use ($operator, &$operationResults, $datasets, $idempotentOperation) {
            $currentDatasetKey = intval($idempotentOperation->getRecoveryPoint());
            $operationResult = $operator($datasets[$currentDatasetKey], $currentDatasetKey);
            if (isset($datasets[$currentDatasetKey]['id'])) {
                $operationResult['operationId'] = $datasets[$currentDatasetKey]['id'];
            }

            if ($operationResults === null) {
                $intermediateState = $idempotentOperation->getIntermediateState();
                $operationResults = ($intermediateState) ? unserialize($intermediateState) : [];
            }
            $operationResults[] = $operationResult;

            return new RecoveryPointResult($currentDatasetKey + 1, serialize($operationResults));
        };

        $mergePhase = function () use (&$operationResults, $idempotentOperation) {
            if ($operationResults === null) {
                $intermediateState = $idempotentOperation->getIntermediateState();
                $operationResults = ($intermediateState) ? unserialize($intermediateState) : [];
            }
            $success = array_reduce(
                $operationResults,
                function ($carry, array $operationResult) {
                    return $carry && $operationResult['success'];
                },
                true
            );
            $responseData = [
                'idempotencyKey' => $idempotentOperation->getIdempotencyKey(),
                'createdAt' => (new DateTime)->format(DateTime::ISO8601),
                'success' => $success,
                'results' => $operationResults,
            ];
            if (!$success) {
                $responseData['message'] = 'At least one operation did not succeed.';
            }

            return new ResponseResult(200, $responseData);
        };

        $executionPhases = array_merge(
            [
                RestApiIdempotentOperation::RECOVERY_POINT_STARTED => $validationPhase,
            ],
            array_fill(0, count($datasets), $processingPhase),
            [
                $mergePhase,
            ]
        );
        $idempotencyService->executeIdempotently($idempotentOperation, $executionPhases);

        // Update response and view with the execution result
        $responseCode = ($idempotentOperation->getResponseCode() !== null) ? $idempotentOperation->getResponseCode() : 500;
        $this->Response()->setHttpResponseCode($responseCode);
        $responseData = ($idempotentOperation->getDecodedResponseData()) ?: [];
        $this->View()->assign($responseData);
    }
}
