<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\RestApi\Idempotency;

/**
 * This trait can be used in API controllers to provide an easy way of implementing idempotent actions.
 */
trait ControllerActionIdempotency
{
    /**
     * Executes the given atomic `$phases` within an idempotency context of the current request and sets the response
     * based on the execution result.
     *
     * @param array $phases An associative array, whose keys are the names of recovery points and the respective values
     *                      are callables that should be executed for each of this points.
     */
    private function executeIdempotently(array $phases)
    {
        // Use the current request to execute the phases idempotently
        $idempotencyService = $this->get('pickware.erp.rest_api_idempotency_service');
        $idempotentOperation = $idempotencyService->createIdempotentOperation($this->Request());
        $idempotencyService->executeIdempotently($idempotentOperation, $phases);

        // Update response and view with the execution result
        $responseCode = ($idempotentOperation->getResponseCode() !== null) ? $idempotentOperation->getResponseCode() : 500;
        $this->Response()->setHttpResponseCode($responseCode);
        $responseData = ($idempotentOperation->getDecodedResponseData()) ?: [];
        $this->View()->assign($responseData);
    }
}
