<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\RestApi\Idempotency;

use Doctrine\DBAL\Connection as DbalConnection;
use Enlight_Controller_Request_RequestHttp as Request;
use Exception;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\RestApi\RestApiIdempotentOperation;
use Shopware\Plugins\ViisonPickwareERP\Components\RestApi\RequestLogging\RestApiRequestLoggerService;

final class RestApiIdempotencyService
{
    /**
     * @var ModelManager
     */
    protected $entityManager;

    /**
     * @var DbalConnection
     */
    protected $dbalConnection;

    /**
     * @var RestApiRequestLoggerService
     */
    protected $restApiRequestLoggerService;

    /**
     * @param ModelManager $entityManager
     * @param DbalConnection $dbalConnection
     * @param RestApiRequestLoggerService $restApiRequestLoggerService
     */
    public function __construct(
        ModelManager $entityManager,
        DbalConnection $dbalConnection,
        RestApiRequestLoggerService $restApiRequestLoggerService
    ) {
        $this->entityManager = $entityManager;
        $this->dbalConnection = $dbalConnection;
        $this->restApiRequestLoggerService = $restApiRequestLoggerService;
    }

    /**
     * Creates and returns a new idempotent operation for the given `$request`.
     *
     * @param Request $request
     * @param string|null $idempotencyKey Either a string that will be used as idempotencyKeyString or null to
     *        retrieve the key automatically from the `Pickware-Idempotency-Key` HTTP header.
     * @return RestApiIdempotentOperation
     * @throws IdempotencyException if the given `$request` does not contain a valid `Pickware-Idempotency-Key` header.
     */
    public function createIdempotentOperation(Request $request, $idempotencyKey = null)
    {
        if ($idempotencyKey === null) {
            $idempotencyKey = $request->getHeader('Pickware-Idempotency-Key');
            if (empty($idempotencyKey)) {
                throw IdempotencyException::invalidOrMissingIdempotencyKeyHeader();
            }
        }

        /** @var RestApiIdempotentOperation $idempotentOperation */
        $idempotentOperation = $this->entityManager->getRepository(RestApiIdempotentOperation::class)->findOneBy([
            'idempotencyKey' => $idempotencyKey,
        ]);
        if (!$idempotentOperation) {
            // Create a new idempotent operation
            $idempotentOperation = new RestApiIdempotentOperation(
                $idempotencyKey,
                $request
            );
            $this->entityManager->persist($idempotentOperation);

            // Use an atomic phase to flush the idempotent operation within a transaction
            $this->executeAtomicPhase($idempotentOperation, function () use ($idempotentOperation) {
                $this->entityManager->flush($idempotentOperation);
            });
        } elseif (!$idempotentOperation->originRequestHasSameRawBodyAsRequest($request)) {
            throw IdempotencyException::invalidRetryRequest();
        }

        // Associate the idempotent operation with the current api log entry, if available
        $apiLogEntry = $this->restApiRequestLoggerService->getCurrentLogEntry($request);
        if ($apiLogEntry) {
            $apiLogEntry->setIdempotentOperation($idempotentOperation);
            $this->entityManager->flush($apiLogEntry);
        }

        return $idempotentOperation;
    }

    /**
     * Executes the given `$executionPhases` within the context of the given `$idempotentOperation`.
     *
     * @param RestApiIdempotentOperation $idempotentOperation
     * @param array $executionPhases
     * @throws IdempotencyException if reaching a non-finished recvery point without respective execution phase.
     */
    public function executeIdempotently(RestApiIdempotentOperation $idempotentOperation, array $executionPhases)
    {
        while (true) {
            $recoveryPoint = $idempotentOperation->getRecoveryPoint();
            if ($recoveryPoint === RestApiIdempotentOperation::RECOVERY_POINT_FINISHED) {
                // Operation has finished, hence break the execution loop
                break;
            }

            if (isset($executionPhases[$recoveryPoint]) && is_callable($executionPhases[$recoveryPoint])) {
                $this->executeAtomicPhase($idempotentOperation, $executionPhases[$recoveryPoint]);
            } else {
                throw IdempotencyException::invalidRecoveryPoint($idempotentOperation, $recoveryPoint);
            }
        }
    }

    /**
     * Executes the given `$phase` within a Doctrine transaction, including the finalization of the execution result
     * (i.e. updating the idempotent operation).
     *
     * @param RestApiIdempotentOperation $idempotentOperation
     * @param callable $phase
     * @throws Exception if executing the `$phase` or handling the transaction throws an exception.
     */
    protected function executeAtomicPhase(RestApiIdempotentOperation $idempotentOperation, callable $phase)
    {
        // The transaction must use the Read Committed transaction isolation level to ensure that the stock ledger
        // service always reads the latest committed stock ledger entry for the article detail during its critical
        // section instead of a stale snapshot that was created earlier in the transaction.
        $this->dbalConnection->exec('SET TRANSACTION ISOLATION LEVEL READ COMMITTED;');
        $this->dbalConnection->beginTransaction();
        try {
            $result = $phase();
            $innerTransactionWasAborted = $this->dbalConnection->isTransactionActive() && $this->dbalConnection->isRollbackOnly();
            if ($innerTransactionWasAborted) {
                // If an inner transaction (e.g. a stock ledger write) was aborted, DBAL's pseudo-nested transaction
                // support will disallow committing the outer transaction. Because of this, it must be rolled back here.
                // If this happens, any changes made to $idempotentOperation as part of calling $result->finalize() will
                // be written outside the transaction context instead of within the aborted outer transaction.
                $this->entityManager->rollback();
            }
            if ($result instanceof AtomicPhaseResult) {
                $result->finalize($idempotentOperation);

                // Flush the key to save any changes made to it during the finalization
                $this->entityManager->flush($idempotentOperation);
            }
            if (!$innerTransactionWasAborted) {
                // If $phase did not result in an aborted transaction, commit the changes it made and any changes to
                // $idempotentOperation made during finalize() as part of a single transaction.
                $this->dbalConnection->commit();
            }
        } catch (Exception $e) {
            $this->dbalConnection->rollback();

            throw $e;
        }
    }
}
