<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\RestApi\RequestLogging;

use Shopware\CustomModels\ViisonPickwareERP\RestApi\RestApiRequest;
use Shopware_Components_Auth;
use Enlight_Components_Db_Adapter_Pdo_Mysql;
use Shopware\Components\Model\ModelManager;
use Enlight_Event_EventManager;

final class RestApiRequestLoggerService
{
    /**
     * A key used for saving the ID of the created log entry in a request object.
     */
    const LOG_RECORD_ID = 'Shopware_Plugins_ViisonPickwareERP_RestApiRequestLogger_LogRecordId';

    /**
     * A key used for saving the time of the log entry creation in a request object.
     */
    const REQUEST_LOG_TIMESTAMP = 'Shopware_Plugins_ViisonPickwareERP_RestApiRequestLogger_RequestLogTimestamp';

    /**
     * A key used for overwriting the max depth of the logged response JSON.
     */
    const MAX_RESPONSE_JSON_DEPTH = 'Shopware_Plugins_ViisonPickwareERP_RestApiRequestLogger_MaxResponseJsonDepth';

    /**
     * The name of a 'filter' event, which is fired before logging a request.
     */
    const EVENT_FILTER_SHOULD_LOG_REQUEST = 'Shopware_Plugins_ViisonPickwareERP_RestApiRequestLogger_FilterShouldLogRequest';

    /**
     * The name of a 'filter' event, which is fired before logging a response JSON.
     */
    const EVENT_FILTER_LOGGED_RESPONSE_JSON = 'Shopware_Plugins_ViisonPickwareERP_RestApiRequestLogger_FilterLoggedResponseJson';

    /**
     * The number of requests between two automatic log clean ups.
     */
    const CLEAN_UP_INTERVAL = 1000;

    /**
     * @var Shopware_Components_Auth
     */
    private $auth;

    /**
     * @var Enlight_Components_Db_Adapter_Pdo_Mysql
     */
    private $db;

    /**
     * Note: Don't access this property directly. Use {@link self::getEntityManager()} instead!
     *
     * @var ModelManager
     */
    private $entityManager;

    /**
     * @var Enlight_Event_EventManager
     */
    private $eventManager;

    /**
     * @param Shopware_Components_Auth $auth
     * @param Enlight_Components_Db_Adapter_Pdo_Mysql $db
     * @param ModelManager $entityManager
     * @param Enlight_Event_EventManager $eventManager
     */
    public function __construct(
        Shopware_Components_Auth $auth,
        Enlight_Components_Db_Adapter_Pdo_Mysql $db,
        ModelManager $entityManager,
        Enlight_Event_EventManager $eventManager
    ) {
        $this->auth = $auth;
        $this->db = $db;
        $this->entityManager = $entityManager;
        $this->eventManager = $eventManager;
    }

    /**
     * @param \Enlight_Controller_Request_RequestHttp $request
     * @return RestApiRequest|null
     */
    public function getCurrentLogEntry(\Enlight_Controller_Request_RequestHttp $request)
    {
        $logEntryId = $request->getAttribute(static::LOG_RECORD_ID, 0);
        /** @var RestApiRequest $logEntry */
        $logEntry = $this->getEntityManager()->find(RestApiRequest::class, $logEntryId);

        return $logEntry;
    }

    /**
     * Creates a new log entity containing the data of the given request plus the request
     * headers and IP address of the requesting machine. Finally the created log entry is
     * added to the request attributes for later use.
     *
     * @param \Enlight_Controller_Request_RequestHttp $request
     */
    public function logRequest(\Enlight_Controller_Request_RequestHttp $request)
    {
        // Allow third parties to whitelist which request are logged. That is, by default no requests are logged.
        $shouldLogRequest = $this->eventManager->filter(
            self::EVENT_FILTER_SHOULD_LOG_REQUEST,
            false,
            [
                'request' => $request,
            ]
        );
        if (!$shouldLogRequest) {
            return;
        }

        // Make sure to not log any requests twice
        if ($request->getAttribute(static::LOG_RECORD_ID)) {
            return;
        }

        // Get a working entity manager
        $entityManager = $this->getEntityManager();

        // Get request headers
        $censoredHeaders = [
            'authorization',
            'php-auth-user',
            'php-auth-pw',
        ];
        $headers = [];
        $httpRequest = \Symfony\Component\HttpFoundation\Request::createFromGlobals();
        foreach ($httpRequest->headers->all() as $name => $values) {
            $value = (in_array(mb_strtolower($name), $censoredHeaders)) ? '<HIDDEN FIELD>' : implode('; ', $values);
            $headers[] = $name . ': ' . $value;
        }

        // Create log entry containing the request information
        $logEntry = new RestApiRequest();
        $logEntry->setMethod($httpRequest->getMethod());
        $logEntry->setUrl($httpRequest->getUri());
        $logEntry->setRequestHeaders(implode("\n", $headers));
        $logEntry->setRequestData(($request->getRawBody()) ?: null);
        $logEntry->setIpAddress(getenv('REMOTE_ADDR'));
        $logEntry->setUserAgent(($_SERVER['HTTP_USER_AGENT']) ?: 'Unknown');

        $entityManager->persist($logEntry);
        $entityManager->flush($logEntry);

        // Save the log entry ID in the requests's attributes
        $request->setAttribute(static::LOG_RECORD_ID, $logEntry->getId());

        // Save the current timestamp in the request
        $request->setAttribute(static::REQUEST_LOG_TIMESTAMP, microtime(true));
    }

    /**
     * Adds information about the given response to the log that is being handles by this logger
     * and saves the changes in the database.
     *
     * @param \Enlight_Controller_Request_RequestHttp $request
     * @param \Enlight_Controller_Response_ResponseHttp $response
     */
    public function logResponse(\Enlight_Controller_Request_RequestHttp $request, \Enlight_Controller_Response_ResponseHttp $response)
    {
        // Get a working entity manager
        $entityManager = $this->getEntityManager();

        // Try to calculate the computation time of the request
        $requestLogTimestamp = $request->getAttribute(static::REQUEST_LOG_TIMESTAMP);
        if ($requestLogTimestamp !== null) {
            $computationTime = microtime(true) - $requestLogTimestamp;
        }

        // Try to get log entry
        $logEntry = $this->getCurrentLogEntry($request);
        if (!$logEntry) {
            return;
        }

        // Get headers
        $headers = [];
        $isJsonResponse = false;
        $isFileDownload = false;
        foreach ($response->getHeaders() as $header) {
            $headers[] = $header['name'] . ': ' . $header['value'];
            if (mb_strtolower($header['name']) === 'content-type') {
                $isJsonResponse = mb_strtolower($header['value']) === 'application/json';
            } elseif (mb_strtolower($header['name']) === 'content-disposition') {
                $isFileDownload = true;
            }
        }
        $logEntry->setResponseHeaders(implode("\n", $headers));
        $logEntry->setResponseCode($response->getHttpResponseCode());

        if (!$isFileDownload) {
            // Get the response body and truncate it, if it is too long
            $responseData = $response->getBody();
            if ($responseData && mb_strlen($responseData) > 10240) {
                $maxJsonDepth = $request->getAttribute(static::MAX_RESPONSE_JSON_DEPTH, 3);
                $responseData = ($isJsonResponse) ? $this->truncateJSONString($responseData, $maxJsonDepth, $request) : (mb_substr($responseData, 0, 10240) . '[...]<TRUNCATED>');
            }
            $logEntry->setResponseData($responseData);
        } else {
            $logEntry->setResponseData('<FILE_DOWNLOAD>');
        }

        // Save requesting user
        $logEntry->setUser($this->auth->getIdentity()->name);

        // Check for exception
        $exception = array_pop($response->getException());
        $logEntry->setException(($exception) ? static::formatException($exception) : null);

        if (isset($computationTime)) {
            $logEntry->setComputationTime($computationTime);
        }

        // Save log entry in database
        $entityManager->flush($logEntry);

        // Clean up the log, if necessary
        if ($logEntry->getId() % self::CLEAN_UP_INTERVAL === 0) {
            $this->truncateLog();
        }
    }

    /**
     * Deletes all log entries from the database, which are older than 10 days, as well as all idempotent operations that are
     * not associated to any remaining log entry.
     */
    public function truncateLog()
    {
        $now = new \DateTime();
        $this->db->query(
            'DELETE FROM pickware_erp_rest_api_requests
            WHERE DATEDIFF(?, date) > 10',
            [
                $now->format('Y-m-d')
            ]
        );
        $this->db->query(
            'DELETE operations
            FROM pickware_erp_rest_api_idempotent_operations operations
            LEFT JOIN pickware_erp_rest_api_requests requests
                ON requests.idempotentOperationId = operations.id
            WHERE requests.id IS NULL'
        );
    }

    /**
     * Returns the extimated size of the 'pickware_erp_rest_api_requests' table on disk.
     *
     * @return string
     */
    public function getLogSize()
    {
        $logSize = $this->db->fetchOne(
            'SELECT round(((data_length + index_length) / 1024 / 1024), 2) AS size
            FROM information_schema.TABLES
            WHERE table_schema = DATABASE()
            AND table_name = "pickware_erp_rest_api_requests"'
        );

        return $logSize;
    }

    /**
     * Checks whether this instance's `entityManager` is still open and, if it is not, replaces it with a new entity
     * manager that uses the same database connection and configuration as the original one, before returning it. This
     * ensures that we have a functional entity manager we can use for logging the request, even if the globally used
     * entity manager was closed due to an error somewhere else in the execution path.
     *
     * Note: You should call this method at the beginning of a method and use the returned entity manager after that
     * within the same method.
     *
     * @return ModelManager
     */
    private function getEntityManager()
    {
        if (!$this->entityManager->isOpen()) {
            // The original entity manager was closed, hence create a new instance
            $this->entityManager = ModelManager::create(
                $this->entityManager->getConnection(),
                $this->entityManager->getConfiguration()
            );
        }

        return $this->entityManager;
    }

    /**
     * Decodes the given jsonString, truncates it to the given maxDepth and encodes
     * it as JSON again. That is, the returned value is a valid JSON, whose nesting
     * has a depth smaller or equal the given maxDepth.
     *
     * @param string $jsonString
     * @param int $maxDepth
     * @param \Enlight_Controller_Request_RequestHttp $request
     * @return string
     */
    private function truncateJSONString($jsonString, $maxDepth, \Enlight_Controller_Request_RequestHttp $request)
    {
        // Try to decode the JSON
        $json = json_decode($jsonString, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            // Invalid JSON, hence cannot truncate
            return $jsonString;
        }

        // Truncate the decoded JSON data to the given maxDepth
        $truncatedJson = static::truncateRecursively($json, $maxDepth);

        // Allow third parties to filter the response JSON
        $filteredJson = $this->eventManager->filter(
            static::EVENT_FILTER_LOGGED_RESPONSE_JSON,
            $truncatedJson,
            [
                'request' => $request,
            ]
        );

        return json_encode($filteredJson);
    }

    /**
     * Recursively checks the given value to truncate it to the given maxDeptch. That is,
     * if the given value is an array and the maxDepth is 0, this function returns the string
     * '<TRUNCATED>'. If however the maxDepth is greater 0, this method is called recursively
     * in the array's values using a maxDepth reduced by 1. If the value is anything else than
     * an array, it is just returned.
     *
     * @param mixed $value
     * @param int $maxDepth
     * @return mixed
     */
    private static function truncateRecursively($value, $maxDepth)
    {
        if (!is_array($value)) {
            // No truncation necessary
            return $value;
        }

        if ($maxDepth > 0) {
            // Value is an array, but max depth is not reached yet, hence truncate the
            // array's values
            foreach ($value as $key => $nestedValue) {
                $value[$key] = static::truncateRecursively($nestedValue, $maxDepth - 1);
            }

            return $value;
        }

        // Max depth is reached, hence truncate
        return '<TRUNCATED>';
    }

    /**
     * @param \Exception|\Throwable $exception
     * @return string
     */
    private static function formatException($exception)
    {
        return json_encode([
            'message' => $exception->getMessage(),
            'code' => $exception->getCode(),
            'file' => $exception->getFile(),
            'line' => $exception->getLine(),
            'trace' => $exception->getTrace(),
            'cause' => ($exception->getPrevious()) ? static::formatException($exception->getPrevious()) : null,
        ]);
    }
}
