<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\RestApi;

use Enlight_Hook;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\WarehouseArticleDetailConfiguration;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\WarehouseRepository;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Plugins\ViisonPickwareERP\Components\DerivedPropertyUpdater\DerivedPropertyUpdater;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockLedgerService;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\StockChangeListFactoryService;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\PositiveStockChangeList;

class RestApiArticleDetailUpdaterService implements RestApiArticleDetailUpdater, Enlight_Hook
{
    /**
     * Comment used for stock ledger entries created by self::updateStockWithPostData()
     */
    const STOCK_LEDGER_ENTRY_COMMENT = '"inStock" change via Shopware REST API';

    /**
     * @var ModelManager
     */
    protected $entityManager;

    /**
     * @var StockLedgerService
     */
    protected $stockLedgerService;

    /**
     * @var StockChangeListFactoryService
     */
    protected $stockChangeListFactoryService;

    /**
     * @var DerivedPropertyUpdater
     */
    protected $derivedPropertyUpdater;

    /**
     * @param ModelManager $entityManager
     * @param StockLedgerService $stockLedgerService
     * @param StockChangeListFactoryService $stockChangeListFactoryService
     * @param DerivedPropertyUpdater $derivedPropertyUpdater
     */
    public function __construct(
        $entityManager,
        StockLedgerService $stockLedgerService,
        StockChangeListFactoryService $stockChangeListFactoryService,
        DerivedPropertyUpdater $derivedPropertyUpdater
    ) {
        $this->entityManager = $entityManager;
        $this->stockLedgerService = $stockLedgerService;
        $this->stockChangeListFactoryService = $stockChangeListFactoryService;
        $this->derivedPropertyUpdater = $derivedPropertyUpdater;
    }

    /**
     * @inheritdoc
     */
    public function removeCustomAttributesFromPostData(array $postData)
    {
        unset($postData['attribute']['pickwareIncomingStock']);
        unset($postData['attribute']['pickwarePhysicalStockForSale']);
        unset($postData['attribute']['pickwareStockManagementDisabled']);

        return $postData;
    }

    /**
     * @inheritdoc
     */
    public function removeStockFieldsFromPostData(array $postData)
    {
        foreach ($postData as $key => $value) {
            if (mb_strtolower($key) === 'instock' || mb_strtolower($key) === 'stockmin') {
                unset($postData[$key]);
            }
        }

        return $postData;
    }

    /**
     * @inheritdoc
     */
    public function updateStockManagementStatusWithPostData(ArticleDetail $articleDetail, array $postData)
    {
        if (isset($postData['attribute']['pickwareStockManagementDisabled'])) {
            if ($postData['attribute']['pickwareStockManagementDisabled']) {
                $this->stockLedgerService->stopRecordingStockChangesForArticleDetail($articleDetail);
            } else {
                $this->stockLedgerService->startRecordingStockChangesForArticleDetail($articleDetail);
            }
        }
    }

    /**
     * @inheritdoc
     */
    public function updateStockLimitsWithPostData(ArticleDetail $articleDetail, array $postData)
    {
        $postData = array_change_key_case($postData, CASE_LOWER);
        if (isset($postData['stockmin'])) {
            /** @var WarehouseArticleDetailConfiguration $warehouseConfig */
            $warehouseConfig = $this->entityManager->getRepository(WarehouseArticleDetailConfiguration::class)->findOneBy([
                'warehouse' => $this->entityManager->getRepository(Warehouse::class)->getDefaultWarehouse(),
                'articleDetail' => $articleDetail,
            ]);

            $warehouseConfig->setTargetStock($postData['stockmin']);
            $warehouseConfig->setMinimumStock($postData['stockmin']);
            $this->entityManager->flush($warehouseConfig);
        }

        $this->derivedPropertyUpdater->recalculateMinimumOnlineStockForArticleDetail($articleDetail);
    }

    /**
     * @inheritdoc
     */
    public function updateStockWithPostData(ArticleDetail $articleDetail, array $postData)
    {
        // Since the API is case insensitive, convert all variantData keys to lowercase
        // before checking for an 'instock' change
        $postData = array_change_key_case($postData, CASE_LOWER);
        if (isset($postData['instock'])) {
            $changeAmount = (int) $postData['instock'] - $articleDetail->getInStock();
            if ($changeAmount === 0) {
                return;
            }

            // Log the stock change to the default warehouse's designated bin location(s)
            /** @var WarehouseRepository $warehouseRepository */
            $warehouseRepository = $this->entityManager->getRepository(Warehouse::class);
            $defaultWarehouse = $warehouseRepository->getDefaultWarehouse();
            $stockChangeList = $this->stockChangeListFactoryService->createStockChangeList(
                $defaultWarehouse,
                $articleDetail,
                $changeAmount
            );
            if ($stockChangeList instanceof PositiveStockChangeList) {
                $this->stockLedgerService->recordIncomingStock(
                    $articleDetail,
                    $stockChangeList,
                    null,
                    self::STOCK_LEDGER_ENTRY_COMMENT
                );
            } else {
                $this->stockLedgerService->recordOutgoingStock(
                    $articleDetail,
                    $stockChangeList,
                    self::STOCK_LEDGER_ENTRY_COMMENT
                );
            }
        }
    }
}
