<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\ReturnShipment;

use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipment;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentItem;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\LocalizableThrowable;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\MessageLocalization;

class ReturnShipmentException extends \Exception implements LocalizableThrowable
{
    use MessageLocalization;

    /**
     * @param string $message
     * @param string $snippetName
     * @param array $snippetArguments
     */
    public function __construct($message, $snippetName, array $snippetArguments = [])
    {
        $this->snippetNamespace = 'viison_pickware_erp/components/return_shipment/return_shipments_exception';
        $this->snippetName = $snippetName;
        $this->snippetArguments = $snippetArguments;

        parent::__construct(vsprintf($message, $snippetArguments));
    }

    /**
     * @param int $newReturned
     * @param int $newWrittenOff
     * @param ReturnShipmentItem $returnShipmentItem
     * @return ReturnShipmentException
     */
    public static function returnedQuantityOrWrittenOffQuantityNotAllowed($newReturned, $newWrittenOff, ReturnShipmentItem $returnShipmentItem)
    {
        $msg = 'The combination of (%d, %d) for returned quantity and written off quantity is not allowed for ' .
            'the return shipment position (item number: "%s", order number: "%s"). Both values must be greater ' .
            'than 0, written off quantity must not be greater than returned quantity and returned quantity must ' .
            'not be greater than the number of items currently sent out to the customer.';

        return new self(
            $msg,
            'returned_quantity_or_written_off_quantity_not_allowed',
            [
                $newReturned,
                $newWrittenOff,
                $returnShipmentItem->getOrderDetail()->getArticleNumber(),
                $returnShipmentItem->getOrderDetail()->getOrder()->getNumber(),
            ]
        );
    }

    /**
     * @param ReturnShipment $returnShipment
     * @return ReturnShipmentException
     */
    public static function targetWarehouseImmutable(ReturnShipment $returnShipment)
    {
        $message = 'The target warehouse of return shipment %d cannot be changed. The target warehouse of a return ' .
            'shipment can only be changes as long as there were no items returned yet.';

        return new self(
            $message,
            'target_warehouse_immutable',
            [
                $returnShipment->getNumber(),
            ]
        );
    }
}
