<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\ReturnShipment;

use Shopware\Models\Order\Order;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\LocalizableThrowable;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\MessageLocalization;

class ReturnShipmentMailingException extends \Exception implements LocalizableThrowable
{
    use MessageLocalization;

    /**
     * @param string $message
     * @param string $snippetName
     * @param array $snippetArguments
     * @param \Exception|null $previousException
     */
    public function __construct($message, $snippetName, array $snippetArguments = [], \Exception $previousException = null)
    {
        parent::__construct($message, 0, $previousException);
        $this->snippetNamespace = 'viison_pickware_erp/components/return_shipment/return_shipments_mailing_exception';
        $this->snippetName = $snippetName;
        $this->snippetArguments = $snippetArguments;
        $this->previousException = $previousException;
    }

    /**
     * @param Order $order
     * @return ReturnShipmentMailingException
     */
    public static function customerForOrderMissing(Order $order)
    {
        $message = sprintf(
            'There is no customer assigned to the order %s. Therefore the email could not be created.',
            $order->getNumber()
        );

        return new self(
            $message,
            'customer_for_order_missing',
            [$order->getNumber()]
        );
    }

    /**
     * @param \Exception $previousException
     * @return ReturnShipmentMailingException
     */
    public static function mailingFailed(\Exception $previousException)
    {
        if ($previousException instanceof \Zend_Mail_Transport_Exception) {
            $snippetName = 'mailing_failed/transport_exception';
            $message = 'The mail could not be sent because of an error. Please check your mail settings in Shopware ' .
                'and PHP. The following error occurred: %s';
        } elseif ($previousException instanceof \SmartyException || $previousException->getPrevious() instanceof \SmartyException) {
            $snippetName = 'mailing_failed/smarty_exception';
            $message = 'The mail could not be created because there was a problem with the mail template. Please ' .
                'check the corresponding mail template. The following error occurred: %s';
        } else {
            $snippetName = 'mailing_failed/exception';
            $message = 'The mail could not be sent or created because the following unknown error occurred: %s';
        }

        return new self(
            sprintf($message, $previousException->getMessage()),
            $snippetName,
            // No snippet arguments because a translated message of $previousException will be added automatically as an
            // element to $snippetArguments by trait MessageLocalization.
            [],
            $previousException
        );
    }
}
