<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\ReturnShipment;

use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentItem;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipment;
use Shopware\Models\Customer\Customer;
use Shopware\Plugins\ViisonCommon\Classes\Util\Document as DocumentUtil;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;

class ReturnShipmentMailingService implements ReturnShipmentMailing, \Enlight_Hook
{
    const RETURN_RECEIVED_NOTICE_EMAIL_TEMPLATE = 'PickwareReturnReceivedNotice';
    const RETURN_COMPLETED_NOTICE_EMAIL_TEMPLATE = 'PickwareReturnCompletedNotice';

    /**
     * @var ModelManager
     */
    protected $entityManager;

    /**
     * @var \Shopware_Components_TemplateMail
     */
    protected $templateMailService;

    /**
     * @param ModelManager $entityManager
     * @param \Shopware_Components_TemplateMail $templateMailService
     */
    public function __construct($entityManager, $templateMailService)
    {
        $this->entityManager = $entityManager;
        $this->templateMailService = $templateMailService;
    }

    /**
     * @inheritdoc
     */
    public function sendReturnReceivedEmail(ReturnShipment $returnShipment)
    {
        $mail = $this->createReturnReceivedMail($returnShipment);
        try {
            $mail->send();
        } catch (\Exception $e) {
            throw ReturnShipmentMailingException::mailingFailed($e);
        }
    }

    /**
     * @inheritdoc
     */
    public function sendReturnCompletedEmail(ReturnShipment $returnShipment)
    {
        $mail = $this->createReturnCompletedMail($returnShipment);
        try {
            $mail->send();
        } catch (\Exception $e) {
            throw ReturnShipmentMailingException::mailingFailed($e);
        }
    }

    /**
     * @inheritdoc
     */
    public function createReturnReceivedMail(ReturnShipment $returnShipment)
    {
        return $this->createMail($returnShipment, self::RETURN_RECEIVED_NOTICE_EMAIL_TEMPLATE);
    }

    /**
     * @inheritdoc
     */
    public function createReturnCompletedMail(ReturnShipment $returnShipment)
    {
        return $this->createMail($returnShipment, self::RETURN_COMPLETED_NOTICE_EMAIL_TEMPLATE);
    }

    /**
     * @param ReturnShipment $returnShipment
     * @param $mailTemplateName
     * @return \Enlight_Components_Mail
     * @throws ReturnShipmentMailingException
     */
    private function createMail(ReturnShipment $returnShipment, $mailTemplateName)
    {
        // Because of a missing foreign key constraint and the lazy loading of Doctrine it may be possible that
        // $returnShipment->getOrder()->getCustomer() returns a valid customer object that does not have a corresponding row
        // in the database. The findOneBy method will always look up the entity in the database.
        $customer = $this->entityManager->getRepository(Customer::class)->findOneBy([
            'id' => $returnShipment->getOrder()->getCustomer()->getId(),
        ]);
        if (!$customer) {
            throw ReturnShipmentMailingException::customerForOrderMissing($returnShipment->getOrder());
        }

        // Send return notice email
        $itemsForNotification = array_map(function (ReturnShipmentItem $item) {
            $orderDetail = $item->getOrderDetail();
            $quantity = $item->getReturnedQuantity();

            return [
                'quantity' => $quantity,
                'articleNumber' => $orderDetail->getArticleNumber(),
                'articleName' => $orderDetail->getArticleName(),
            ];
        }, $returnShipment->getItems()->toArray());

        $context = [
            'userSalutation' => $customer->getSalutation(),
            'userFirstName' => $customer->getFirstname(),
            'userLastName' => $customer->getLastname(),
            'orderNumber' => $returnShipment->getOrder()->getNumber(),
            'returnItems' => $itemsForNotification,
            'returnShipmentNumber' => $returnShipment->getNumber(),
        ];

        $document = $returnShipment->getDocument();
        if ($document) {
            $context['invoiceAmount'] = $document->getAmount();
        }

        // Get config of subshop where order was placed since Shopware always uses the data of the main shop if not set in 'overrideConfig', no matter which shop is passed
        $scopeConfig = ViisonCommonUtil::getShopConfig($returnShipment->getOrder()->getShop());

        try {
            // Create the mail template
            $mail = $this->templateMailService->createMail(
                $mailTemplateName,
                $context,
                $returnShipment->getOrder()->getShop(),
                $scopeConfig
            );
            $mail->addTo($customer->getEmail());

            if ($document) {
                DocumentUtil::attachDocumentToMail($document, $mail);
            }

            return $mail;
        } catch (\Exception $e) {
            throw ReturnShipmentMailingException::mailingFailed($e);
        }
    }
}
