<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\ReturnShipment;

use Shopware\Components\Model\ModelManager;
use Shopware\Components\NumberRangeIncrementerInterface;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipment;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentItem;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentRepository;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentStatus;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\Models\Order\Order;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\NumberRangeNotFoundException;
use Shopware\Plugins\ViisonCommon\Classes\Util\OrderDetailUtil;
use Shopware\Plugins\ViisonPickwareERP\Components\Cancellation\OrderCanceler;
use Shopware\Plugins\ViisonPickwareERP\Components\OrderDetailQuantityCalculator\OrderDetailQuantityCalculator;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\StockChangeListFactory;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockUpdater;

class ReturnShipmentProcessorService implements ReturnShipmentProcessor, \Enlight_Hook
{
    const NUMBER_RANGE_NAME = 'returns';

    /**
     * @var ModelManager
     */
    protected $entityManager;

    /**
     * @var StockUpdater
     */
    protected $stockUpdater;

    /**
     * @var StockChangeListFactory
     */
    protected $stockChangeListFactory;

    /**
     * @var NumberRangeIncrementerInterface
     */
    private $numberRangeIncrementer;

    /**
     * @var OrderCanceler
     */
    private $orderCanceler;

    /**
     * @var OrderDetailQuantityCalculator
     */
    private $orderDetailQuantityCalculator;

    /**
     * @param ModelManager $entityManager
     * @param StockUpdater $stockUpdater
     * @param StockChangeListFactory $stockChangeListFactory
     * @param NumberRangeIncrementerInterface $numberRangeIncrementer
     * @param OrderCanceler $orderCanceler
     * @param OrderDetailQuantityCalculator $orderDetailQuantityCalculator
     */
    public function __construct(
        $entityManager,
        StockUpdater $stockUpdater,
        StockChangeListFactory $stockChangeListFactory,
        NumberRangeIncrementerInterface $numberRangeIncrementer,
        OrderCanceler $orderCanceler,
        OrderDetailQuantityCalculator $orderDetailQuantityCalculator
    ) {
        $this->entityManager = $entityManager;
        $this->stockUpdater = $stockUpdater;
        $this->stockChangeListFactory = $stockChangeListFactory;
        $this->numberRangeIncrementer = $numberRangeIncrementer;
        $this->orderCanceler = $orderCanceler;
        $this->orderDetailQuantityCalculator = $orderDetailQuantityCalculator;
    }

    /**
     * @inheritdoc
     */
    public function createReturnShipment(Order $order)
    {
        try {
            $nextNumber = $this->numberRangeIncrementer->increment(self::NUMBER_RANGE_NAME);

            return new ReturnShipment(
                $order,
                $nextNumber,
                $this->entityManager->getRepository(ReturnShipmentStatus::class)->getInitialReturnShipmentStatus()
            );
        } catch (\RuntimeException $e) {
            throw new NumberRangeNotFoundException(self::NUMBER_RANGE_NAME);
        }
    }

    /**
     * @inheritdoc
     */
    public function isReturnedAndWrittenOffQuantityAllowed(
        ReturnShipmentItem $returnShipmentItem,
        $newReturnedQuantity,
        $newWrittenOffQuantity
    ) {
        if ($newReturnedQuantity < 0 || $newWrittenOffQuantity < 0) {
            return false;
        }

        if ($newWrittenOffQuantity > $newReturnedQuantity) {
            return false;
        }

        return $this->isReturnedQuantityAllowed($returnShipmentItem, $newReturnedQuantity);
    }

    /**
     * @param ReturnShipmentItem $returnShipmentItem
     * @param int $newReturnedQuantity
     * @return bool
     */
    private function isReturnedQuantityAllowed(ReturnShipmentItem $returnShipmentItem, $newReturnedQuantity)
    {
        if ($newReturnedQuantity < $returnShipmentItem->getCancelledQuantity()) {
            return false;
        }

        $returnable = $this->orderDetailQuantityCalculator->calculateShippedAndNotReturnedQuantity($returnShipmentItem->getOrderDetail());
        $maxReturnedQuantity = $returnable + $returnShipmentItem->getReturnedQuantity();

        return $newReturnedQuantity <= $maxReturnedQuantity;
    }

    /**
     * @inheritdoc
     */
    public function writeStockEntriesForItemChangesWithAutomaticBinLocationSelection(ReturnShipmentItem $item)
    {
        $articleDetail = OrderDetailUtil::getArticleDetailForOrderDetail($item->getOrderDetail());
        if (!$articleDetail) {
            return;
        }
        $targetWarehouse = $item->getReturnShipment()->getTargetWarehouse();
        if (!$targetWarehouse) {
            return;
        }

        if ($item->getDeltaReturnedQuantity() !== 0) {
            $stockChanges = $this->stockChangeListFactory->createStockChangeList(
                $targetWarehouse,
                $articleDetail,
                $item->getDeltaReturnedQuantity()
            );
            $this->stockUpdater->recordReturnShipmentItemReturnedChange($articleDetail, $item, $stockChanges);
        }

        if ($item->getDeltaWrittenOffQuantity() !== 0) {
            $stockChanges = $this->stockChangeListFactory->createStockChangeList(
                $targetWarehouse,
                $articleDetail,
                -1 * $item->getDeltaWrittenOffQuantity()
            );
            $this->stockUpdater->recordReturnShipmentItemWriteOffChange($articleDetail, $item, $stockChanges);
        }

        $item->resetDeltas();
    }

    /**
     * @inheritdoc
     */
    public function writeStockEntriesForItemChangesOnBinLocation(ReturnShipmentItem $item, BinLocation $binLocation)
    {
        $articleDetail = OrderDetailUtil::getArticleDetailForOrderDetail($item->getOrderDetail());
        if (!$articleDetail) {
            return;
        }
        $targetWarehouse = $item->getReturnShipment()->getTargetWarehouse();
        if (!$targetWarehouse) {
            return;
        }

        if ($item->getDeltaReturnedQuantity() !== 0) {
            $stockChanges = $this->stockChangeListFactory->createSingleBinLocationStockChangeList(
                $binLocation,
                $item->getDeltaReturnedQuantity()
            );
            $this->stockUpdater->recordReturnShipmentItemReturnedChange($articleDetail, $item, $stockChanges);
        }

        if ($item->getDeltaWrittenOffQuantity() !== 0) {
            $stockChanges = $this->stockChangeListFactory->createSingleBinLocationStockChangeList(
                $binLocation,
                -1 * $item->getDeltaWrittenOffQuantity()
            );
            $this->stockUpdater->recordReturnShipmentItemWriteOffChange($articleDetail, $item, $stockChanges);
        }

        $item->resetDeltas();
    }

    /**
     * @inheritdoc
     */
    public function updateAccumulatedReturnShipmentStatus(Order $order)
    {
        /** @var ReturnShipmentRepository $returnShipmentStatusRepo */
        $returnShipmentStatusRepo = $this->entityManager->getRepository(ReturnShipmentStatus::class);
        $allReturnShipmentStatus = $returnShipmentStatusRepo->findAllReturnShipmentStatusOfOrder($order);

        $minReturnProcessStep = array_reduce(
            $allReturnShipmentStatus,
            function ($carry, ReturnShipmentStatus $returnShipmentStatus) {
                return min($carry, $returnShipmentStatus->getProcessStep());
            },
            PHP_INT_MAX
        );

        $combinedReturnStatus = null;
        if ($minReturnProcessStep !== PHP_INT_MAX) {
            $combinedReturnStatus = $returnShipmentStatusRepo->findOneBy([
                'processStep' => $minReturnProcessStep,
            ]);
        }

        // Save the new status
        $orderAttribute = $order->getAttribute();
        if ($orderAttribute) {
            $orderAttribute->setPickwareReturnShipmentStatusId($combinedReturnStatus ? $combinedReturnStatus->getId() : null);
            $entitiesToFlush[] = $orderAttribute;
            $this->entityManager->flush($orderAttribute);
        }
    }

    /**
     * @inheritdoc
     */
    public function safelyDeleteReturnShipment(ReturnShipment $returnShipment)
    {
        array_map(function (ReturnShipmentItem $returnShipmentItem) {
            $this->safelyDeleteReturnShipmentItem($returnShipmentItem);
        }, $returnShipment->getItems()->toArray());

        $this->entityManager->remove($returnShipment);
        $this->entityManager->flush($returnShipment);
    }

    /**
     * @inheritdoc
     */
    public function safelyDeleteReturnShipmentItem(ReturnShipmentItem $returnShipmentItem)
    {
        // Reverse the cancellation of the returned positions
        $this->orderCanceler->cancelReturnedQuantityOfReturnShipmentItem($returnShipmentItem, -1 * $returnShipmentItem->getCancelledQuantity());

        // Reverse the complete return
        $returnShipmentItem->setCancelledQuantity(0);
        $returnShipmentItem->setReturnedQuantity(0);
        $returnShipmentItem->setWrittenOffQuantity(0);
        $this->entityManager->flush($returnShipmentItem);
        $this->writeStockEntriesForItemChangesWithAutomaticBinLocationSelection($returnShipmentItem);

        // Remove the item from the ReturnShipment and from the database
        $returnShipmentItem->getReturnShipment()->removeItem($returnShipmentItem);
        $this->entityManager->remove($returnShipmentItem);
        $this->entityManager->flush($returnShipmentItem);
    }
}
