<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockLedger;

use Doctrine\DBAL\DBALException;
use Exception;
use PDOException;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\LocalizableThrowable;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\MessageLocalization;

class ArticleDetailConcurrencyException extends Exception implements LocalizableThrowable
{
    use MessageLocalization;

    public function __construct($message, $code = 0, Exception $previousException = null)
    {
        parent::__construct($message, $code, $previousException);
        $this->setSnippetNamespace('services/viison_pickware_erp/article_detail_concurrency_coordinator');
    }

    public static function map(DBALException $dbalException, $articleDetailIds)
    {
        $cause = $dbalException->getPrevious();
        if (!($cause instanceof PDOException)) {
            Shopware()->Container()->get('viison_common.logger')->info('No PDO.');

            return $dbalException;
        }

        if ($cause->getCode() === 'HY000') {
            Shopware()->Container()->get('viison_common.logger')->info('HY000');

            return self::lockWaitTimeoutExceeded($dbalException, $articleDetailIds);
        }

        if (intval($cause->getCode()) === 40001) {
            Shopware()->Container()->get('viison_common.logger')->info('40001');

            return self::serializationFailure($dbalException, $articleDetailIds);
        }

        Shopware()->Container()->get('viison_common.logger')->info('Other exception');

        return $dbalException;
    }

    private static function lockWaitTimeoutExceeded(DBALException $dbalException, $articleDetailIds)
    {
        $articleDetailIdsList = implode(', ', $articleDetailIds);
        $exception = new self(
            sprintf(
                'The lock wait timeout was exceeded while trying to acquire a stock change lock for the variants with ID(s) %s. Please try again.',
                $articleDetailIdsList
            ),
            0,
            $dbalException
        );

        $exception->setSnippetName('exception/lock_wait_timeout_exceeded');
        $exception->setSnippetArguments([$articleDetailIdsList]);

        return $exception;
    }

    private static function serializationFailure(DBALException $dbalException, $articleDetailIds)
    {
        $articleDetailIdsList = implode(', ', $articleDetailIds);
        $exception = new self(
            sprintf(
                'Serialization failure while trying to acquire a stock change lock for the variants with ID(s) %s. Please try again.',
                $articleDetailIdsList
            ),
            0,
            $dbalException
        );

        $exception->setSnippetName('exception/serialization_failure');
        $exception->setSnippetArguments([$articleDetailIdsList]);

        return $exception;
    }
}
