<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockLedger;

use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Article\Detail as ArticleDetail;

interface ArticleDetailStockInfoProvider
{
    /**
     * Returns the current physical stock of the passed `$articleDetail` in the passed `$warehouse` as written in the
     * lastly created stock ledger entry. Returns `0`, if no stock ledger entry exists.
     *
     * @param ArticleDetail $articleDetail
     * @param Warehouse $warehouse
     * @return int
     */
    public function getStockInWarehouse(ArticleDetail $articleDetail, Warehouse $warehouse);

    /**
     * Returns the the sum of all stock of the passed `$articleDetail` in all warehouses.
     *
     * @param ArticleDetail $articleDetail
     * @return int
     */
    public function getTotalPhysicalStock(ArticleDetail $articleDetail);

    /**
     * Returns the the sum of all stock of the passed `$articleDetail` in all warehouses, whose stock is available
     * for sale in the online shop.
     *
     * @param ArticleDetail $articleDetail
     * @return int
     */
    public function getTotalPhysicalStockForSale(ArticleDetail $articleDetail);

    /**
     * Returns the lastest stock ledger entry of the passed `$articleDetail` in the passed `$warehouse`. If the optional
     * `$stockLedgerEntryType` is passed, the latest stock ledger entry of that type is returned.
     *
     * @param ArticleDetail $articleDetail
     * @param Warehouse $warehouse
     * @param string|null $stockLedgerEntryType (optional)
     * @return StockLedgerEntry|null
     */
    public function getLatestStockLedgerEntryForWarehouse(
        ArticleDetail $articleDetail,
        Warehouse $warehouse,
        $stockLedgerEntryType = null
    );

    /**
     * @param ArticleDetail $articleDetail
     * @return boolean `false` iff stock changes of the passed `$articleDetail` are not recored by the stock ledger.
     */
    public function isStockManaged(ArticleDetail $articleDetail);

    /**
     * @param ArticleDetail $articleDetail
     * @return boolean `true` iff the stock of the passed `$articleDetail` has been initialized in the stock ledger.
     */
    public function isStockInitialized(ArticleDetail $articleDetail);
}
