<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockLedger;

use Enlight_Hook;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Article\Detail as ArticleDetail;

class ArticleDetailStockInfoProviderService implements ArticleDetailStockInfoProvider, Enlight_Hook
{
    /**
     * @var ModelManager
     */
    protected $entityManager;

    /**
     * @param ModelManager $entityManager
     */
    public function __construct($entityManager)
    {
        $this->entityManager = $entityManager;
    }

    /**
     * @inheritdoc
     */
    public function getStockInWarehouse(ArticleDetail $articleDetail, Warehouse $warehouse)
    {
        $latestStockLedgerEntry = $this->getLatestStockLedgerEntryForWarehouse($articleDetail, $warehouse);

        return ($latestStockLedgerEntry) ? $latestStockLedgerEntry->getNewStock() : 0;
    }

    /**
     * @inheritdoc
     */
    public function getTotalPhysicalStock(ArticleDetail $articleDetail)
    {
        return $this->getSumOfStocksInWarehouses(
            $articleDetail,
            $this->entityManager->getRepository(Warehouse::class)->findAll()
        );
    }

    /**
     * @inheritdoc
     */
    public function getTotalPhysicalStockForSale(ArticleDetail $articleDetail)
    {
        return $this->getSumOfStocksInWarehouses(
            $articleDetail,
            $this->entityManager->getRepository(Warehouse::class)->findBy(['stockAvailableForSale' => true])
        );
    }

    /**
     * @param ArticleDetail $articleDetail
     * @param Warehouse[] $warehouses
     * @return int
     */
    protected function getSumOfStocksInWarehouses(ArticleDetail $articleDetail, array $warehouses)
    {
        return array_reduce($warehouses, function ($carry, Warehouse $warehouse) use ($articleDetail) {
            return $carry + $this->getStockInWarehouse($articleDetail, $warehouse);
        }, 0);
    }

    /**
     * @inheritdoc
     */
    public function getLatestStockLedgerEntryForWarehouse(
        ArticleDetail $articleDetail,
        Warehouse $warehouse,
        $stockLedgerEntryType = null
    ) {
        $builder = $this->entityManager->createQueryBuilder();
        $builder->select('stockLedgerEntries')
                ->from(StockLedgerEntry::class, 'stockLedgerEntries')
                ->where('stockLedgerEntries.articleDetail = :articleDetail')
                ->andWhere('stockLedgerEntries.warehouse = :warehouse')
                ->orderBy('stockLedgerEntries.id', 'DESC')
                ->setMaxResults(1)
                ->setParameter('articleDetail', $articleDetail)
                ->setParameter('warehouse', $warehouse);
        if ($stockLedgerEntryType !== null) {
            $builder->andWhere('stockLedgerEntries.type = :stockLedgerEntryType')
                    ->setParameter('stockLedgerEntryType', $stockLedgerEntryType);
        }

        return $builder->getQuery()->getOneOrNullResult();
    }

    /**
     * @inheritdoc
     */
    public function isStockManaged(ArticleDetail $articleDetail)
    {
        return !($articleDetail->getAttribute() && $articleDetail->getAttribute()->getPickwareStockManagementDisabled());
    }

    /**
     * @inheritdoc
     */
    public function isStockInitialized(ArticleDetail $articleDetail)
    {
        return $articleDetail->getAttribute() && $articleDetail->getAttribute()->getPickwareStockInitialized();
    }
}
