<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\PurchasePriceAssignment;

use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Article\Detail as ArticleDetail;

/**
 * Base class for all pricing types (classes). Pricing are used to assign
 * purchase prices to a given number of items. A pricing consists of one
 * or more price items. A price item comprises a purchase price together
 * with its applicability (number of items the purchase price can be
 * applied to) and, if it exists, a reference to the related incoming
 * history entry. The price items form a partition in the following sense:
 * the sum of the applicabilities of all price items equal the given number
 * of items, which should be priced using the pricing.
 */
abstract class AbstractPurchasePriceAssignment implements \Iterator
{
    /**
     * @var int
     */
    protected $position = 0;

    /**
     * @var AssignablePurchasePrice[]
     */
    protected $items = [];

    /**
     * @var ModelManager
     */
    protected $entityManager;

    /**
     * @var ArticleDetail
     */
    protected $articleDetail;

    /**
     * @var Warehouse
     */
    protected $warehouse;

    /**
     * @var int
     */
    protected $quantity;

    /**
     * @var float
     */
    protected $defaultPrice;

    /**
     * @param ModelManager $entityManager
     * @param ArticleDetail $articleDetail
     * @param Warehouse $warehouse
     * @param int $quantity
     */
    public function __construct(ModelManager $entityManager, ArticleDetail $articleDetail, Warehouse $warehouse, $quantity)
    {
        if ($quantity < 0) {
            throw new \InvalidArgumentException(sprintf(
                'Parameter "$quantity" of method "%s" must not be negative, %d given',
                __METHOD__,
                $quantity
            ));
        }

        $this->entityManager = $entityManager;
        $this->articleDetail = $articleDetail;
        $this->warehouse = $warehouse;
        $this->quantity = $quantity;
        $this->defaultPrice = $articleDetail->getPurchasePrice();

        // Generate pricing items
        $this->items = $this->generatePricingAssignments();
    }

    /**
     * Returns the default purchase price of the article associated with the instance.
     *
     * @return float
     */
    public function getDefaultPrice()
    {
        return $this->defaultPrice;
    }

    /**
     * Resets the list pointer to its initial state.
     */
    public function rewind()
    {
        $this->position = 0;
    }

    /**
     * Returns the list item, which is referenced by the list pointer.
     *
     * @return AssignablePurchasePrice
     */
    public function current()
    {
        return $this->items[$this->position];
    }

    /**
     * Returns the current position of the list pointer.
     *
     * @return int
     */
    public function key()
    {
        return $this->position;
    }

    /**
     * Moves the list pointer to the next list item.
     */
    public function next()
    {
        $this->position += 1;
    }

    /**
     * Checks if the list pointer points to a list item.
     *
     * @return boolean
     */
    public function valid()
    {
        return isset($this->items[$this->position]);
    }

    /**
     * Generates and returns the pricing assignments.
     *
     * @return AssignablePurchasePrice[]
     */
    abstract protected function generatePricingAssignments();
}
