<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\PurchasePriceAssignment;

use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry;

/**
 * The sale pricing assigns purchase prices to outgoing items based on all incoming entries of a given article.
 */
class OutgoingPurchasePriceAssignment extends AbstractPurchasePriceAssignment
{
    use PricingBucketSelection;

    /**
     * @inheritdoc
     */
    protected function generatePricingAssignments()
    {
        $buckets = array_merge(
            [new AssignablePurchasePrice($this->defaultPrice, PHP_INT_MAX)],
            $this->getUnusedPricingBuckets()
        );

        return self::reverselySelectPricingBuckets($buckets, $this->quantity);
    }

    /**
     * Determines all pricing buckets that are unused, that is not completely used by other stock entries and returns
     * in an array, sorted in ascending order of creation.
     *
     * @return AssignablePurchasePrice[]
     */
    public function getUnusedPricingBuckets()
    {
        // Note: We must use `COALESCE` instead of `IFNULL` for calculating the quantity, because the latter is not
        // available in DQL.
        $query = $this->entityManager->createQuery(
            'SELECT
                stockLedgerEntry AS entry,
                stockLedgerEntry.changeAmount + COALESCE(SUM(lotAssignedEntries.changeAmount), 0) AS quantity
            FROM Shopware\\CustomModels\\ViisonPickwareERP\\StockLedger\\StockLedgerEntry AS stockLedgerEntry
            LEFT JOIN stockLedgerEntry.lotAssignedEntries AS lotAssignedEntries
            WHERE
                stockLedgerEntry.articleDetail = :articleDetail
                AND stockLedgerEntry.warehouse = :warehouse
                AND stockLedgerEntry.type IN (:stockTypes)
                AND stockLedgerEntry.sourceLotEntry IS NULL
                AND stockLedgerEntry.oldStock >= 0
            GROUP BY stockLedgerEntry.id
            HAVING quantity > 0
            ORDER BY
                stockLedgerEntry.created ASC,
                stockLedgerEntry.id ASC'
        );
        $query->setParameters([
            'articleDetail' => $this->articleDetail,
            'warehouse' => $this->warehouse,
            'stockTypes' => [
                StockLedgerEntry::TYPE_PURCHASE,
                StockLedgerEntry::TYPE_STOCKTAKE,
                StockLedgerEntry::TYPE_INITIALIZATION,
                StockLedgerEntry::TYPE_RETURN,
                StockLedgerEntry::TYPE_MANUAL,
                StockLedgerEntry::TYPE_INCOMING,
            ],
        ]);

        return array_map(
            function (array $row) {
                return new AssignablePurchasePrice(
                    $row['entry']->getPurchasePrice(),
                    intval($row['quantity']),
                    $row['entry']
                );
            },
            $query->getResult()
        );
    }
}
