<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\PurchasePriceAssignment;

use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Order\Detail as OrderDetail;

/**
 * The return pricing assigns purchase prices to returned items based on all outgoing entries of a given article and a
 * given order item.
 */
class ReturningPurchasePriceAssignment extends AbstractPurchasePriceAssignment
{
    use PricingBucketSelection;

    /**
     * @var OrderDetail
     */
    protected $orderDetail;

    /**
     * Adds a parameter to pass the order item to the parameter list of the default constructor.
     *
     * @param ModelManager $entityManager
     * @param ArticleDetail $articleDetail
     * @param Warehouse $warehouse
     * @param OrderDetail $orderDetail
     * @param int $quantity
     */
    public function __construct(ModelManager $entityManager, ArticleDetail $articleDetail, Warehouse $warehouse, OrderDetail $orderDetail, $quantity)
    {
        $this->orderDetail = $orderDetail;
        parent::__construct($entityManager, $articleDetail, $warehouse, $quantity);
    }

    /**
     * @inheritdoc
     */
    protected function generatePricingAssignments()
    {
        $buckets = array_merge(
            [new AssignablePurchasePrice($this->defaultPrice, PHP_INT_MAX)],
            $this->getUnusedUnassignedPricingBucktes(),
            $this->getUnusedAssignedPricingBuckets()
        );

        return self::reverselySelectPricingBuckets($buckets, $this->quantity);
    }

    /**
     * Finds all unassigned outgoing entries, that are not fully returned, and returns respective price items. Outgoing
     * entries are grouped by their purchase price.
     *
     * @return AssignablePurchasePrice[]
     */
    private function getUnusedUnassignedPricingBucktes()
    {
        // Find all outgoing entries, which are not fully returned
        $query = $this->entityManager->createQuery(
            'SELECT
                stockLedgerEntry,
                SUM(stockLedgerEntry.changeAmount) AS quantity,
                stockLedgerEntry.purchasePrice AS purchasePrice,
                MAX(stockLedgerEntry.created) AS HIDDEN sortingDate,
                MAX(stockLedgerEntry.id) AS HIDDEN sortingId
            FROM Shopware\\CustomModels\\ViisonPickwareERP\\StockLedger\\StockLedgerEntry AS stockLedgerEntry
            WHERE
                stockLedgerEntry.sourceLotEntry IS NULL
                AND stockLedgerEntry.articleDetail = :articleDetail
                AND stockLedgerEntry.warehouse = :warehouse
                AND stockLedgerEntry.orderDetail = :orderdetail
            GROUP BY stockLedgerEntry.purchasePrice
            HAVING quantity < 0
            ORDER BY
                sortingDate ASC,
                sortingId ASC'
        );
        $query->setParameters([
            'articleDetail' => $this->articleDetail,
            'warehouse' => $this->warehouse,
            'orderdetail' => $this->orderDetail,
        ]);

        return array_map(
            function (array $row) {
                return new AssignablePurchasePrice(floatval($row['purchasePrice']), abs(intval($row['quantity'])));
            },
            $query->getResult()
        );
    }

    /**
     * Finds all assigned outgoing entries, that are not fully returned, and returns respective price items. Outgoing
     * entries are grouped by their source lot entry.
     *
     * @return AssignablePurchasePrice[]
     */
    private function getUnusedAssignedPricingBuckets()
    {
        // Find all outgoing entries, which are not fully returned
        $query = $this->entityManager->createQuery(
            'SELECT
                stockLedgerEntry AS entry,
                SUM(stockLedgerEntry.changeAmount) AS quantity
            FROM Shopware\\CustomModels\\ViisonPickwareERP\\StockLedger\\StockLedgerEntry AS stockLedgerEntry
            LEFT JOIN stockLedgerEntry.sourceLotEntry AS owner
            WHERE
                stockLedgerEntry.sourceLotEntry IS NOT NULL
                AND stockLedgerEntry.articleDetail = :articleDetail
                AND stockLedgerEntry.warehouse = :warehouse
                AND stockLedgerEntry.orderDetail = :orderdetail
            GROUP BY stockLedgerEntry.sourceLotEntry
            HAVING quantity < 0
            ORDER BY
                owner.created ASC,
                owner.id ASC'
        );
        $query->setParameters([
            'articleDetail' => $this->articleDetail,
            'warehouse' => $this->warehouse,
            'orderdetail' => $this->orderDetail,
        ]);

        return array_map(
            function (array $row) {
                return new AssignablePurchasePrice(
                    $row['entry']->getSourceLotEntry()->getPurchasePrice(),
                    abs(intval($row['quantity'])),
                    $row['entry']->getSourceLotEntry()
                );
            },
            $query->getResult()
        );
    }
}
