<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList;

use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;

/**
 * @internal
 */
abstract class AbstractStockChangeList
{
    /**
     * @var BinLocationStockChange[]
     */
    protected $changes = [];

    /**
     * @param BinLocationStockChange[] $changes
     * @throws \InvalidArgumentException
     */
    public function __construct(array $changes)
    {
        // Check and process the passed changes
        $warehouse = null;
        foreach ($changes as $change) {
            if (!($change instanceof BinLocationStockChange)) {
                throw new \InvalidArgumentException('Parameter "changes" must only contain instances of "BinLocationStockChange"');
            }
            if ($warehouse && !$change->getBinLocation()->getWarehouse()->equals($warehouse)) {
                throw new \InvalidArgumentException('The bin location of the passed parameter "changes" must all be part of the same warehouse');
            }
            $warehouse = $change->getBinLocation()->getWarehouse();
            if (!$this->validateBinLocationStockChange($change)) {
                continue;
            }

            // Check for duplicate bin location
            $existingChange = null;
            foreach ($this->changes as $savedChange) {
                if ($savedChange->getBinLocation()->equals($change->getBinLocation())) {
                    $existingChange = $savedChange;
                    break;
                }
            }
            if ($existingChange) {
                // Update existing location
                $existingChange->setStockChange($existingChange->getStockChange() + $change->getStockChange());
            } else {
                // New location, hence just add it to the changes
                $this->changes[] = $change;
            }
        }
    }

    /**
     * @return BinLocationStockChange[]
     */
    public function getBinLocationStockChanges()
    {
        return $this->changes;
    }

    /**
     * @return int
     */
    public function getTotalStockChange()
    {
        return array_reduce($this->changes, function ($carry, $item) {
            return $carry + $item->getStockChange();
        }, 0);
    }

    /**
     * @return BinLocation[]
     */
    public function getChangedLocations()
    {
        return array_map(function ($change) {
            return $change->getBinLocation();
        }, $this->changes);
    }

    /**
     * @return Warehouse
     */
    public function getChangedWarehouse()
    {
        return $this->getChangedLocations()[0]->getWarehouse();
    }

    /**
     * @param BinLocation $binLocation
     * @return BinLocationStockChange|null
     */
    public function getStockChangeForLocation(BinLocation $binLocation)
    {
        foreach ($this->changes as $change) {
            if ($change->getBinLocation()->equals($binLocation)) {
                return $change;
            }
        }

        return null;
    }

    /**
     * Generates a new change list of the same class as this list, by reducing the total stock change
     * of this list by the given $delta. The total stock change of the new list equals the given $delta
     * and contains only as many changed bin locations as necessary to achieve this reduction.
     *
     * @param int $delta
     * @return self
     * @throws \InvalidArgumentException
     */
    public function reduceAbsoluteTotalStockChange($delta)
    {
        // Validate input
        $delta = abs($delta);
        $absTotalChange = abs($this->getTotalStockChange());
        if ($delta > $absTotalChange) {
            throw new \InvalidArgumentException(
                sprintf('The absolute value of parameter "delta" (%d) must not be greater than the current absolute total stock change (%d) of the list', $delta, $absTotalChange)
            );
        } elseif ($delta === 0) {
            // It is valid to reduce this list by zero, but this case must be handled separately. That is,
            // we can create a new change list with a total change of zero by using all changed bin locations
            // of this list, whose stock change is zero as well. Please note that this special case is only
            // valid, if this list contains at least one bin location with a stock change of zero. Otherwise
            // reducing this list by zero results in an exception.
            $zeroChanges = array_filter($this->changes, function ($change) {
                return $change->getStockChange() === 0;
            });
            if (count($zeroChanges) === 0) {
                throw new \InvalidArgumentException(
                    sprintf('You\'re trying to reduce a change list by a "delta" of %d, although the list does not contain any bin locations with a stock change of zero. This is not allowed.', $delta)
                );
            }

            // Create the new change list containing all bin locations with zero stock change
            $class = get_class($this);
            $deltaList = new $class(array_map(function ($change) {
                return new BinLocationStockChange($change->getBinLocation(), 0);
            }, $zeroChanges));

            return $deltaList;
        }

        $deltaStockChanges = [];
        foreach ($this->changes as $change) {
            if ($delta === 0) {
                break;
            }
            // Check for remaining stock change
            if ($change->getStockChange() === 0) {
                continue;
            }

            // Determine the stock change that can be taken from the current location and update
            // both the diff and the current stock change accordingly
            $deltaStockChange = min(abs($change->getStockChange()), $delta);
            $delta -= $deltaStockChange;
            if ($change->getStockChange() < 0) {
                // Negate the change, since the stock is reduced by this list
                $deltaStockChange *= -1;
            }
            $change->setStockChange($change->getStockChange() - $deltaStockChange);

            // Create a new stock change instance based on the current location and the delta stock change
            $deltaStockChanges[] = new BinLocationStockChange($change->getBinLocation(), $deltaStockChange);
        }

        // Create a new list of the same class as this list
        $class = get_class($this);
        $deltaList = new $class($deltaStockChanges);

        return $deltaList;
    }

    /**
     * @param BinLocationStockChange $change
     * @return Boolean
     * @throws \InvalidArgumentException
     */
    abstract protected function validateBinLocationStockChange(BinLocationStockChange $change);
}
