<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList;

use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;

/**
 * @internal Use the service `pickware.erp.stock_change_list_factory_service` to generte stock change lists instead.
 */
class RelocationStockChangeList extends AbstractStockChangeList
{
    /**
     * @var BinLocationStockChange $destinationChange;
     */
    protected $destinationChange;

    /**
     * @param BinLocationStockChange[] $sourceChanges
     * @param BinLocationStockChange $destinationChange
     * @throws \InvalidArgumentException
     */
    public function __construct(array $sourceChanges, BinLocationStockChange $destinationChange)
    {
        // Validate and save the destination change
        if (!$destinationChange) {
            throw new \InvalidArgumentException('Second parameter "destinationChange" must not be null when creating a RelocationStockChangeList');
        }
        if ($destinationChange->getStockChange() < 0) {
            throw new \InvalidArgumentException(
                sprintf('The stock change of parameter "destinationChange" must be greater than or equal to 0 (%d given)', $destinationChange->getStockChange())
            );
        }
        foreach ($sourceChanges as $change) {
            if ($change->getBinLocation()->equals($destinationChange->getBinLocation())) {
                throw new \InvalidArgumentException('The bin location used by "destinationChange" must not be contained in "sourceChanges"');
            }
        }
        $this->destinationChange = $destinationChange;

        // Use a NegativeStockChangeList to validate the source changes
        $NegativeStockChangeList = new NegativeStockChangeList($sourceChanges);

        // Call the main constructor and pass it all source changes to perform additional validation
        parent::__construct($sourceChanges);

        // Save the destination change separately to bypass the 'same warehouse' validation
        $this->changes[] = $destinationChange;

        // Validate the total stock change of this list
        if ($this->getTotalStockChange() !== 0) {
            throw new \InvalidArgumentException(
                sprintf('The total sum of stock changes passed to a RelocationStockChangeList must be 0 (%d given)', $this->getTotalStockChange())
            );
        }
    }

    /**
     * @return BinLocation[]
     */
    public function getSourceLocations()
    {
        return array_values(array_map(function ($change) {
            return $change->getBinLocation();
        }, array_filter($this->changes, function ($change) {
            return !$change->getBinLocation()->equals($this->destinationChange->getBinLocation());
        })));
    }

    /**
     * @return BinLocation
     */
    public function getDestinationLocation()
    {
        return $this->destinationChange->getBinLocation();
    }

    /**
     * @return int
     */
    public function getRelocatedStock()
    {
        return $this->destinationChange->getStockChange();
    }

    /**
     * @Inheritdoc
     *
     * Overrides the default implementation to return this exact same change list instead of computing a new one.
     * This is necessary because a relocation cannot be comprised of more than one stock change set and the total of
     * such a list must always be zero. Hence the given $delta is validated first and an exception is thrown, if it
     * is not zero.
     */
    public function reduceAbsoluteTotalStockChange($delta)
    {
        // Validate the delta to be zero
        if ($delta !== 0) {
            throw new \InvalidArgumentException(
                sprintf('The value of parameter "delta" (%d) must be 0 when calling "reduceAbsoluteTotalStockChange()" on a RelocationStockChangeList', $delta)
            );
        }

        return $this;
    }

    /**
     * @Inheritdoc
     */
    protected function validateBinLocationStockChange(BinLocationStockChange $change)
    {
        return true;
    }
}
