<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList;

use InvalidArgumentException;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\ArticleDetailBinLocationMapping;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Article\Detail as ArticleDetail;

interface StockChangeListFactory
{
    /**
     * Creates and returns a stock change list that changes exactly one bin location.
     *
     * @param BinLocation $binLocation
     * @param int $stockChange
     * @return PositiveStockChangeList|NegativeStockChangeList
     */
    public function createSingleBinLocationStockChangeList(BinLocation $binLocation, $stockChange);

    /**
     * Creates and returns either a positive change list containing a single bin location stock change for the
     * 'stocking' bin location in case the given $changeAmount is positive. If it is negative, the 'picking' bin
     * locations are used to create a negative change list.
     *
     * @param Warehouse $warehouse
     * @param ArticleDetail $articleDetail
     * @param int $changeAmount
     * @return PositiveStockChangeList|NegativeStockChangeList
     * @throws InvalidArgumentException if the passed `$changeAmount` is 0.
     */
    public function createStockChangeList(Warehouse $warehouse, ArticleDetail $articleDetail, $changeAmount);

    /**
     * Given a quantity and an ordered list of bin location mappings, creates a list of changes that pick quantity units
     * of stock by successively reducing all passed mappings to zero in the order they are given.
     *
     * @param int $quantity how many items of stock to pick
     * @param ArticleDetailBinLocationMapping[] $binLocationMappings the mappings to pick from in order of precedence
     * @param BinLocation $fallbackBinLocation
     * @return BinLocationStockChange[]
     */
    public function createPickingStockChangesFromBinLocationMappings(
        $quantity,
        array $binLocationMappings,
        BinLocation $fallbackBinLocation
    );
}
