<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList;

use Enlight_Hook;
use InvalidArgumentException;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\ArticleDetailBinLocationMapping;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Order\Detail as OrderDetail;

class StockChangeListFactoryService implements StockChangeListFactory, Enlight_Hook
{
    /**
     * @var ModelManager
     */
    protected $entityManager;

    /**
     * @param ModelManager $entityManager
     */
    public function __construct($entityManager)
    {
        $this->entityManager = $entityManager;
    }

    /**
     * @inheritdoc
     */
    public function createSingleBinLocationStockChangeList(BinLocation $binLocation, $stockChange)
    {
        // Determine the class to be used for the change list
        if ($stockChange > 0) {
            $class = PositiveStockChangeList::class;
        } elseif ($stockChange < 0) {
            $class = NegativeStockChangeList::class;
        } else {
            throw StockChangeListFactoryException::stockChangeZero();
        }

        return new $class([
            new BinLocationStockChange($binLocation, $stockChange)
        ]);
    }

    /**
     * @inheritdoc
     */
    public function createStockChangeList(Warehouse $warehouse, ArticleDetail $articleDetail, $changeAmount)
    {
        if ($changeAmount === 0) {
            throw new InvalidArgumentException('The passed "changeAmount" must not be 0.');
        }

        if ($changeAmount > 0) {
            // Incoming stock, hence use the stocking strategy
            $binLocation = $this->findStockingBinLocation(
                $warehouse,
                $articleDetail
            );

            return new PositiveStockChangeList([
                new BinLocationStockChange($binLocation, $changeAmount)
            ]);
        } else {
            // Outgoing stock, hence use the default first picking strategy
            $stockChanges = $this->createDefaultFirstPickingStockChanges(
                $warehouse,
                $articleDetail,
                abs($changeAmount)
            );

            return new NegativeStockChangeList($stockChanges);
        }
    }

    /**
     * Fetches all bin location mappings sorted according to the 'default first' strategy:
     *
     *  - First, reduce all stock on the assigned default bin location, if it exists, then
     *  - reduce all non-default, non-null bin location mappings to zero starting from the one with the lowest stock
     *    (using code as a tie-breaker), then
     *  - subtract the remaining quantity from the null bin location.
     *
     * This sort order is always used when we don't know where the stock was actually taken from, so we use this because
     * most of the time, this will reduce stock from the default bin location, which is what is least surprising to the
     * user, and in all other cases, it will defragment the warehouse by reducing the number of existing bin locations
     * as fast as possible.
     *
     * @param Warehouse $warehouse
     * @param ArticleDetail $articleDetail
     * @param int $quantity
     * @return BinLocationStockChange[]
     */
    private function createDefaultFirstPickingStockChanges(
        Warehouse $warehouse,
        ArticleDetail $articleDetail,
        $quantity
    ) {
        // Find all bin location mappings, sorted so that the default bin location is first, then all non-default, non-
        // null bin locations sorted by ascending stock (and by code as a tie-breaker), then the null bin location
        $builder = $this->entityManager->createQueryBuilder();
        $builder
            ->select(
                'binLocationMapping',
                'CASE WHEN binLocation = :nullBinLocation THEN 1 ELSE 0 END AS HIDDEN isNullBinLocation'
            )
            ->from(ArticleDetailBinLocationMapping::class, 'binLocationMapping')
            ->join('binLocationMapping.binLocation', 'binLocation')
            ->where('binLocationMapping.articleDetail = :articleDetail')
            ->andWhere('binLocation.warehouse = :warehouse')
            ->orderBy('isNullBinLocation', 'ASC')
            ->addOrderBy('binLocationMapping.defaultMapping', 'DESC')
            ->addOrderBy('binLocationMapping.stock', 'ASC')
            ->addOrderBy('binLocation.code', 'ASC')
            ->setParameters([
                'articleDetail' => $articleDetail,
                'nullBinLocation' => $warehouse->getNullBinLocation(),
                'warehouse' => $warehouse,
            ]);
        $binLocationMappings = $builder->getQuery()->getResult();

        // Create as many stock changes as necessary for picking the quantity
        return $this->createPickingStockChangesFromBinLocationMappings(
            $quantity,
            $binLocationMappings,
            $warehouse->getNullBinLocation()
        );
    }

    /**
     * @inheritdoc
     */
    public function createPickingStockChangesFromBinLocationMappings(
        $quantity,
        array $binLocationMappings,
        BinLocation $fallbackBinLocation
    ) {
        // Create as many stock changes as necessary for picking the quantity
        /** @var BinLocationStockChange[] $stockChanges */
        $stockChanges = [];
        while ($quantity > 0 && count($binLocationMappings) > 0) {
            $binLocationMapping = array_shift($binLocationMappings);
            if (count($binLocationMappings) === 0) {
                // Reserve all remaining quantity for the current bin location mapping
                $stockChange = $quantity;
            } elseif ($binLocationMapping->getStock() > $binLocationMapping->getReservedStock()) {
                // Reserve as much stock as necessary and possible for the current bin location mapping
                $stockChange = min(
                    $quantity,
                    ($binLocationMapping->getStock() - $binLocationMapping->getReservedStock())
                );
            } else {
                continue;
            }
            $stockChanges[] = new BinLocationStockChange(
                $binLocationMapping->getBinLocation(),
                -1 * $stockChange
            );
            $quantity -= $stockChange;
        }

        if (count($stockChanges) === 0) {
            // For some reason we don't have any bin location mappings, hence create a single stock change using the
            // warehouse's null bin location as fallback
            $stockChanges[] = new BinLocationStockChange(
                $fallbackBinLocation,
                -1 * $quantity
            );
        }

        return $stockChanges;
    }

    /**
     * Fetches all bin location mappings sorted according to the 'stocking' strategy:
     *
     *  - default bin location mapping
     *  - non-default bin location mappings sorted by descending stock (alphabetical order in case of same stock)
     *  - null bin location mapping
     *
     * Finally the bin location of the first mapping in the result is returned.
     *
     * @param Warehouse $warehouse
     * @param ArticleDetail $articleDetail
     * @return BinLocation
     */
    private function findStockingBinLocation(Warehouse $warehouse, ArticleDetail $articleDetail)
    {
        // Find all bin location mappings, sorted by descending stock (default location first, null bin location last)
        $builder = $this->entityManager->createQueryBuilder();
        $builder
            ->select(
                'binLocationMapping',
                'CASE WHEN binLocation = :nullBinLocation THEN 1 ELSE 0 END AS HIDDEN isNullBinLocation'
            )
            ->from(ArticleDetailBinLocationMapping::class, 'binLocationMapping')
            ->join('binLocationMapping.binLocation', 'binLocation')
            ->where('binLocationMapping.articleDetail = :articleDetail')
            ->andWhere('binLocation.warehouse = :warehouse')
            ->orderBy('isNullBinLocation', 'ASC')
            ->addOrderBy('binLocationMapping.defaultMapping', 'DESC')
            ->addOrderBy('binLocationMapping.stock', 'DESC')
            ->addOrderBy('binLocation.code', 'ASC')
            ->setParameters([
                'articleDetail' => $articleDetail,
                'nullBinLocation' => $warehouse->getNullBinLocation(),
                'warehouse' => $warehouse,
            ]);
        $binLocationMappings = $builder->getQuery()->getResult();

        // Use the first bin location of the sorted result
        return $binLocationMappings[0]->getBinLocation();
    }
}
