<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockLedger;

use Shopware\Models\Article\Detail as ArticleDetail;

interface StockInitialization
{
    const DEFAULT_BATCH_SIZE = 100;

    /**
     * Performs stock initialization for all article details which are not already initialized.
     *
     * @throws StockLedgerConcurrencyException if a concurrent stock initialization is detected
     */
    public function initializeStocksOfAllUninitializedArticleDetails();

    /**
     * Performs stock initialization for a batch of article details which are not already initialized.
     *
     * @param int|null $batchSize
     * @return int The number of stocks which are not yet initialized
     * @throws StockLedgerConcurrencyException if a concurrent stock initialization is detected
     */
    public function initializeStocksForABatchOfUninitializedArticleDetails($batchSize = self::DEFAULT_BATCH_SIZE);

    /**
     * Initializes the stock and bin location mappings of all passed `$articleDetails`, if necessary. Any stock change
     * resulting from the initialization will be done in the default warehouse.
     *
     * @param ArticleDetail[] $articleDetails
     * @throws StockLedgerConcurrencyException if a concurrent stock initialization is detected
     */
    public function initializeStocksOfArticleDetails(array $articleDetails);

    /**
     * @return array
     */
    public function getAllUninitializedArticleDetails();

    /**
     * @param string|null $filterStr
     * @param string|null $orderBy
     * @param string|null $direction
     * @param int $start
     * @param int $limit
     * @return array
     */
    public function getPaginatedUninitializedArticleDetails(
        $filterStr,
        $orderBy,
        $direction,
        $start,
        $limit
    );
}
