<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockLedger;

use Enlight_Hook;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\WarehouseRepository;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Order\Status as OrderStatus;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonPickwareERP\Components\Warehouse\WarehouseIntegrity;
use Zend_Db_Adapter_Pdo_Abstract;

class StockInitializationService implements StockInitialization, Enlight_Hook
{
    /**
     * A sub query that matches all s_order rows whose status is one of 'aborted', 'cancelled' or 'completely delivered'.
     */
    const OPEN_ORDERS_STATUS_CONDITION = '(
        s_order.status NOT IN (' .
            OrderStatus::ORDER_STATE_CANCELLED . ',' .
            OrderStatus::ORDER_STATE_CANCELLED_REJECTED . ',' .
            OrderStatus::ORDER_STATE_COMPLETELY_DELIVERED .
        ')
    )';

    /**
     * @var Zend_Db_Adapter_Pdo_Abstract
     */
    protected $database;

    /**
     * @var ModelManager
     */
    protected $entityManager;

    /**
     * @var StockLedgerService
     */
    protected $stockLedgerService;

    /**
     * @var WarehouseIntegrity
     */
    protected $warehouseIntegrity;

    /**
     * @param Zend_Db_Adapter_Pdo_Abstract $database
     * @param ModelManager $entityManager
     * @param StockLedgerService $stockLedgerService
     * @param WarehouseIntegrity $warehouseIntegrity
     */
    public function __construct(
        $database,
        $entityManager,
        StockLedgerService $stockLedgerService,
        WarehouseIntegrity $warehouseIntegrity
    ) {
        $this->database = $database;
        $this->entityManager = $entityManager;
        $this->stockLedgerService = $stockLedgerService;
        $this->warehouseIntegrity = $warehouseIntegrity;
    }

    /**
     * @inheritdoc
     */
    public function initializeStocksOfAllUninitializedArticleDetails()
    {
        $inventory = $this->getAllUninitializedArticleDetails();

        $articleDetailIds = array_map(
            function ($article) {
                return $article['detailsId'];
            },
            $inventory['data']
        );

        $articleDetails = $this->entityManager->getRepository(ArticleDetail::class)->findBy([
            'id' => $articleDetailIds,
        ]);

        $this->initializeStocksOfArticleDetails($articleDetails);
    }

    /**
     * @inheritdoc
     */
    public function initializeStocksForABatchOfUninitializedArticleDetails($batchSize = self::DEFAULT_BATCH_SIZE)
    {
        $inventory = $this->getPaginatedUninitializedArticleDetails(
            null,
            null,
            null,
            0,
            $batchSize
        );

        $articleDetailIds = array_map(
            function ($article) {
                return $article['detailsId'];
            },
            $inventory['data']
        );

        $articleDetails = $this->entityManager->getRepository(ArticleDetail::class)->findBy([
            'id' => $articleDetailIds,
        ]);

        $this->initializeStocksOfArticleDetails($articleDetails);

        return $inventory['total'] - count($articleDetailIds);
    }

    /**
     * @inheritdoc
     */
    public function initializeStocksOfArticleDetails(array $articleDetails)
    {
        /** @var WarehouseRepository $warehouseRepository */
        $warehouseRepository = $this->entityManager->getRepository(Warehouse::class);
        $defaultWarehouse = $warehouseRepository->getDefaultWarehouse();

        foreach ($articleDetails as $articleDetail) {
            // Initialize the warehouse/bin location mappings, if necessary
            $this->warehouseIntegrity->ensureAllWarehouseMappingsForArticleDetail($articleDetail);

            // Determine the designated bin location for the stock change
            $binLocations = $warehouseRepository->findAllBinLocations($defaultWarehouse, $articleDetail);
            $binLocation = (array_shift($binLocations)) ?: $defaultWarehouse->getNullBinLocation();

            // Initialize the stock in the default warehouse
            $actualStock = $this->getActualStockForArticle($articleDetail);
            $this->stockLedgerService->initializeStockOfArticleDetail($articleDetail, $binLocation, $actualStock);
        }
    }

    /**
     * @inheritdoc
     */
    public function getPaginatedUninitializedArticleDetails(
        $filterStr,
        $orderBy,
        $direction,
        $offset,
        $rowCount
    ) {
        $sortableColumns = [
            'orderNumber',
            'instock',
            'articleName',
            'openOrders',
            'actualStock',
            'purchasePrice',
            'supplier',
        ];

        $whereClause = '';
        // Filter articles by a search string
        if ($filterStr && is_string($filterStr)) {
            $escapedFilter = $this->database->quote('%' . $filterStr . '%');
            $whereClause = ' AND ( s_articles_details.orderNumber LIKE ' . $escapedFilter .' OR
                s_articles_details.additionalText LIKE ' . $escapedFilter .' OR
                s_articles.name LIKE ' . $escapedFilter .')';
        }

        // order the results
        // $filterStockProblems must not be set when sorting, query comment for reason
        $orderClause = '';
        if ($orderBy && in_array($orderBy, $sortableColumns)) {
            $identifier = $this->database->quoteIdentifier($orderBy);
            if ($direction === 'ASC') {
                $sortDirection = 'ASC';
            } else {
                $sortDirection = 'DESC';
            }

            $orderClause = 'ORDER BY ' . $identifier . ' ' . $sortDirection;
        }

        $limitClause = sprintf('LIMIT %d, %d', $offset, $rowCount);

        $sql = $this->buildUnininitializedArticleDetailsQuery($whereClause, $orderClause, $limitClause);

        // Fetch both the paginated result and the total number of results
        $result = $this->database->fetchAll($sql);
        $total = $this->database->fetchOne('SELECT FOUND_ROWS()');

        return [
            'data' => $result,
            'total' => $total,
        ];
    }

    /**
     * @inheritdoc
     */
    public function getAllUninitializedArticleDetails()
    {
        $sql = $this->buildUnininitializedArticleDetailsQuery();

        // Fetch both the paginated result and the total number of results
        $result = $this->database->fetchAll($sql);
        $total = $this->database->fetchOne('SELECT FOUND_ROWS()');

        return [
            'data' => $result,
            'total' => $total,
        ];
    }

    private function buildUnininitializedArticleDetailsQuery(
        $additionalWhereClause = '',
        $orderClause = '',
        $limitClause = ''
    ) {
        // Conditional additional joins
        $additionalJoins = [];
        if (ViisonCommonUtil::isPluginInstalledAndActive(null, 'ViisonPickwareMobile')) {
            $additionalJoins[] = 'LEFT JOIN s_order_details_attributes ON s_order_details_attributes.detailID = s_order_details.id';
        }

        // Build the query
        $sql = '
            SELECT SQL_CALC_FOUND_ROWS
                s_articles_details.id,
                s_articles_details.orderNumber AS orderNumber,
                s_articles_details.instock AS instock,
                s_articles_details.id AS detailsId,
                s_articles_details.articleId AS articleId,
                s_articles.name AS articleName,
                s_articles_supplier.name AS supplier,
                ' . $this->getOpenOrdersQuantitySubquery() . ' AS openOrders,
                # Select the sum of inStock and openOrders as well to be able to sort by that value
                (' . $this->getOpenOrdersQuantitySubquery() . ' + s_articles_details.instock) AS actualStock,
                s_articles_details.purchaseprice AS purchasePrice
            FROM s_articles_details
            LEFT JOIN s_order_details
                ON s_order_details.articleordernumber = s_articles_details.ordernumber
            LEFT JOIN s_order
                ON s_order.id = s_order_details.orderID
            INNER JOIN s_articles
                ON s_articles.id = s_articles_details.articleID
            LEFT JOIN s_articles_supplier
                ON s_articles.supplierID = s_articles_supplier.id
            LEFT JOIN s_articles_esd
                ON s_articles_esd.articledetailsID = s_articles_details.id
            LEFT JOIN s_articles_attributes
                ON s_articles_attributes.articledetailsID = s_articles_details.id
            ' . implode(' ', $additionalJoins) . '
            LEFT JOIN (
                SELECT
                    SUM(items.cancelledQuantity) AS cancelledReturned,
                    SUM(items.returnedQuantity) AS returned,
                    items.orderDetailId AS orderDetailId
                FROM pickware_erp_return_shipment_items AS items
                GROUP BY items.orderDetailId
            ) AS derivedQuantities 
                ON derivedQuantities.orderDetailId = s_order_details.id
            WHERE
                s_articles_attributes.pickware_stock_initialized = 0
            AND (
                s_articles_attributes.pickware_stock_management_disabled IS NULL
                OR s_articles_attributes.pickware_stock_management_disabled = 0
            )
            AND s_articles_esd.id IS NULL
            '. $additionalWhereClause .'
            GROUP BY s_articles_details.id
            ' . $orderClause . '
            ' . $limitClause;

        return $sql;
    }

    /**
     * @param ArticleDetail $articleDetail
     * @return int
     */
    protected function getActualStockForArticle(ArticleDetail $articleDetail)
    {
        // Conditional additional joins
        $additionalJoins = [];
        if (ViisonCommonUtil::isPluginInstalledAndActive(null, 'ViisonPickwareMobile')) {
            $additionalJoins[] = 'LEFT JOIN s_order_details_attributes ON s_order_details_attributes.detailID = s_order_details.id';
        }

        $result = $this->database->fetchOne(
            'SELECT (' . $this->getOpenOrdersQuantitySubquery() . ' + s_articles_details.instock) AS actualStock
            FROM s_articles_details
            LEFT JOIN s_order_details
                ON s_order_details.articleordernumber = s_articles_details.ordernumber
            ' . implode(' ', $additionalJoins) . '
            LEFT JOIN s_order
                ON s_order.id = s_order_details.orderID
            LEFT JOIN (
                SELECT
                    SUM(items.cancelledQuantity) AS cancelledReturned,
                    SUM(items.returnedQuantity) AS returned,
                    items.orderDetailId AS orderDetailId
                FROM pickware_erp_return_shipment_items AS items
                GROUP BY items.orderDetailId
            ) AS derivedQuantities 
                ON derivedQuantities.orderDetailId = s_order_details.id
            WHERE s_articles_details.id = :articleDetailId
            GROUP BY s_articles_details.id',
            [
                'articleDetailId' => $articleDetail->getId(),
            ]
        );

        return intval($result);
    }

    /**
     * A sub query that sums up all open (not shipped) quantities of orders that match OPEN_ORDERS_STATUS_CONDITION.
     * That is the order quantity - any shipped quantity and (if ViisonPickwareMobile is installed) - any picked
     * quantity.
     *
     * @return string
     */
    protected function getOpenOrdersQuantitySubquery()
    {
        return sprintf(
            '(
                SUM(CASE
                    WHEN ' . self::OPEN_ORDERS_STATUS_CONDITION . '
                        THEN
                            GREATEST(0, s_order_details.quantity + IFNULL(derivedQuantities.cancelledReturned, 0) - s_order_details.shipped %s)
                    ELSE
                        0
                END)
            )',
            ViisonCommonUtil::isPluginInstalledAndActive(null, 'ViisonPickwareMobile') ? '- s_order_details_attributes.pickware_picked_quantity' : ''
        );
    }
}
