<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockLedger;

use Enlight_Hook;
use OverflowException;
use Psr\Log\LoggerInterface;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\ItemProperty\ItemProperty;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockItem;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockItemPropertyValue;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry;

class StockItemUpdaterService implements StockItemUpdater, Enlight_Hook
{
    /**
     * @var ModelManager
     */
    protected $entityManager;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @param ModelManager $entityManager
     * @param LoggerInterface $logger
     */
    public function __construct($entityManager, LoggerInterface $logger)
    {
        $this->entityManager = $entityManager;
        $this->logger = $logger;
    }

    /**
     * @inheritdoc
     */
    public function addStockItemsWithProperties(StockLedgerEntry $stockEntry, array $propertyValues)
    {
        if (!$this->canAddStockItems($stockEntry, count($propertyValues))) {
            throw new OverflowException(sprintf(
                'Cannot add properties for %d new stock entry items, because the stock entry already has %d items (change amount: %d)',
                count($propertyValues),
                $stockEntry->getStockItems()->count(),
                $stockEntry->getChangeAmount()
            ));
        }

        // Create the given number of items
        $createdStockEntryItems = array_map(
            function (array $values) use ($stockEntry) {
                return $this->addStockItem($stockEntry, $values);
            },
            $propertyValues
        );

        return $createdStockEntryItems;
    }

    /**
     * @inheritdoc
     */
    public function addStockItem(StockLedgerEntry $stockEntry, array $propertyValues = [])
    {
        if (!$this->canAddStockItems($stockEntry, 1)) {
            throw new OverflowException(sprintf(
                'Cannot add stock entry item, because the stock entry already has %d items (change amount: %d)',
                $stockEntry->getStockItems()->count(),
                $stockEntry->getChangeAmount()
            ));
        }

        // Create and save the new stock Entry item
        $stockEntryItem = new StockItem($stockEntry);
        $this->entityManager->persist($stockEntryItem);
        $this->entityManager->flush($stockEntryItem);

        // Add properties
        $this->updateStockItemPropertyValues($stockEntryItem, $propertyValues);

        return $stockEntryItem;
    }

    /**
     * @inheritdoc
     */
    public function updateStockItemPropertyValues(StockItem $stockEntryItem, array $propertyValues)
    {
        $updatedStockEntryItemProperties = [];
        foreach ($propertyValues as $itemPropertyId => $value) {
            // Try to find the property type
            $propertyType = $this->entityManager->find(ItemProperty::class, $itemPropertyId);
            if (!$propertyType) {
                $this->logger->info(
                    sprintf(
                        '%s: Trying to update item property of invalid type with ID %d for stock entry item with ID %d',
                        self::class,
                        $itemPropertyId,
                        $stockEntryItem->getId()
                    ),
                    $propertyValues
                );
                continue;
            }

            // Check for an existing item property of that type
            $stockEntryItemProperty = $stockEntryItem->getPropertyValues()->filter(
                function (StockItemPropertyValue $property) use ($propertyType) {
                    return $property->getItemProperty()->getId() === $propertyType->getId();
                }
            )->first();
            if (!$stockEntryItemProperty) {
                // Create new item property
                $stockEntryItemProperty = new StockItemPropertyValue($stockEntryItem, $propertyType);
                $this->entityManager->persist($stockEntryItemProperty);
            }

            // Update value
            $stockEntryItemProperty->setValue($value);
            $updatedStockEntryItemProperties[] = $stockEntryItemProperty;
        }

        // Save changes
        $this->entityManager->flush($updatedStockEntryItemProperties);
    }

    /**
     * @inheritdoc
     */
    public function canAddStockItems(StockLedgerEntry $stockEntry, $quantity)
    {
        return ($stockEntry->getStockItems()->count() + $quantity) <= abs($stockEntry->getChangeAmount());
    }
}
