<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockLedger;

use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentItem;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry;
use Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderItem;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Models\User\User;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;

/**
 * Factory class to create StockLedgerEntries that will share the same value for specific properties.
 * @internal
 */
class StockLedgerEntryTemplate
{
    /**
     * @var string $type
     */
    private $type;

    /**
     * @var ArticleDetail
     */
    private $articleDetail;

    /**
     * @var Warehouse $warehouse
     */
    private $warehouse;

    /**
     * @var OrderDetail|null
     */
    private $orderDetail = null;

    /**
     * @var SupplierOrderItem|null
     */
    private $supplierOrderItem = null;

    /**
     * @var ReturnShipmentItem|null
     */
    private $returnShipmentItem = null;

    /**
     * @var bool
     */
    private $correction = false;

    /**
     * @var User|null
     */
    private $user = null;

    /**
     * @var string $comment
     */
    private $comment = '';

    /**
     * @param ArticleDetail $articleDetail
     * @param Warehouse $warehouse
     * @param string $type
     */
    public function __construct(ArticleDetail $articleDetail, Warehouse $warehouse, $type)
    {
        $this->articleDetail = $articleDetail;
        $this->warehouse = $warehouse;
        $this->type = $type;
    }

    /**
     * @param string $transactionId
     * @param int $oldStock
     * @param int $changeAmount
     * @return StockLedgerEntry
     */
    public function createStockLedgerEntry($transactionId, $oldStock, $changeAmount)
    {
        $stockLedgerEntry = new StockLedgerEntry();
        $stockLedgerEntry->setTransactionId($transactionId);
        $stockLedgerEntry->setType($this->type);
        $stockLedgerEntry->setArticleDetail($this->articleDetail);
        $stockLedgerEntry->setWarehouse($this->warehouse);
        $stockLedgerEntry->setOldStock($oldStock);
        $stockLedgerEntry->setChangeAmount($changeAmount);
        $stockLedgerEntry->setOrderDetail($this->orderDetail);
        $stockLedgerEntry->setReturnShipmentItem($this->returnShipmentItem);
        $stockLedgerEntry->setSupplierOrderItem($this->supplierOrderItem);
        $stockLedgerEntry->setComment($this->comment);
        $stockLedgerEntry->setUser($this->user ?: ViisonCommonUtil::getCurrentUser());
        $stockLedgerEntry->setCorrection($this->correction);

        return $stockLedgerEntry;
    }

    /**
     * @return string
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * @param string $type
     */
    public function setType($type)
    {
        $this->type = $type;
    }

    /**
     * @return ArticleDetail
     */
    public function getArticleDetail()
    {
        return $this->articleDetail;
    }

    /**
     * @param ArticleDetail $articleDetail
     */
    public function setArticleDetail(ArticleDetail $articleDetail)
    {
        $this->articleDetail = $articleDetail;
    }

    /**
     * @return Warehouse
     */
    public function getWarehouse()
    {
        return $this->warehouse;
    }

    /**
     * @param Warehouse $warehouse
     */
    public function setWarehouse(Warehouse $warehouse)
    {
        $this->warehouse = $warehouse;
    }

    /**
     * @return OrderDetail|null
     */
    public function getOrderDetail()
    {
        return $this->orderDetail;
    }

    /**
     * @param OrderDetail|null $orderDetail
     */
    public function setOrderDetail(OrderDetail $orderDetail = null)
    {
        $this->orderDetail = $orderDetail;
    }

    /**
     * @return User|null
     */
    public function getUser()
    {
        return $this->user;
    }

    /**
     * @param User|null $user
     */
    public function setUser($user = null)
    {
        $this->user = $user;
    }

    /**
     * @return string
     */
    public function getComment()
    {
        return $this->comment;
    }

    /**
     * @param string $comment
     */
    public function setComment($comment)
    {
        $this->comment = $comment;
    }

    /**
     * @return SupplierOrderItem|null
     */
    public function getSupplierOrderItem()
    {
        return $this->supplierOrderItem;
    }

    /**
     * @param SupplierOrderItem|null $supplierOrderItem
     */
    public function setSupplierOrderItem(SupplierOrderItem $supplierOrderItem = null)
    {
        $this->supplierOrderItem = $supplierOrderItem;
    }

    /**
     * @return ReturnShipmentItem|null
     */
    public function getReturnShipmentItem()
    {
        return $this->returnShipmentItem;
    }

    /**
     * @param ReturnShipmentItem|null $returnShipmentItem
     */
    public function setReturnShipmentItem(ReturnShipmentItem $returnShipmentItem = null)
    {
        $this->returnShipmentItem = $returnShipmentItem;
    }

    /**
     * @return bool
     */
    public function isCorrection()
    {
        return $this->correction;
    }

    /**
     * @param bool $isCorrection
     */
    public function setCorrection($isCorrection)
    {
        $this->correction = $isCorrection;
    }
}
