<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockLedger;

use Closure;
use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\UnitOfWork;
use Exception;
use Psr\Log\LoggerInterface;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentItem;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\BinLocationStockSnapshot;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\OrderStockReservation;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry;
use Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderItem;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\ArticleDetailBinLocationMapping;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\WarehouseRepository;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Attribute\Article as ArticleAttribute;
use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Models\User\User;
use Shopware\Plugins\ViisonCommon\Classes\Util\UuidUtil;
use Shopware\Plugins\ViisonPickwareERP\Components\DerivedPropertyUpdater\DerivedPropertyUpdater;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\AbstractStockChangeList;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\BinLocationStockChange;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\NegativeStockChangeList;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\PositiveStockChangeList;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\RelocationStockChangeList;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\StockChangeListFactory;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\PurchasePriceAssignment\AbstractPurchasePriceAssignment;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\PurchasePriceAssignment\FallbackPurchasePriceAssignment;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\PurchasePriceAssignment\IncomingPurchasePriceAssignment;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\PurchasePriceAssignment\NullPurchasePriceAssignment;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\PurchasePriceAssignment\OutgoingPurchasePriceAssignment;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\PurchasePriceAssignment\ReturningPurchasePriceAssignment;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\ZeroStockChangeList;
use Shopware_Components_Snippet_Manager as SnippetManager;

/**
 * Manager class used to create appropriate stock history entries for the various stock actions.
 * The following stock actions are supported:
 *
 *   1) PURCHASE
 *      New items of an article are purchased for a given purchase price and stored
 *      within the warehouse.
 *
 *   2) STOCKTAKE
 *      All available items of an article within the warehouse are counted and the
 *      article's instock quantity is updated accordingly.
 *
 *   3) SALE
 *      A given amount of items of an article has been sold and is removed
 *      from the warehouse.
 *
 *   4) RETURN
 *      A given amount of items of an article has been returned by the customer
 *      and is added to the warehouse.
 *
 *   5) MANUAL
 *      The instock quantity of an article is changed manually.
 *
 *   6) INCOMING
 *      An arbitrary incoming amount is added to the stock.
 *
 *   7) OUTGOING
 *      An arbitrary outgoing amount is removed from the stock.
 *
 * These actions can be roughly divided into two groups: "incoming" and "outgoing",
 * whereas "incoming" refers to all history entries, which are related to the storage
 * of items within the warehouse and "outgoing" refers to all entries, which are
 * concerned with the removal of items from the warehouse:
 *
 *   a) "incoming":
 *      - purchase entry
 *      - stocktake entry with positive change amount
 *      - return entry
 *      - manual entry with positive change amount
 *
 *   b) "outgoing"
 *      - stocktake entry with negative change amount
 *      - sale entry
 *      - manual entry with negative change amount
 *
 * History entries for "neutral" stocktake actions (change amount is zero) play a minor role
 * and therefore are not listed above, since they "have no effect" and are created only for
 * informational purposes.
 *
 * To enable the computation of the inventory value (Bewerteter Warenbestand) for a given reference
 * date, we need to link "outgoing" and "incoming" entries in a suitable way. If possible, each
 * "outgoing" entry is therefore related to an "incoming" entry, which is called it's source lot entry. "Incoming"
 * entries are assigned to "outgoing" entries based on their "free capacity" and in accordance with the
 * LIFO (Last In First Out) principle, meaning "incoming" entries are assigned in their reversed creation
 * order. The assignment process is (roughly) as follows:
 *
 *   1) get a list of all "incoming" entries for the given article in their reversed creation order
 *   2) compute the "free capacity" for each "incoming" entry in the list, whereas the "free capacity"
 *      is calculated as follows: change amount of "incoming" entry minus the sum of the change amounts
 *      of all already assigned "outgoing" entries
 *   3) Remove all entries from the list, whose "free capacity" = 0
 *   4) Check "free capacity" of the first list entry:
 *      a) "free capacity" >= change amount of "outgoing" entry?
 *         Create a new "outgoing" entry for the given change amount and assign it to  the "incoming" entry
 *      b) "free capacity" < change amount of "outgoing" entry?
 *         Create a new "outgoing" entry with change amount = "free capacity" and assign the
 *         "incoming" entry to the newly created "outgoing" entry. Remove the "incoming" entry
 *         from the list and repeat step 4) until no unprocessed change amount remains or
 *         the list is empty
 *   5) If some unprocessed change amount remains, create a new unassigned "outgoing" entry for the
 *      remaining change amount (purchase price = default purchase price)
 *
 * "Outgoing" entries created in step 4) are called assigned, whereas those created in step 5) are called
 * unassigned. If the assignment process results in a single "outgoing" entry, this entry is called a
 * singlepart entry otherwise the set of newly created entries is called a multipart entry.
 *
 * The instock quantity of each article, which is tracked within the stock history and the one maintained
 * by Shopware (within the article details) may differ, since Shopware updates his counter as soon as a new
 * order has been created, whereas the one maintained within the stock history is intended to track the
 * physical item movements and is therefore (usually) not updated before the order is completely fulfilled
 * and all items have been shipped. To keep these counters in sync the manager class takes care of updating
 * the Shopware counter, when creating new history entries.
 *
 * Remark:
 * The purchase price of the default customer group of the active shop is used as default purchase price.
 */
final class StockLedgerService
{
    /**
     * The comment, which is appended to a stock entry comment when changing the physical stock from a negative to
     * a positive value.
     */
    const POSTFIX_FOR_COMMENTS_OF_INACTIVE_INCOMING_ENTRIES = 'Korrektur Negativ-Bestand, ggf. EKs der vorangegangen Verkäufe unter Bestand überprüfen';

    /**
     * The divider which is inserted between given and generated stock entry comments.
     */
    const DIVIDER_FOR_COMMENT_POSTFIXES = ' / ';

    /**
     * @var ModelManager $entityManager
     */
    private $entityManager;

    /**
     * @var StockChangeListFactory
     */
    private $stockChangeListFactory;

    /**
     * @var DerivedPropertyUpdater
     */
    private $derivedPropertyUpdater;

    /**
     * @var ArticleDetailStockInfoProvider
     */
    private $articleDetailStockInfoProvider;

    /**
     * @var ArticleDetailConcurrencyCoordinator
     */
    private $articleDetailConcurrencyCoordinator;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var \Enlight_Components_Snippet_Namespace
     */
    private $snippetNamespace;

    /**
     * The random transaction ID that is added to every stock entry created by this manager instance.
     *
     * @var string $transactionId
     */
    private $transactionId;

    /**
     * @var WarehouseRepository
     */
    private $warehouseRepository;

    /**
     * @param ModelManager $entityManager
     * @param SnippetManager $snippetManager
     * @param StockChangeListFactory $stockChangeListFactory
     * @param DerivedPropertyUpdater $derivedPropertyUpdater
     * @param ArticleDetailStockInfoProvider $articleDetailStockInfoProvider
     * @param ArticleDetailConcurrencyCoordinator $articleDetailConcurrencyCoordinator
     * @param LoggerInterface $logger
     */
    public function __construct(
        $entityManager,
        $snippetManager,
        StockChangeListFactory $stockChangeListFactory,
        DerivedPropertyUpdater $derivedPropertyUpdater,
        ArticleDetailStockInfoProvider $articleDetailStockInfoProvider,
        ArticleDetailConcurrencyCoordinator $articleDetailConcurrencyCoordinator,
        LoggerInterface $logger
    ) {
        $this->entityManager = $entityManager;
        $this->stockChangeListFactory = $stockChangeListFactory;
        $this->derivedPropertyUpdater = $derivedPropertyUpdater;
        $this->articleDetailStockInfoProvider = $articleDetailStockInfoProvider;
        $this->articleDetailConcurrencyCoordinator = $articleDetailConcurrencyCoordinator;
        $this->logger = $logger;
        $this->snippetNamespace = $snippetManager->getNamespace(
            'viison_pickware_erp/components/stock_ledger/stock_ledger_service'
        );
        $this->transactionId = UuidUtil::generateUuidV4();
        $this->warehouseRepository = $this->entityManager->getRepository(Warehouse::class);
    }

    /**
     * @return string
     */
    public function getTransactionId()
    {
        return $this->transactionId;
    }

    /**
     * Updates the given `$articleDetail`'s attributes by setting its `pickwareStockManagementDisabled` flag to
     * `false`, if the `$articleDetail`'s stock is not being recorded in the stock ledger.
     *
     * @param ArticleDetail $articleDetail
     */
    public function startRecordingStockChangesForArticleDetail(ArticleDetail $articleDetail)
    {
        if ($this->articleDetailStockInfoProvider->isStockManaged($articleDetail)) {
            return;
        }

        // Remove the 'stock not managed' flag
        $this->ensureArticleDetailHasAttribute($articleDetail);
        $attribute = $articleDetail->getAttribute();
        $attribute->setPickwareStockManagementDisabled(false);
        $this->entityManager->flush($attribute);

        $this->derivedPropertyUpdater->recalculateMinimumOnlineStockForArticleDetail($articleDetail);
    }

    /**
     * Checks whether the given `$articleDetail`' stock is not recorded in the stock ledger and, if it is not, removes
     * all physical stock and relocates the `$articleDetail` to the null bin location in all warehouses. Finally the
     * `$articleDetail`'s attributes are updated by setting its `pickwareStockManagementDisabled` to `true`.
     *
     * @param ArticleDetail $articleDetail
     */
    public function stopRecordingStockChangesForArticleDetail(ArticleDetail $articleDetail)
    {
        if (!$this->articleDetailStockInfoProvider->isStockManaged($articleDetail)) {
            return;
        }

        // Make sure the article detail has attributes in the database
        $this->ensureArticleDetailHasAttribute($articleDetail);
        $attribute = $articleDetail->getAttribute();
        $this->entityManager->flush($attribute);

        // Reset its physical stock to 0 and relocate the article detail to the null bin location in all warehouses
        $comment = $this->snippetNamespace->get('stop_recording_stock_changes_for_article_detail/entry_comment');
        $allWarehouses = $this->warehouseRepository->findAll();
        foreach ($allWarehouses as $warehouse) {
            $this->discardStock($articleDetail, $warehouse, $comment);
        }

        // Set the 'stock not managed' flag ONLY AFTER moving all stock, because otherwise no stock entries
        // would be written
        $attribute->setPickwareStockManagementDisabled(true);
        $this->entityManager->flush($attribute);

        $this->derivedPropertyUpdater->recalculateMinimumOnlineStockForArticleDetail($articleDetail);
    }

    /**
     * Initializes the managed article detail by updating its total, physical stock to the given `$actualStock`.
     *
     * @param ArticleDetail $articleDetail
     * @param BinLocation $binLocation The bin location that should be changed when adjusting the stock.
     * @param int $actualStock The actual, physical stock across all warehouses.
     * @param string $comment
     * @param User|null $user
     * @return StockLedgerEntry[]
     * @throws StockLedgerConcurrencyException If the managed article detail has already been initialized.
     */
    public function initializeStockOfArticleDetail(
        ArticleDetail $articleDetail,
        BinLocation $binLocation,
        $actualStock,
        $comment = '',
        User $user = null
    ) {
        return $this->performWithLockForArticleDetail($articleDetail, function () use ($articleDetail, $binLocation, $actualStock, $comment, $user) {
            $this->ensureArticleDetailHasAttribute($articleDetail);

            // Verify that the article detail has not been initialized yet
            if ($this->articleDetailStockInfoProvider->isStockInitialized($articleDetail)) {
                throw StockLedgerConcurrencyException::concurrentStockInitialization(
                    $articleDetail,
                    $binLocation->getWarehouse()
                );
            }

            $stockEntries = [];
            if ($this->articleDetailStockInfoProvider->isStockManaged($articleDetail)) {
                // Calculate the total, physical stock across all warehouses using their last stock entries
                $totalStock = $this->entityManager->getConnection()->fetchColumn(
                    'SELECT SUM(newStock)
                    FROM `pickware_erp_stock_ledger_entries`
                    WHERE id IN (
                        SELECT MAX(id)
                        FROM `pickware_erp_stock_ledger_entries`
                        WHERE articleDetailId = :articleDetailId
                        GROUP BY warehouseId
                    )',
                    [
                        'articleDetailId' => $articleDetail->getId(),
                    ]
                );

                // Calculate how much stock must be initialized on the passed bin location
                $binLocationMapping = $this->findRefreshedArticleDetailBinLocationMapping(
                    $articleDetail,
                    $binLocation
                );
                $binLocationStock = ($binLocationMapping) ? $binLocationMapping->getStock() : 0;
                $stockOnOtherBinLocations = $totalStock - $binLocationStock;
                $initializationStock = $actualStock - $stockOnOtherBinLocations;

                // Initialize the stock in the managed warehouse
                $stockEntries = $this->recordStockInitialization(
                    $articleDetail,
                    $binLocation,
                    $initializationStock,
                    $comment,
                    $user
                );
            }

            // Mark the manager article detail as initialized
            $articleDetail->getAttribute()->setPickwareStockInitialized(true);
            $initializationTime = (count($stockEntries) > 0) ? $stockEntries[0]->getCreated() : new DateTime();
            $articleDetail->getAttribute()->setPickwareStockInitializationTime($initializationTime);
            $this->entityManager->flush($articleDetail->getAttribute());

            return $stockEntries;
        });
    }

    /**
     * Creates a new history entry for a initialization action.
     *
     * @param ArticleDetail $articleDetail
     * @param BinLocation $binLocation The bin location, whose actual stock shall be logged
     * @param int $actualStock The actual physical stock quantity
     * @param string $comment An optional comment
     * @param User|null $user
     * @return StockLedgerEntry[]
     * @throws StockLedgerConcurrencyException
     */
    private function recordStockInitialization(
        ArticleDetail $articleDetail,
        BinLocation $binLocation,
        $actualStock,
        $comment = '',
        User $user = null
    ) {
        // Verify that the (article, warehouse) tuple was not initialized yet - this can happen as a race condition
        // in \Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockBuilder. In this case just throw to cancel the
        // running initialization.
        $warehouse = $binLocation->getWarehouse();
        if ($this->articleDetailStockInfoProvider->isStockInitialized($articleDetail)) {
            throw StockLedgerConcurrencyException::concurrentStockInitialization(
                $articleDetail,
                $warehouse
            );
        }

        // In case the attribute is set to false while it should be true, return the existing stock ledger entry here.
        // This ensures the attribute is set to its correct value including the initialization time.
        $existingInitializationEntry = $this->articleDetailStockInfoProvider->getLatestStockLedgerEntryForWarehouse(
            $articleDetail,
            $warehouse,
            StockLedgerEntry::TYPE_INITIALIZATION
        );
        if ($existingInitializationEntry
            && $articleDetail->getAttribute()
            && !$this->articleDetailStockInfoProvider->isStockInitialized($articleDetail)
        ) {
            $this->logger->info(
                sprintf(
                    '%s: Trying to initialize stocks of article with ID %d which has an existing ' .
                    'initialization stock ledger entry with ID %d, not writing a new stock ledger entry.',
                    self::class,
                    $articleDetail->getId(),
                    $existingInitializationEntry->getId()
                ),
                [
                    'articleDetailId' => $articleDetail->getId(),
                    'warehouseId' => $warehouse->getId(),
                    'binLocationId' => $binLocation->getId(),
                    'existingStockLedgerEntryId' => $existingInitializationEntry->getId(),
                ]
            );

            return [$existingInitializationEntry];
        }

        // Compute change list based on the actual stock and the stock currently in the database
        $binLocationMapping = $this->findRefreshedArticleDetailBinLocationMapping($articleDetail, $binLocation);
        $currentStock = ($binLocationMapping) ? $binLocationMapping->getStock() : 0;
        if (!$this->articleDetailStockInfoProvider->isStockManaged($articleDetail)) {
            // Articles whose stock is not managed must always be initialized with 0
            $actualStock = 0;
        }
        $stockChange = $actualStock - $currentStock;
        if ($stockChange === 0) {
            $stockChanges = ZeroStockChangeList::fromBinLocation($binLocation);
        } else {
            $stockChanges = $this->stockChangeListFactory->createSingleBinLocationStockChangeList(
                $binLocation,
                $stockChange
            );
        }

        // Determine the pricing model to be used
        if ($stockChange > 0) {
            // The initialization action results in an increase of the stock, hence create a new initialization
            // entry for the given stock change using the base pricing (= default purchase price)
            $pricing = new FallbackPurchasePriceAssignment($this->entityManager, $articleDetail, $warehouse, $stockChange);
        } elseif ($stockChange < 0) {
            // The initialization action results in an decrease of the instock quantity, hence create a new
            // singlepart / multipart initialization entry for the given stock change using the sale pricing strategy,
            // meaning that the "lost" items are treated just like sold ones with respect to the purchase price assignment
            $pricing = new OutgoingPurchasePriceAssignment($this->entityManager, $articleDetail, $warehouse, abs($stockChange));
        } else {
            // The stock is not affected by the initialization action, create a new initialization entry with change
            // amount 0 and purchase price null
            $pricing = new NullPurchasePriceAssignment($this->entityManager, $articleDetail, $warehouse, $stockChange);
        }

        $stockLedgerEntryTemplate = new StockLedgerEntryTemplate($articleDetail, $warehouse, StockLedgerEntry::TYPE_INITIALIZATION);
        $stockLedgerEntryTemplate->setComment($comment);
        $stockLedgerEntryTemplate->setUser($user);

        return $this->recordStockChange(
            $stockLedgerEntryTemplate,
            $pricing,
            $stockChanges
        );
    }

    /**
     * Creates a new history entry for a stocktake action.
     *
     * Happens when a stocktake is performed using the app.
     *
     * @param ArticleDetail $articleDetail
     * @param BinLocation $binLocation The bin location, whose actual stock shall be logged
     * @param int $actualStock The actual physical stock quantity
     * @param string $comment An optional comment
     * @param User|null $user
     * @return StockLedgerEntry[]
     * @throws StockLedgerException If the managed article detail has not been initialized yet.
     */
    public function recordStocktake(
        ArticleDetail $articleDetail,
        BinLocation $binLocation,
        $actualStock,
        $comment = '',
        User $user = null
    ) {
        // Validate passed stock
        if ($actualStock < 0) {
            throw new \InvalidArgumentException(
                sprintf('The value of parameter "actualStock" must not be less than 0 (%d given).', $actualStock)
            );
        }

        return $this->performWithLockForArticleDetail($articleDetail, function () use ($articleDetail, $binLocation, $actualStock, $comment, $user) {
            if ($this->articleDetailStockInfoProvider->isStockManaged($articleDetail) && !$this->articleDetailStockInfoProvider->isStockInitialized($articleDetail)) {
                throw StockLedgerException::stockTakeRequiresInitialization($articleDetail);
            }

            $warehouse = $binLocation->getWarehouse();
            $binLocationMapping = $this->findRefreshedArticleDetailBinLocationMapping($articleDetail, $binLocation);
            // Remember whether there was a mapping for the null bin location before taking stock - if yes, we should
            // not update that pre-existing null bin location mapping's lastStocktake timestamp after writing the
            // "stocktake" entry.
            $nullBinLocationMapping = $this->warehouseRepository->findArticleDetailBinLocationMapping(
                $articleDetail,
                $warehouse->getNullBinLocation()
            );

            // Compute change list based on the actual stock and the stock currently in the database
            $currentStock = ($binLocationMapping) ? $binLocationMapping->getStock() : 0;
            $stockChange = $actualStock - $currentStock;
            if ($stockChange === 0) {
                $stockChanges = ZeroStockChangeList::fromBinLocation($binLocation);
            } else {
                $stockChanges = $this->stockChangeListFactory->createSingleBinLocationStockChangeList(
                    $binLocation,
                    $stockChange
                );
            }

            // Determine the pricing model to be used
            if ($stockChange > 0) {
                // The stocktake action results in an increase of the stock, hence create a new stocktake
                // entry for the given stock change using the base pricing (= default purchase price)
                $pricing = new FallbackPurchasePriceAssignment($this->entityManager, $articleDetail, $warehouse, $stockChange);
            } elseif ($stockChange < 0) {
                // The stocktake action results in an decrease of the instock quantity, hence create a new
                // singlepart / multipart stocktake entry for the given stock change using the sale pricing strategy,
                // meaning that the "lost" items are treated just like sold ones with respect to the purchase price assignment
                $pricing = new OutgoingPurchasePriceAssignment($this->entityManager, $articleDetail, $warehouse, abs($stockChange));
            } else {
                // The stock is not affected by the stocktake action, create a new stocktake entry with change amount 0
                // and purchase price null
                $pricing = new NullPurchasePriceAssignment($this->entityManager, $articleDetail, $warehouse, $stockChange);
            }

            $stockLedgerEntryTemplate = new StockLedgerEntryTemplate($articleDetail, $warehouse, StockLedgerEntry::TYPE_STOCKTAKE);
            $stockLedgerEntryTemplate->setComment($comment);
            $stockLedgerEntryTemplate->setUser($user);
            $stockEntries = $this->recordStockChange(
                $stockLedgerEntryTemplate,
                $pricing,
                $stockChanges
            );

            if ($warehouse->isStockAvailableForSale()) {
                $this->increaseInstock($articleDetail, $stockChange);
            }

            // Re-fetch the bin location mapping because a new one may have been created or a previously existing mapping
            // dissolved
            $binLocationMapping = $this->warehouseRepository->findArticleDetailBinLocationMapping($articleDetail, $binLocation);

            // If the variant was not mapped to the null bin location before writing the stock entry and the mapping we
            // performed the stock-taking on was dissolved in the process, update the lastStocktake on the newly-created
            // bin location mapping so the variant does not unnecessarily get re-selected for stock-taking right away
            $shouldUpdateNullBinLocationMappingTimestamp = !$nullBinLocationMapping && !$binLocationMapping;
            if ($shouldUpdateNullBinLocationMappingTimestamp) {
                $binLocationMapping = $this->warehouseRepository->findArticleDetailBinLocationMapping(
                    $articleDetail,
                    $warehouse->getNullBinLocation()
                );
            }

            // Update the bin location mapping with the date of the stocktake entry
            $shouldUpdateLastStocktakeTimestamp = $binLocationMapping && (count($stockEntries) > 0) &&
                ($stockEntries[0]->getCreated() > $binLocationMapping->getLastStocktake());
            if ($shouldUpdateLastStocktakeTimestamp) {
                $binLocationMapping->setLastStocktake($stockEntries[0]->getCreated());
                $this->entityManager->flush($binLocationMapping);
            }

            return $stockEntries;
        });
    }

    /**
     * Creates a new history entry for a purchase action.
     *
     * This happens when a supplier order is received and processed in the warehouse.
     *
     * @param ArticleDetail $articleDetail
     * @param PositiveStockChangeList $stockChanges
     * @param float|null $purchasePrice The price the quantity was purchased for
     * @param string $comment An optional comment
     * @param SupplierOrderItem|null $supplierOrderItem An optional supplier order article to be associated with the
     *        purchase
     * @param User|null $user
     * @return StockLedgerEntry[]
     */
    public function recordPurchasedStock(
        ArticleDetail $articleDetail,
        PositiveStockChangeList $stockChanges,
        $purchasePrice = null,
        $comment = '',
        SupplierOrderItem $supplierOrderItem = null,
        User $user = null
    ) {
        return $this->performWithLockForArticleDetail($articleDetail, function () use ($articleDetail, $stockChanges, $purchasePrice, $comment, $supplierOrderItem, $user) {
            $purchasedQuantity = $stockChanges->getTotalStockChange();
            $warehouse = $stockChanges->getChangedWarehouse();

            // Determine pricing model
            if ($purchasePrice !== null) {
                $pricing = new IncomingPurchasePriceAssignment($this->entityManager, $articleDetail, $warehouse, $purchasedQuantity, $purchasePrice);
            } else {
                $pricing = new FallbackPurchasePriceAssignment($this->entityManager, $articleDetail, $warehouse, $purchasedQuantity);
            }

            $stockLedgerEntryTemplate = new StockLedgerEntryTemplate($articleDetail, $warehouse, StockLedgerEntry::TYPE_PURCHASE);
            $stockLedgerEntryTemplate->setComment($comment);
            $stockLedgerEntryTemplate->setUser($user);
            $stockLedgerEntryTemplate->setSupplierOrderItem($supplierOrderItem);
            $createdStockEntries = $this->recordStockChange(
                $stockLedgerEntryTemplate,
                $pricing,
                $stockChanges
            );

            if ($warehouse->isStockAvailableForSale()) {
                $this->increaseInstock($articleDetail, $purchasedQuantity);
            }

            if ($purchasePrice !== null) {
                // Update the article's default purchase price
                $this->updateArticlePurchasePrice($articleDetail, $purchasePrice);
            }

            return $createdStockEntries;
        });
    }

    /**
     * Creates a new history entry for a sale action.
     *
     * This happens when a customer order is shipped.
     *
     * @param ArticleDetail $articleDetail
     * @param OrderDetail $orderDetail The order detail, which was sold
     * @param NegativeStockChangeList $stockChanges
     * @param string $comment An optional comment
     * @param User|null $user
     * @return StockLedgerEntry[]
     */
    public function recordSoldStock(
        ArticleDetail $articleDetail,
        OrderDetail $orderDetail,
        NegativeStockChangeList $stockChanges,
        $comment = '',
        User $user = null
    ) {
        return $this->performWithLockForArticleDetail($articleDetail, function () use ($articleDetail, $orderDetail, $stockChanges, $comment, $user) {
            $soldQuantity = abs($stockChanges->getTotalStockChange());
            $warehouse = $stockChanges->getChangedWarehouse();

            $stockLedgerEntryTemplate = new StockLedgerEntryTemplate($articleDetail, $warehouse, StockLedgerEntry::TYPE_SALE);
            $stockLedgerEntryTemplate->setComment($comment);
            $stockLedgerEntryTemplate->setUser($user);
            $stockLedgerEntryTemplate->setOrderDetail($orderDetail);
            $createdStockEntries = $this->recordStockChange(
                $stockLedgerEntryTemplate,
                new OutgoingPurchasePriceAssignment($this->entityManager, $articleDetail, $warehouse, $soldQuantity),
                $stockChanges
            );

            // Remark: The article's instock value will not be updated automatically, since Shopware takes
            // care of this on it's own in case that a new order is created or the quantity of an order
            // position is changed (Shopware\Models\Order\Detail::beforeUpdate).
            //
            // The available stock of the managed article detail was reduced by the ordered quantity when the order
            // detail was created. However, if the warehouse of by this stock manager does not make its stock available
            // to the store front, the original reduction of the available stock must be reverted by as much as the
            // change of the stock entry, because the availability backed by the physical stock in the available
            // warehouses has not changed.
            // See https://github.com/VIISON/ShopwarePickwareERP/issues/777
            if (!$warehouse->isStockAvailableForSale()) {
                $this->increaseInstock($articleDetail, $soldQuantity);
            }

            return $createdStockEntries;
        });
    }

    /**
     * Creates a new history entry for a sale correction.
     *
     * Correction for {@see self::recordSoldStock()}
     *
     * @param ArticleDetail $articleDetail
     * @param OrderDetail $orderDetail
     * @param PositiveStockChangeList $stockChanges
     * @param string $comment
     * @param User|null $user
     * @return StockLedgerEntry[]
     */
    public function recordSoldStockCorrection(
        ArticleDetail $articleDetail,
        OrderDetail $orderDetail,
        PositiveStockChangeList $stockChanges,
        $comment = '',
        User $user = null
    ) {
        return $this->performWithLockForArticleDetail($articleDetail, function () use ($articleDetail, $orderDetail, $stockChanges, $comment, $user) {
            $correctionQuantity = abs($stockChanges->getTotalStockChange());
            $warehouse = $stockChanges->getChangedWarehouse();

            $stockLedgerEntryTemplate = new StockLedgerEntryTemplate($articleDetail, $warehouse, StockLedgerEntry::TYPE_SALE);
            $stockLedgerEntryTemplate->setCorrection(true);
            $stockLedgerEntryTemplate->setComment($comment);
            $stockLedgerEntryTemplate->setUser($user);
            $stockLedgerEntryTemplate->setOrderDetail($orderDetail);
            $createdStockEntries = $this->recordStockChange(
                $stockLedgerEntryTemplate,
                new ReturningPurchasePriceAssignment($this->entityManager, $articleDetail, $warehouse, $orderDetail, $correctionQuantity),
                $stockChanges
            );

            // Remark: The article's instock value will not be updated automatically, since Shopware takes
            // care of this on it's own in case that a new order is created or the quantity of an order
            // position is changed (Shopware\Models\Order\Detail::beforeUpdate).
            //
            // The available stock of the managed article detail was reduced by the ordered quantity when the order
            // detail was created. However, if the warehouse of this stock manager does not make its stock available
            // to the store front, the original reduction of the available stock must be reverted by as much as the
            // change of the stock entry, because the availability backed by the physical stock in the available
            // warehouses has not changed.
            // See https://github.com/VIISON/ShopwarePickwareERP/issues/777
            if (!$warehouse->isStockAvailableForSale()) {
                $articleDetail->setInStock($articleDetail->getInStock() - $correctionQuantity);
                $this->entityManager->flush($articleDetail);
            }

            return $createdStockEntries;
        });
    }

    /**
     * Creates a new history entry for a return action.
     *
     * This happens when a return shipment is received from a customer and unpacked in the warehouse.
     *
     * @param ArticleDetail $articleDetail
     * @param ReturnShipmentItem $returnShipmentItem An optional reshipment item to be associated with the return
     * @param PositiveStockChangeList $stockChanges
     * @param string $comment An optional comment
     * @param User|null $user
     * @return StockLedgerEntry[]
     */
    public function recordReturnedStock(
        ArticleDetail $articleDetail,
        ReturnShipmentItem $returnShipmentItem,
        PositiveStockChangeList $stockChanges,
        $comment = '',
        User $user = null
    ) {
        return $this->performWithLockForArticleDetail($articleDetail, function () use ($articleDetail, $returnShipmentItem, $stockChanges, $comment, $user) {
            $warehouse = $stockChanges->getChangedWarehouse();
            $returnedQuantity = $stockChanges->getTotalStockChange();
            $stockLedgerEntryTemplate = new StockLedgerEntryTemplate($articleDetail, $warehouse, StockLedgerEntry::TYPE_RETURN);
            $stockLedgerEntryTemplate->setComment($comment);
            $stockLedgerEntryTemplate->setUser($user);
            $stockLedgerEntryTemplate->setOrderDetail($returnShipmentItem->getOrderDetail());
            $stockLedgerEntryTemplate->setReturnShipmentItem($returnShipmentItem);
            $stockEntries = $this->recordStockChange(
                $stockLedgerEntryTemplate,
                new ReturningPurchasePriceAssignment(
                    $this->entityManager,
                    $articleDetail,
                    $warehouse,
                    $returnShipmentItem->getOrderDetail(),
                    $returnedQuantity
                ),
                $stockChanges
            );

            if ($warehouse->isStockAvailableForSale()) {
                $this->increaseInstock($articleDetail, $returnedQuantity);
            }

            return $stockEntries;
        });
    }

    /**
     * Creates a new history entry for a return correction.
     *
     * Correction for {@see self::recordReturnedStock()}
     *
     * @param ArticleDetail $articleDetail
     * @param ReturnShipmentItem $returnShipmentItem An optional reshipment item to be associated with the return
     * @param NegativeStockChangeList $stockChanges
     * @param string $comment An optional comment
     * @param User|null $user
     * @return StockLedgerEntry[]
     */
    public function recordReturnedStockCorrection(
        ArticleDetail $articleDetail,
        ReturnShipmentItem $returnShipmentItem,
        NegativeStockChangeList $stockChanges,
        $comment = '',
        User $user = null
    ) {
        return $this->performWithLockForArticleDetail($articleDetail, function () use ($articleDetail, $returnShipmentItem, $stockChanges, $comment, $user) {
            $correctionQuantity = $stockChanges->getTotalStockChange();
            $warehouse = $stockChanges->getChangedWarehouse();

            $stockLedgerEntryTemplate = new StockLedgerEntryTemplate($articleDetail, $warehouse, StockLedgerEntry::TYPE_RETURN);
            $stockLedgerEntryTemplate->setCorrection(true);
            $stockLedgerEntryTemplate->setComment($comment);
            $stockLedgerEntryTemplate->setUser($user);
            $stockLedgerEntryTemplate->setReturnShipmentItem($returnShipmentItem);
            $stockLedgerEntryTemplate->setOrderDetail($returnShipmentItem->getOrderDetail());

            $stockEntries = $this->recordStockChange(
                $stockLedgerEntryTemplate,
                new OutgoingPurchasePriceAssignment($this->entityManager, $articleDetail, $warehouse, abs($correctionQuantity)),
                $stockChanges
            );

            if ($warehouse->isStockAvailableForSale()) {
                $this->increaseInstock($articleDetail, $correctionQuantity);
            }

            return $stockEntries;
        });
    }

    /**
     * Create a history entry for a return action without a ReturnShipment.
     *
     * A special function made exclusively for POS, because POS does not create ReturnShipments.
     *
     * @param ArticleDetail $articleDetail
     * @param OrderDetail $orderDetail
     * @param PositiveStockChangeList $stockChanges
     * @param string $comment
     * @param User|null $user
     * @return StockLedgerEntry[]
     */
    public function recordReturnedStockWithoutReturnShipment(
        ArticleDetail $articleDetail,
        OrderDetail $orderDetail,
        PositiveStockChangeList $stockChanges,
        $comment = '',
        User $user = null
    ) {
        return $this->performWithLockForArticleDetail($articleDetail, function () use ($articleDetail, $orderDetail, $stockChanges, $comment, $user) {
            $returnedQuantity = $stockChanges->getTotalStockChange();
            $warehouse = $stockChanges->getChangedWarehouse();

            $stockLedgerEntryTemplate = new StockLedgerEntryTemplate(
                $articleDetail,
                $warehouse,
                StockLedgerEntry::TYPE_RETURN
            );
            $stockLedgerEntryTemplate->setComment($comment);
            $stockLedgerEntryTemplate->setUser($user);
            $stockLedgerEntryTemplate->setOrderDetail($orderDetail);

            $stockEntries = $this->recordStockChange(
                $stockLedgerEntryTemplate,
                new ReturningPurchasePriceAssignment(
                    $this->entityManager,
                    $articleDetail,
                    $warehouse,
                    $orderDetail,
                    abs($returnedQuantity)
                ),
                $stockChanges
            );

            if ($warehouse->isStockAvailableForSale()) {
                $this->increaseInstock($articleDetail, $returnedQuantity);
            }

            return $stockEntries;
        });
    }

    /**
     * Creates a new history entry for a manual manipulation of the instock quantity.
     *
     * This happens when the stock of an individual article is changed using the backend's article stock field set.
     *
     * @param ArticleDetail $articleDetail
     * @param AbstractStockChangeList $stockChanges
     * @param float $purchasePrice The price the (positive) quantity change was purchases for
     * @param string $comment An optional comment
     * @param boolean $updateArticleDetailInStock An optional boolean controlling whether to update the article detail's
     *        'inStock' field; defaults to true
     * @param User|null $user
     * @return StockLedgerEntry[]
     */
    private function recordManualStockChange(
        ArticleDetail $articleDetail,
        AbstractStockChangeList $stockChanges,
        $purchasePrice = 0.0,
        $comment = '',
        $updateArticleDetailInStock = true,
        User $user = null
    ) {
        return $this->performWithLockForArticleDetail($articleDetail, function () use ($articleDetail, $stockChanges, $purchasePrice, $comment, $updateArticleDetailInStock, $user) {
            $changeAmount = $stockChanges->getTotalStockChange();
            $warehouse = $stockChanges->getChangedWarehouse();
            // Select pricing strategy
            if ($changeAmount > 0) {
                // The stock is increased, hence create a manual entry for the given change amount
                // and the given purchase price
                $pricing = new IncomingPurchasePriceAssignment($this->entityManager, $articleDetail, $warehouse, $changeAmount, $purchasePrice);
            } elseif ($changeAmount < 0) {
                // The stock is decreased, hence create a singlepart / multipart manual entry for the given
                // change amount using the sale pricing strategy, meaning that the "removed" items are treated
                // just like sold ones with respect to the purchase price assignment
                $pricing = new OutgoingPurchasePriceAssignment($this->entityManager, $articleDetail, $warehouse, abs($changeAmount));
            } else {
                // "Effect-less" manipulations are prohibited
                throw new \InvalidArgumentException('ViisonPickwareERP - Stock history: Change amount must be unequal zero.');
            }

            $stockLedgerEntryTemplate = new StockLedgerEntryTemplate($articleDetail, $warehouse, StockLedgerEntry::TYPE_MANUAL);
            $stockLedgerEntryTemplate->setComment($comment);
            $stockLedgerEntryTemplate->setUser($user);
            $createdStockEntries = $this->recordStockChange(
                $stockLedgerEntryTemplate,
                $pricing,
                $stockChanges
            );

            if ($updateArticleDetailInStock && $warehouse->isStockAvailableForSale()) {
                $this->increaseInstock($articleDetail, $changeAmount);
            }

            if ($changeAmount > 0) {
                $this->updateArticlePurchasePrice($articleDetail, $purchasePrice);
            }

            return $createdStockEntries;
        });
    }

    /**
     * Creates a new history entry for arbitrary incoming stock changes.
     *
     * This happens when the stock amount for an individual article is increased through the app.
     *
     * @param ArticleDetail $articleDetail
     * @param PositiveStockChangeList $stockChanges
     * @param float|null $purchasePrice The price the incoming quantity was purchased for
     * @param string $comment An optional comment
     * @param User|null $user
     * @return StockLedgerEntry[]
     */
    public function recordIncomingStock(
        ArticleDetail $articleDetail,
        PositiveStockChangeList $stockChanges,
        $purchasePrice = null,
        $comment = '',
        User $user = null
    ) {
        return $this->performWithLockForArticleDetail($articleDetail, function () use ($articleDetail, $stockChanges, $purchasePrice, $comment, $user) {
            $incomingQuantity = $stockChanges->getTotalStockChange();
            $warehouse = $stockChanges->getChangedWarehouse();
            // Select pricing strategy
            if ($purchasePrice !== null) {
                $pricing = new IncomingPurchasePriceAssignment($this->entityManager, $articleDetail, $warehouse, $incomingQuantity, $purchasePrice);
            } else {
                $pricing = new FallbackPurchasePriceAssignment($this->entityManager, $articleDetail, $warehouse, $incomingQuantity);
            }

            $stockLedgerEntryTemplate = new StockLedgerEntryTemplate($articleDetail, $warehouse, StockLedgerEntry::TYPE_INCOMING);
            $stockLedgerEntryTemplate->setComment($comment);
            $stockLedgerEntryTemplate->setUser($user);
            $createdStockEntries = $this->recordStockChange(
                $stockLedgerEntryTemplate,
                $pricing,
                $stockChanges
            );

            if ($warehouse->isStockAvailableForSale()) {
                $this->increaseInstock($articleDetail, $incomingQuantity);
            }

            if ($purchasePrice !== null) {
                $this->updateArticlePurchasePrice($articleDetail, $purchasePrice);
            }

            return $createdStockEntries;
        });
    }

    /**
     * Creates a new history entry for arbitrary outgoing stock changes.
     *
     * This happens when the stock amount for an individual article is decreased through the app.
     *
     * @param ArticleDetail $articleDetail
     * @param NegativeStockChangeList $stockChanges
     * @param string $comment An optional comment
     * @param User|null $user
     * @return StockLedgerEntry[]
     */
    public function recordOutgoingStock(
        ArticleDetail $articleDetail,
        NegativeStockChangeList $stockChanges,
        $comment = '',
        User $user = null
    ) {
        return $this->performWithLockForArticleDetail($articleDetail, function () use ($articleDetail, $stockChanges, $comment, $user) {
            $outgoingQuantity = $stockChanges->getTotalStockChange();
            $warehouse = $stockChanges->getChangedWarehouse();

            $stockLedgerEntryTemplate = new StockLedgerEntryTemplate($articleDetail, $warehouse, StockLedgerEntry::TYPE_OUTGOING);
            $stockLedgerEntryTemplate->setComment($comment);
            $stockLedgerEntryTemplate->setUser($user);
            $stockEntries = $this->recordStockChange(
                $stockLedgerEntryTemplate,
                new OutgoingPurchasePriceAssignment($this->entityManager, $articleDetail, $warehouse, abs($outgoingQuantity)),
                $stockChanges
            );

            if ($warehouse->isStockAvailableForSale()) {
                $this->increaseInstock($articleDetail, $outgoingQuantity);
            }

            return $stockEntries;
        });
    }

    /**
     * First validates the given $stockChanges to make sure that all source bin locations are currently mapped to the
     * article detail and their stock is sufficient for the relocation. The respective stock change quantity does not
     * exceed the stock currently residing on that bin location. The latter validation prevents any bin location from
     * containing negative stock after the relocation, which would be a violation of our data model.
     * Those validations apply to all source bin locations, except for the warehouse's null bin location, if present.
     * We allow this exception, because we have to acknowledge that stocks might be corrupt in a way, that in the real
     * world stock exists, but that stock is not reflected in the database, which results in zero or negative stock on
     * the null bin location (see {@link self::updateBinLocationMappings()}). In that case, relocating that stock in
     * the real world must still be possible (see {@link https://github.com/VIISON/ShopwarePickwareMobile/issues/163}).
     * Once validated, depending on the warehouses of the source and destination bin locations, the relocation is
     * performed by either logging a 'relocation' in the managed warehouse or by logging a sequence of stock entries for
     * a relocation to a different warehouse than the one managed by this stock manager.
     *
     * @param ArticleDetail $articleDetail
     * @param RelocationStockChangeList $stockChanges
     * @param string $comment
     * @param User|null $user
     * @return StockLedgerEntry[]
     * @throws StockLedgerValidationException if any of the source bin locations is currently not mapped to the variant
     *         or its stock is not sufficient for the relocation.
     */
    public function recordRelocatedStock(
        ArticleDetail $articleDetail,
        RelocationStockChangeList $stockChanges,
        $comment = '',
        User $user = null
    ) {
        return $this->performWithLockForArticleDetail($articleDetail, function () use ($articleDetail, $stockChanges, $comment, $user) {
            if (!$this->articleDetailStockInfoProvider->isStockManaged($articleDetail)) {
                // Don't do anything, since relocations only change the physical stock, not the available stock
                return [];
            }

            $sourceWarehouse = $stockChanges->getChangedWarehouse();

            // Validate that the source locations are all currently mapped to the article detail and that the moved
            // stock does not exceed the stock at those locations. Note that the warehouse's null bin location is
            // exempt from these validations.
            foreach ($stockChanges->getSourceLocations() as $sourceBinLocation) {
                if ($sourceBinLocation->equals($sourceWarehouse->getNullBinLocation())) {
                    // The null bin location can always be used as a source location
                    continue;
                }
                $sourceBinLocationMapping = $this->findRefreshedArticleDetailBinLocationMapping(
                    $articleDetail,
                    $sourceBinLocation
                );
                if (!$sourceBinLocationMapping) {
                    throw StockLedgerValidationException::binLocationNotMapped(
                        $articleDetail,
                        $sourceWarehouse,
                        $sourceBinLocation
                    );
                }
                $stockChange = $stockChanges->getStockChangeForLocation($sourceBinLocation)->getStockChange();
                if (abs($stockChange) > $sourceBinLocationMapping->getStock()) {
                    throw StockLedgerValidationException::stockChangeExceedsCurrentStock(
                        $articleDetail,
                        $sourceWarehouse,
                        $sourceBinLocation,
                        $stockChange,
                        $sourceBinLocationMapping->getStock()
                    );
                }
            }

            // Check for a relocation between warehouses
            if ($stockChanges->getDestinationLocation()->getWarehouse()->equals($sourceWarehouse)) {
                // Just log the stock change
                $stockLedgerEntryTemplate = new StockLedgerEntryTemplate(
                    $articleDetail,
                    $sourceWarehouse,
                    StockLedgerEntry::TYPE_RELOCATION
                );
                $stockLedgerEntryTemplate->setComment($comment);
                $stockLedgerEntryTemplate->setUser($user);

                return $this->recordStockChange(
                    $stockLedgerEntryTemplate,
                    new NullPurchasePriceAssignment($this->entityManager, $articleDetail, $sourceWarehouse, 0),
                    $stockChanges
                );
            }

            return $this->recordStockRelocatedBetweenWarehouses($articleDetail, $stockChanges, $comment, $user);
        });
    }

    /**
     * Finds all bin location mappings of the selected article detail in the selected warehouse and both unsets any
     * 'default mapping' flags and removes any non-zero stock from the bin location mappings by logging manual stock
     * entries using the default purchase price.
     *
     * @param ArticleDetail $articleDetail
     * @param Warehouse $warehouse
     * @param string|null $comment
     * @param User|null $user
     */
    public function discardStock(
        ArticleDetail $articleDetail,
        Warehouse $warehouse,
        $comment = null,
        User $user = null
    ) {
        $this->performWithLockForArticleDetail($articleDetail, function () use ($articleDetail, $warehouse, $comment, $user) {
            $binLocationMappings = $this->findRefreshedSortedWarehouseBinLocationMappings(
                $warehouse,
                $articleDetail
            );

            // Make sure no mapping is marked as the article detail's default mapping
            foreach ($binLocationMappings as $binLocationMapping) {
                $binLocationMapping->setDefaultMapping(false);
            }
            $this->entityManager->flush($binLocationMappings);

            foreach ($binLocationMappings as $binLocationMapping) {
                if ($binLocationMapping->getStock() === 0) {
                    // Warehouse's null bin location, which can have zero stock
                    continue;
                }

                // Remove any stock from the bin location mapping, which automatically removes the mapping too
                $stockChanges = $this->stockChangeListFactory->createSingleBinLocationStockChangeList(
                    $binLocationMapping->getBinLocation(),
                    (0 - $binLocationMapping->getStock())
                );
                $this->recordManualStockChange(
                    $articleDetail,
                    $stockChanges,
                    $articleDetail->getPurchasePrice(),
                    $comment,
                    false,
                    $user
                );
            }
        });
    }

    /**
     * Creates a new history entry for a write-off action.
     *
     * That is when an item from a return must be written-off or an item in the warehouse must be written-off.
     *
     * @param ArticleDetail $articleDetail
     * @param NegativeStockChangeList $stockChanges
     * @param ReturnShipmentItem|null $returnShipmentItem
     * @param string $comment
     * @param User|null $user
     * @return StockLedgerEntry[]
     */
    public function recordWriteOff(
        ArticleDetail $articleDetail,
        NegativeStockChangeList $stockChanges,
        ReturnShipmentItem $returnShipmentItem = null,
        $comment = '',
        User $user = null
    ) {
        return $this->performWithLockForArticleDetail($articleDetail, function () use ($articleDetail, $stockChanges, $returnShipmentItem, $comment, $user) {
            $warehouse = $stockChanges->getChangedWarehouse();
            $instockChange = $stockChanges->getTotalStockChange();

            $stockLedgerEntryTemplate = new StockLedgerEntryTemplate($articleDetail, $warehouse, StockLedgerEntry::TYPE_WRITE_OFF);
            $stockLedgerEntryTemplate->setComment($comment);
            $stockLedgerEntryTemplate->setUser($user);
            $stockLedgerEntryTemplate->setReturnShipmentItem($returnShipmentItem);
            $stockLedgerEntryTemplate->setOrderDetail($returnShipmentItem ? $returnShipmentItem->getOrderDetail() : null);

            $stockLedgerEntries = $this->recordStockChange(
                $stockLedgerEntryTemplate,
                new OutgoingPurchasePriceAssignment($this->entityManager, $articleDetail, $warehouse, abs($stockChanges->getTotalStockChange())),
                $stockChanges
            );

            if ($warehouse->isStockAvailableForSale()) {
                $this->increaseInstock($articleDetail, $instockChange);
            }

            return $stockLedgerEntries;
        });
    }

    /**
     * Creates a new history entry for a write-off correction.
     *
     * Correction for {@see self::recordSoldStock()}
     *
     * @param ArticleDetail $articleDetail
     * @param PositiveStockChangeList $stockChanges
     * @param ReturnShipmentItem|null $returnShipmentItem
     * @param string $comment
     * @param User|null $user
     * @return StockLedgerEntry[]
     */
    public function recordWriteOffCorrection(
        ArticleDetail $articleDetail,
        PositiveStockChangeList $stockChanges,
        ReturnShipmentItem $returnShipmentItem = null,
        $comment = '',
        User $user = null
    ) {
        return $this->performWithLockForArticleDetail($articleDetail, function () use ($articleDetail, $stockChanges, $returnShipmentItem, $comment, $user) {
            $warehouse = $stockChanges->getChangedWarehouse();
            $correctionQuantity = $stockChanges->getTotalStockChange();

            $priceAssignment = null;
            if ($returnShipmentItem) {
                $priceAssignment = new ReturningPurchasePriceAssignment(
                    $this->entityManager,
                    $articleDetail,
                    $warehouse,
                    $returnShipmentItem->getOrderDetail(),
                    $correctionQuantity
                );
            } else {
                $priceAssignment = new FallbackPurchasePriceAssignment(
                    $this->entityManager,
                    $articleDetail,
                    $warehouse,
                    $correctionQuantity
                );
            }

            $stockLedgerEntryTemplate = new StockLedgerEntryTemplate($articleDetail, $warehouse, StockLedgerEntry::TYPE_WRITE_OFF);
            $stockLedgerEntryTemplate->setCorrection(true);
            $stockLedgerEntryTemplate->setComment($comment);
            $stockLedgerEntryTemplate->setUser($user);
            $stockLedgerEntryTemplate->setReturnShipmentItem($returnShipmentItem);
            $stockLedgerEntryTemplate->setOrderDetail($returnShipmentItem ? $returnShipmentItem->getOrderDetail() : null);

            $stockEntries = $this->recordStockChange(
                $stockLedgerEntryTemplate,
                $priceAssignment,
                $stockChanges
            );

            if ($warehouse->isStockAvailableForSale()) {
                $this->increaseInstock($articleDetail, $correctionQuantity);
            }

            return $stockEntries;
        });
    }

    /**
     * Logs a relocation between two warehouses, the managed warehouse and the one of the given $destinationLocation, by
     * creating several stock entries:
     *
     *  1. An 'outgoing' stock entry in the managed warehouse for all stock currently located at the given
     *     $sourceLocation. This entry might actually be split into several stock entries depending on the used pricing
     *     model.
     *  2. One 'incoming' stock entry for each 'outgoing' stock entry created in step 1, in reverse order, to the
     *     current bin location in the destination warehouse. This ensures that the stock value is preserved when moving
     *     it to a different warehouse.
     *  3. If the $destinationLocation does not match the current bin location of the article detail in the destination
     *     warehouse, a 'relocation' of all stock on the current bin location to the new $destinationLocation is logged.
     *
     * @param ArticleDetail $articleDetail
     * @param RelocationStockChangeList $stockChanges
     * @param string $comment (optional)
     * @param User|null $user
     * @return StockLedgerEntry[]
     */
    private function recordStockRelocatedBetweenWarehouses(
        ArticleDetail $articleDetail,
        RelocationStockChangeList $stockChanges,
        $comment = '',
        User $user = null
    ) {
        // Try to find the bin location mapping for the destination bin location in the destination warehouse
        $destinationWarehouseBinLocation = $stockChanges->getDestinationLocation();
        $destinationWarehouseBinLocationMapping = $this->warehouseRepository->findArticleDetailBinLocationMapping(
            $articleDetail,
            $destinationWarehouseBinLocation
        );
        if (!$destinationWarehouseBinLocationMapping) {
            // The article is currently not mapped to the destination location, hence use the warehouse's default
            // location as temporary fallback
            $destinationWarehouseBinLocation = $stockChanges->getDestinationLocation()->getWarehouse()->getNullBinLocation();
        }

        // Log all source changes as 'outgoing' entries in this manager
        $outgoingStockEntries = [];
        foreach ($stockChanges->getSourceLocations() as $sourceLocation) {
            $sourceStockChanges = new NegativeStockChangeList([
                $stockChanges->getStockChangeForLocation($sourceLocation),
            ]);
            $outgoingStockEntries = array_merge(
                $outgoingStockEntries,
                $this->recordOutgoingStock($articleDetail, $sourceStockChanges, $comment, $user)
            );
        }

        // Log all relocated stock as incoming in the destination warehouse by creating an 'incoming' stock entry
        // for each outgoing stock entry, in revers order
        $incomingStockEntries = [];
        /** @var StockLedgerEntry $outgoingStockEntry */
        foreach (array_reverse($outgoingStockEntries) as $outgoingStockEntry) {
            $changeAmount = (-1 * $outgoingStockEntry->getChangeAmount());
            if ($changeAmount === 0) {
                continue;
            }
            $incomingStockChanges = $this->stockChangeListFactory->createSingleBinLocationStockChangeList(
                $destinationWarehouseBinLocation,
                $changeAmount
            );
            $incomingStockEntries = array_merge($incomingStockEntries, $this->recordIncomingStock(
                $articleDetail,
                $incomingStockChanges,
                $outgoingStockEntry->getPurchasePrice(),
                $comment,
                $user
            ));
        }
        $createdStockEntries = array_merge($outgoingStockEntries, $incomingStockEntries);

        if (!$destinationWarehouseBinLocation->equals($stockChanges->getDestinationLocation()) && $stockChanges->getRelocatedStock() !== 0) {
            // Relocate all 'incoming' stock from the temporary location to the desired location
            $relocationStockChanges = new RelocationStockChangeList(
                [new BinLocationStockChange($destinationWarehouseBinLocation, (-1 * $stockChanges->getRelocatedStock()))],
                new BinLocationStockChange($stockChanges->getDestinationLocation(), $stockChanges->getRelocatedStock())
            );
            $relocationStockEntries = $this->recordRelocatedStock(
                $articleDetail,
                $relocationStockChanges,
                $comment,
                $user
            );
            $createdStockEntries = array_merge($createdStockEntries, $relocationStockEntries);
        }

        return $createdStockEntries;
    }

    /**
     * Creates a new history entry.
     *
     * @param StockLedgerEntryTemplate $stockLedgerEntryTemplate
     * @param AbstractPurchasePriceAssignment $purchasePriceAssignment The computed pricing (purchase price (source lot
     *        entry) assignment)
     * @param AbstractStockChangeList $stockChanges
     * @return StockLedgerEntry[]
     */
    private function recordStockChange(
        StockLedgerEntryTemplate $stockLedgerEntryTemplate,
        AbstractPurchasePriceAssignment $purchasePriceAssignment,
        AbstractStockChangeList $stockChanges
    ) {
        $articleDetail = $stockLedgerEntryTemplate->getArticleDetail();
        // Make sure the not log any non-initialization stock entries for articles, whose stock is not managed
        if ($stockLedgerEntryTemplate->getType() !== StockLedgerEntry::TYPE_INITIALIZATION && !$this->articleDetailStockInfoProvider->isStockManaged($articleDetail)) {
            return [];
        }

        $warehouse = $stockLedgerEntryTemplate->getWarehouse();
        $incoming = $stockChanges->getTotalStockChange() >= 0;
        // Get stock and snapshots of last stock entry, if available
        $lastStockEntry = $this->articleDetailStockInfoProvider->getLatestStockLedgerEntryForWarehouse(
            $articleDetail,
            $stockLedgerEntryTemplate->getWarehouse()
        );
        $oldStock = ($lastStockEntry) ? $lastStockEntry->getNewStock() : 0;

        // Create a new history entry for each item within the pricing list
        $createdStockEntries = [];
        foreach ($purchasePriceAssignment as $pricingItem) {
            // Set the algebraic sign of the change amount
            $changeAmount = $incoming ? $pricingItem->getQuantity() : -1 * $pricingItem->getQuantity();
            $newEntryComment = $stockLedgerEntryTemplate->getComment();

            // Create a new stock history entry for the given price item
            //
            // Remark: Unassigned (source lot entry-less) incoming entries ("purchase", "stocktake", "return", "manual"),
            // may change the instock quantity from a negative to a positive value. In this case the new entry
            // needs to be split into two entries: an "inactive" (related items are sold already) and an
            // "active" entry (new items). Notice, that the unassigned condition is only necessary for "return"
            // entries.
            if ($incoming && $oldStock < 0) {
                if (($oldStock + $changeAmount > 0 && $pricingItem->getStockentry() === null) || $oldStock + $changeAmount <= 0) {
                    $newEntryComment .= (empty($newEntryComment) ? '' : self::DIVIDER_FOR_COMMENT_POSTFIXES).self::POSTFIX_FOR_COMMENTS_OF_INACTIVE_INCOMING_ENTRIES;
                }

                if ($oldStock + $changeAmount > 0 && $pricingItem->getStockentry() === null) {
                    // Create "inactive" incoming entry
                    $changeAmountNegativeEntry = 0 - $oldStock;
                    $inactiveStockEntry = $stockLedgerEntryTemplate->createStockLedgerEntry($this->transactionId, $oldStock, $changeAmountNegativeEntry);
                    $inactiveStockEntry->setPurchasePrice($pricingItem->getPrice());
                    $inactiveStockEntry->setSourceLotEntry($pricingItem->getStockentry());
                    $inactiveStockEntry->setComment($newEntryComment);
                    $this->entityManager->persist($inactiveStockEntry);
                    $this->entityManager->flush($inactiveStockEntry);
                    $createdStockEntries[] = $inactiveStockEntry;
                    $this->updateBinLocationMappings(
                        $articleDetail,
                        $stockChanges->reduceAbsoluteTotalStockChange($inactiveStockEntry->getChangeAmount())
                    );
                    $this->createStockSnapshot($inactiveStockEntry);

                    $changeAmount -= $changeAmountNegativeEntry;
                    $oldStock += $changeAmountNegativeEntry;
                    $newEntryComment = $stockLedgerEntryTemplate->getComment();
                }
            }

            $stockEntry = $stockLedgerEntryTemplate->createStockLedgerEntry($this->transactionId, $oldStock, $changeAmount);
            $stockEntry->setPurchasePrice($pricingItem->getPrice());
            $stockEntry->setSourceLotEntry($pricingItem->getStockentry());
            $stockEntry->setComment($newEntryComment);
            $this->entityManager->persist($stockEntry);
            $this->entityManager->flush($stockEntry);
            $createdStockEntries[] = $stockEntry;
            $this->updateBinLocationMappings(
                $articleDetail,
                $stockChanges->reduceAbsoluteTotalStockChange($stockEntry->getChangeAmount())
            );
            $this->createStockSnapshot($stockEntry);

            // Update "local" instock quantity for next loop run
            $oldStock += $changeAmount;
        }

        $this->updateCachedStocks($articleDetail, $warehouse);

        return $createdStockEntries;
    }

    /**
     * Increases the Instock of an ArticleDetail by $instockChange.
     *
     * Pro tip: Call with a negative value for $instockChange to decrease.
     *
     * @param ArticleDetail $articleDetail
     * @param int $instockChange
     */
    private function increaseInstock(ArticleDetail $articleDetail, $instockChange)
    {
        $newInStock = $articleDetail->getInStock() + $instockChange;
        $articleDetail->setInStock($newInStock);
        $this->entityManager->flush($articleDetail);
    }

    /**
     * Marks the $newDefaultBinLocation as the default bin location in the $warehouse for the $articleDetail.
     *
     * Uses the given $newDefaultBinLocation to either create or update a mapping from the managed article detail to
     * that bin location and mark it as the default mapping in the managed warehouse. If any other bin location is
     * currently mapped as the default location, the respective mapping is updated and removed, if it does not contain
     * any stock. Passing null as the new default bin location causes the current default bin location mapping to be
     * deselected as the default, effectively leaving the managed article detail without a default bin location mapping.
     * In any case, the bin location mappings of the managed article detail are updated to ensure that any obsolete
     * bin location mappings are removed or new mappings are created as necessary.
     *
     * @param ArticleDetail $articleDetail
     * @param Warehouse $warehouse
     * @param BinLocation|null $newDefaultBinLocation
     */
    public function selectDefaultBinLocation(
        ArticleDetail $articleDetail,
        Warehouse $warehouse,
        BinLocation $newDefaultBinLocation = null
    ) {
        $this->performWithLockForArticleDetail($articleDetail, function () use ($articleDetail, $warehouse, $newDefaultBinLocation) {
            if ($newDefaultBinLocation && $newDefaultBinLocation->equals($warehouse->getNullBinLocation())) {
                throw new Exception(
                    'A warehouse\'s null bin location cannot be selected as an article detail\'s default bin location.'
                );
            }

            // Try to find the current default bin location mapping
            $binLocationMappings = $this->warehouseRepository->findSortedWarehouseBinLocationMappings(
                $warehouse,
                $articleDetail
            );
            $oldDefaultBinLocationMapping = self::getFirstDefaultBinLocationMapping($binLocationMappings);
            if ($oldDefaultBinLocationMapping && $oldDefaultBinLocationMapping->getBinLocation()->equals($newDefaultBinLocation)) {
                // The bin location is already mapped as the default location, hence we don't have to do anything
                return;
            }

            $changedBinLocationMappings = [];
            if ($oldDefaultBinLocationMapping) {
                // Deselect the current default mapping
                $oldDefaultBinLocationMapping->setDefaultMapping(false);
                $changedBinLocationMappings[] = $oldDefaultBinLocationMapping;
            }

            if ($newDefaultBinLocation) {
                // Try to find an existing bin location mapping for the new default location
                $newDefaultBinLocationMapping = $this->entityManager->getRepository(ArticleDetailBinLocationMapping::class)->findOneBy([
                    'articleDetail' => $articleDetail,
                    'binLocation' => $newDefaultBinLocation,
                ]);
                if (!$newDefaultBinLocationMapping) {
                    // No mapping, hence create a new one
                    $newDefaultBinLocationMapping = new ArticleDetailBinLocationMapping($newDefaultBinLocation, $articleDetail);
                    $this->entityManager->persist($newDefaultBinLocationMapping);
                }

                // Select the new mapping as default
                $newDefaultBinLocationMapping->setDefaultMapping(true);
                $changedBinLocationMappings[] = $newDefaultBinLocationMapping;
            }

            // Save all affected bin location mappings
            $this->entityManager->flush($changedBinLocationMappings);

            // Remove any stale bin location mappings using a pseudo relocation change list that contains both the old
            // and the new default bin location
            $destinationBinLocationMapping = array_shift($changedBinLocationMappings);
            $destinationChange = new BinLocationStockChange(
                $destinationBinLocationMapping->getBinLocation(),
                0
            );
            $sourceChanges = array_map(
                function (ArticleDetailBinLocationMapping $binLocationMapping) {
                    return new BinLocationStockChange($binLocationMapping->getBinLocation(), 0);
                },
                $changedBinLocationMappings
            );
            $this->updateBinLocationMappings(
                $articleDetail,
                new RelocationStockChangeList($sourceChanges, $destinationChange)
            );
        });
    }

    /**
     * Returns the bin locations changed by a stock ledger entry in comparison to its direct predecessor.
     *
     * Tries to find the stock entry with the given $stockLedgerEntryId as well as the entry directly preceding that
     * stock entry to calculate the difference between their snapshots, which is then returned.
     *
     * @param int $stockLedgerEntryId
     * @return array
     */
    public function calculateChangedBinLocations($stockLedgerEntryId)
    {
        $stockEntry = $this->entityManager->find(StockLedgerEntry::class, $stockLedgerEntryId);
        if (!$stockEntry) {
            return [];
        }

        // Find the stock entry preceding the current stock entry
        $prevStockEntryQueryBuilder = $this->entityManager->createQueryBuilder();
        $prevStockEntryQueryBuilder
            ->select('stockLedgerEntry')
            ->from(StockLedgerEntry::class, 'stockLedgerEntry')
            ->where('stockLedgerEntry.articleDetail = :articleDetail')
            ->andWhere('stockLedgerEntry.id < :stockLedgerEntryId')
            ->andWhere('stockLedgerEntry.warehouse = :warehouse')
            ->orderBy('stockLedgerEntry.id', 'DESC')
            ->setMaxResults(1)
            ->setParameters([
                'articleDetail' => $stockEntry->getArticleDetail(),
                'stockLedgerEntryId' => $stockEntry->getId(),
                'warehouse' => $stockEntry->getWarehouse(),
            ]);
        $prevStockEntry = $prevStockEntryQueryBuilder->getQuery()->getOneOrNullResult();

        // Determine all changed bin locations
        /** @var ArrayCollection & BinLocationStockSnapshot[] $stockEntrySnapshots */
        $stockEntrySnapshots = $stockEntry->getBinLocationStockSnapshots();
        $prevStockEntrySnapshots = ($prevStockEntry) ? $prevStockEntry->getBinLocationStockSnapshots() : new ArrayCollection();
        $changedBinLocations = [];

        $nullBinLocationTitle = $this->snippetNamespace->get('null_bin_location', 'Unknown');

        // Process current snapshot
        foreach ($stockEntrySnapshots as $snapshot) {
            // Calculate stock change, considering a change of the same bin location in the preceding snapshot
            $stockChange = $snapshot->getStock();
            $prevSnapshot = $prevStockEntrySnapshots->filter(function (BinLocationStockSnapshot $prevSnapshot) use ($snapshot) {
                return $prevSnapshot->getBinLocation()->equals($snapshot->getBinLocation());
            })->first();
            if ($prevSnapshot) {
                $stockChange -= $prevSnapshot->getStock();
            }
            if ($stockChange !== 0) {
                $binLocationCode = ($snapshot->getBinLocation()->isNullBinLocation()) ? $nullBinLocationTitle : $snapshot->getBinLocation()->getCode();
                $changedBinLocations[$binLocationCode] = $stockChange;
            }
        }

        // Process previous snapshot
        foreach ($prevStockEntrySnapshots as $prevSnapshot) {
            $snapshot = $stockEntrySnapshots->filter(function (BinLocationStockSnapshot $currentSnapshot) use ($prevSnapshot) {
                return $currentSnapshot->getBinLocation()->equals($prevSnapshot->getBinLocation());
            })->first();
            $stockChange = ($snapshot) ? ($snapshot->getStock() - $prevSnapshot->getStock()) : -1 * $prevSnapshot->getStock();
            if ($stockChange !== 0) {
                $binLocationCode = ($prevSnapshot->getBinLocation()->isNullBinLocation()) ? $nullBinLocationTitle : $prevSnapshot->getBinLocation()->getCode();
                $changedBinLocations[$binLocationCode] = $stockChange;
            }
        }

        // Sort changes by bin location code
        uksort(
            $changedBinLocations,
            function ($lhs, $rhs) {
                return strnatcasecmp($lhs, $rhs);
            }
        );

        return $changedBinLocations;
    }

    /**
     * Updates the bin location mappings for an $articleDetail in the $warehouse based on the made $stockChanges.
     *
     * For all bin locations affected by the given $stockChanges either an existing mapping is found and
     * updated/removed or a new mapping is created. Then it is asserted that at least one bin location mapping exists
     * and all negative stock resulting from the $stockChanges is moved to the null bin location mapping.
     *
     * @param ArticleDetail $articleDetail
     * @param Warehouse $warehouse
     * @param AbstractStockChangeList $stockChanges
     */
    public function updateBinLocationMappings(
        ArticleDetail $articleDetail,
        AbstractStockChangeList $stockChanges
    ) {
        // Build a debug logging context to help solving issue https://github.com/VIISON/ShopwarePickwareERP/issues/1044
        // TODO: remove this code when the issue is closed
        $debugLoggingContext = [
            'stockChangeList' => array_map(
                function (BinLocationStockChange $stockChange) {
                    return [
                        'binLocationId' => $stockChange->getBinLocation()->getId(),
                        'stockChangeAmount' => $stockChange->getStockChange(),
                    ];
                },
                $stockChanges->getBinLocationStockChanges()
            ),
        ];

        $warehouse = $stockChanges->getChangedWarehouse();

        // Get all article detail bin location mappings in the managed warehouse
        $binLocationMappings = $this->findRefreshedSortedWarehouseBinLocationMappings(
            $warehouse,
            $articleDetail
        );

        $binLocationMappingsToArray = function (array $binLocationMappings) {
            return array_map(
                function (ArticleDetailBinLocationMapping $binLocationMapping) {
                    return [
                        'id' => $binLocationMapping->getId(),
                        'binLocationId' => $binLocationMapping->getBinLocation()->getId(),
                        'articleDetailId' => $binLocationMapping->getArticleDetail()->getId(),
                        'stock' => $binLocationMapping->getStock(),
                        'reservedStock' => $binLocationMapping->getReservedStock(),
                    ];
                },
                $binLocationMappings
            );
        };
        $debugLoggingContext['binLocationArticleDetailMappingsBefore'] = $binLocationMappingsToArray(
            $binLocationMappings
        );

        // Update/remove the ones that are affected by the stock changes
        $negativeStock = 0;
        $nullBinLocationMapping = null;
        $removedBinLocationMappings = [];
        $reservedStocks = [];
        $orderDetails = [];
        $updatedStockReservations = [];
        foreach ($stockChanges->getChangedLocations() as $binLocation) {
            $stockChange = $stockChanges->getStockChangeForLocation($binLocation)->getStockChange();

            // Try to find an existing mapping
            $binLocationMapping = self::getFirstMappingForBinLocation($binLocationMappings, $binLocation);

            if ($binLocationMapping) {
                // Update mapping
                $binLocationMapping->setStock($binLocationMapping->getStock() + $stockChange);
                if ($binLocationMapping->getStock() < 0 && !$binLocationMapping->getBinLocation()->isNullBinLocation()) {
                    // Only the null bin location can have negative stock, hence save it for later
                    $negativeStock += $binLocationMapping->getStock();
                }

                if ($stockChange < 0 && !$binLocationMapping->getBinLocation()->isNullBinLocation()) {
                    // Remove all reserved stock that exceeds the new stock of the mapping
                    /** @var OrderStockReservation[] $stockReservations */
                    $stockReservations = $binLocationMapping->getStockReservations()->toArray();
                    usort($stockReservations, function (OrderStockReservation $lhs, OrderStockReservation $rhs) {
                        return $rhs->getQuantity() - $lhs->getQuantity();
                    });
                    $processedReservedStock = 0;
                    foreach ($stockReservations as $reservation) {
                        $processedReservedStock += $reservation->getQuantity();
                        // Determine how much quantity of the reservation exceeds the new stock of the bin location mapping
                        $exceedingStock = min(
                            ($processedReservedStock - $binLocationMapping->getStock()),
                            $reservation->getQuantity()
                        );
                        if ($exceedingStock <= 0) {
                            continue;
                        }

                        // Remove exceeding quantity from reservation
                        $reservation->setQuantity($reservation->getQuantity() - $exceedingStock);
                        $reservation->getArticleDetailBinLocationMapping()->setReservedStock(
                            $reservation->getArticleDetailBinLocationMapping()->getReservedStock() - $exceedingStock
                        );
                        $updatedStockReservations[] = $reservation;
                        $orderDetailId = $reservation->getOrderDetail()->getId();
                        $orderDetails[$orderDetailId] = $reservation->getOrderDetail();
                        $reservedStocks[$orderDetailId] = (($reservedStocks[$orderDetailId]) ?: 0) + $exceedingStock;
                        if ($reservation->getQuantity() === 0) {
                            // Remove whole reservation
                            $this->entityManager->remove($reservation);
                            $binLocationMapping->getStockReservations()->removeElement($reservation);
                        }
                    }
                }

                // Decide whether mapping must be removed
                if ($binLocationMapping->isDefaultMapping()) {
                    // Default mappings are never removed, but must not have negative stock
                    $binLocationMapping->setStock(max(0, $binLocationMapping->getStock()));
                } elseif ($binLocationMapping->getStock() <= 0 && !$binLocationMapping->getBinLocation()->isNullBinLocation()) {
                    $this->entityManager->remove($binLocationMapping);
                    $removedBinLocationMappings[] = $binLocationMapping;
                    $index = array_search($binLocationMapping, $binLocationMappings);
                    unset($binLocationMappings[$index]);
                }
            } elseif ($stockChange <= 0) {
                // Location not mapped yet, but the stock would be negative or zero. Hence save that negative stock for later.
                $negativeStock += $stockChange;
            } else {
                // Create new mapping
                $binLocationMapping = new ArticleDetailBinLocationMapping($binLocation, $articleDetail);
                $binLocationMapping->setStock($stockChange);
                $this->entityManager->persist($binLocationMapping);
                $binLocationMappings[] = $binLocationMapping;
                if ($binLocation->isNullBinLocation()) {
                    $nullBinLocationMapping = $binLocationMapping;
                }
            }
        }

        // Make sure to have the existing null bin location mapping
        if (!$nullBinLocationMapping) {
            $nullBinLocationMapping = self::getFirstNullBinLocationMapping(
                array_merge($binLocationMappings, $removedBinLocationMappings)
            );
        }

        // Migrate any previously removed stock reservations to the remaining mappings. For that, we order the updated
        // bin location mappings by their 'open' stock (i.e. stock that is not reserved) and the null bin location last.
        // That allows us to apply our strategy of always reducing the number of bin locations per article, if possible.
        usort($binLocationMappings, function (ArticleDetailBinLocationMapping $lhs, ArticleDetailBinLocationMapping $rhs) {
            // Check for default mapping
            if ($lhs->getBinLocation()->isNullBinLocation() && !$rhs->getBinLocation()->isNullBinLocation()) {
                return 1;
            }
            if (!$lhs->getBinLocation()->isNullBinLocation() && $rhs->getBinLocation()->isNullBinLocation()) {
                return -1;
            }

            // No default mappings, hence compare the open stocks
            $lhsOpenStock = $lhs->getStock() - $lhs->getReservedStock();
            $rhsOpenStock = $rhs->getStock() - $rhs->getReservedStock();

            return $lhsOpenStock - $rhsOpenStock;
        });
        while (count($reservedStocks) > 0) {
            $orderDetailId = array_shift(array_keys($reservedStocks));
            foreach ($binLocationMappings as $binLocationMapping) {
                // Check for remaining open stock
                $openStock = $binLocationMapping->getStock() - $binLocationMapping->getReservedStock();
                if ($openStock <= 0) {
                    continue;
                }

                // Reserve as much stock as possible
                $quantity = min($openStock, $reservedStocks[$orderDetailId]);
                $reservation = $binLocationMapping->getStockReservations()->filter(function (OrderStockReservation $stockReservation) use ($orderDetailId) {
                    return $stockReservation->getOrderDetail()->getId() === $orderDetailId;
                })->first();
                if ($reservation) {
                    $reservation->setQuantity($reservation->getQuantity() + $quantity);
                } else {
                    $reservation = new OrderStockReservation(
                        $binLocationMapping,
                        $orderDetails[$orderDetailId],
                        $quantity
                    );
                    $this->entityManager->persist($reservation);
                }
                $binLocationMapping->setReservedStock($binLocationMapping->getReservedStock() + $quantity);
                $updatedStockReservations[] = $reservation;

                $reservedStocks[$orderDetailId] -= $quantity;
                if ($reservedStocks[$orderDetailId] === 0) {
                    // Completely migrated
                    unset($reservedStocks[$orderDetailId]);
                    break;
                }
            }

            if (isset($reservedStocks[$orderDetailId])) {
                // Reserved stock could not be migrated completely
                break;
            }
        }

        // Make sure a null bin location mapping exists, if some negative stock must be saved, all other bin location
        // mappings were removed and/or some reserved stock has not been migrated yet
        $remainingBinLocationMappingsCount = count($binLocationMappings);
        if ($nullBinLocationMapping) {
            $remainingBinLocationMappingsCount -= 1;
        }
        if ($negativeStock < 0 || $remainingBinLocationMappingsCount <= 0 || count($reservedStocks) > 0) {
            if (!$nullBinLocationMapping) {
                // Create a new mapping
                $nullBinLocationMapping = new ArticleDetailBinLocationMapping(
                    $warehouse->getNullBinLocation(),
                    $articleDetail
                );
                $this->entityManager->persist($nullBinLocationMapping);
                $binLocationMappings[] = $nullBinLocationMapping;
            }

            // Add all negative stock to the null bin location mapping
            $nullBinLocationMapping->setStock($nullBinLocationMapping->getStock() + $negativeStock);

            // Add all remaining reserved stock
            foreach ($reservedStocks as $orderDetailId => $quantity) {
                $reservation = $nullBinLocationMapping->getStockReservations()->filter(function (OrderStockReservation $stockReservation) use ($orderDetailId) {
                    return $stockReservation->getOrderDetail()->getId() === $orderDetailId;
                })->first();
                if ($reservation) {
                    $reservation->setQuantity($reservation->getQuantity() + $quantity);
                } else {
                    $reservation = new OrderStockReservation(
                        $nullBinLocationMapping,
                        $orderDetails[$orderDetailId],
                        $quantity
                    );
                    $this->entityManager->persist($reservation);
                }
                $nullBinLocationMapping->setReservedStock($nullBinLocationMapping->getReservedStock() + $quantity);
                $updatedStockReservations[] = $reservation;
            }
        } elseif ($nullBinLocationMapping && $nullBinLocationMapping->getStock() === 0
            && $nullBinLocationMapping->getReservedStock() === 0
        ) {
            // Remove the null bin location mapping, since its new stock is zero and other mappings exist
            $this->entityManager->remove($nullBinLocationMapping);
            if ($this->entityManager->getUnitOfWork()->getEntityState($nullBinLocationMapping) === UnitOfWork::STATE_REMOVED) {
                $removedBinLocationMappings[] = $nullBinLocationMapping;
            }
            $index = array_search($nullBinLocationMapping, $binLocationMappings);
            unset($binLocationMappings[$index]);
        }

        // Save all created/updated/removed mappings and stock reservations
        $this->entityManager->flush(array_merge(
            $binLocationMappings,
            $removedBinLocationMappings,
            $updatedStockReservations
        ));

        $debugLoggingContext['binLocationArticleDetailMappingsAfter'] = $binLocationMappingsToArray(
            $binLocationMappings
        );
        $debugLoggingContext['binLocationArticleDetailMappingsRemoved'] = $binLocationMappingsToArray(
            $removedBinLocationMappings
        );
        $debugLoggingContext['stackTrace'] = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS);

        $this->logger->debug(__METHOD__, $debugLoggingContext);
    }

    /**
     * Creates a new stock snapshot.
     *
     * Loads all bin location mappings of the managed article detail in the managed warehouse and creates a new
     * BinLocationStockSnapshot instance for each of them. All created snapshots are associated with the given
     * $stockEntry.
     *
     * @param StockLedgerEntry $stockEntry
     */
    private function createStockSnapshot(StockLedgerEntry $stockEntry)
    {
        $binLocationMappings = $this->findRefreshedSortedWarehouseBinLocationMappings(
            $stockEntry->getWarehouse(),
            $stockEntry->getArticleDetail()
        );
        $createdSnapshots = [];
        foreach ($binLocationMappings as $binLocationMapping) {
            $snapshot = new BinLocationStockSnapshot(
                $stockEntry,
                $binLocationMapping->getBinLocation(),
                $binLocationMapping->getStock()
            );
            $this->entityManager->persist($snapshot);
            $createdSnapshots[] = $snapshot;
        }
        $this->entityManager->flush($createdSnapshots);
    }

    /**
     * Updates the default purchase price of the article of the stock manager instance.
     *
     * @param ArticleDetail $articleDetail
     * @param float $purchasePrice
     */
    private function updateArticlePurchasePrice(ArticleDetail $articleDetail, $purchasePrice)
    {
        $articleDetail->setPurchasePrice($purchasePrice);
        $this->entityManager->flush($articleDetail);
    }

    /**
     * Updates the cached stocks for an $articleDetail in $warehouse.
     *
     * Gets the article detail stock cache of the current warehouse and updates it with the stock
     * of the latest stock entry. Furthermore gets the current physical stock and physical stock
     * for sale using the respective methods of this manager and saves the values in the respective
     * article detail attributes.
     *
     * @param ArticleDetail $articleDetail
     * @param Warehouse $warehouse
     */
    private function updateCachedStocks(ArticleDetail $articleDetail, Warehouse $warehouse)
    {
        $this->derivedPropertyUpdater->recalculateTotalPhysicalStockForSaleForArticleDetail($articleDetail);
        $this->derivedPropertyUpdater->recalculatePhysicalStockForSaleForArticleDetailInWarehouse(
            $articleDetail,
            $warehouse
        );
        $this->derivedPropertyUpdater->recalculateReservedStockForArticleDetailInWarehouse($articleDetail, $warehouse);
    }

    /**
     * @param ArticleDetail $articleDetail
     */
    private function ensureArticleDetailHasAttribute(ArticleDetail $articleDetail)
    {
        if ($articleDetail->getAttribute()) {
            return;
        }

        // Create attribute
        $attribute = new ArticleAttribute();
        $articleDetail->setAttribute($attribute);
        $attribute->setArticleDetail($articleDetail);
        $this->entityManager->persist($attribute);
        $this->entityManager->flush($attribute);
    }

    /**
     * Tries to acquire a global stock change lock for the article detail managed by this instance.
     *
     * @param ArticleDetail $articleDetail
     * @param Closure $closure
     * @return mixed
     * @throws StockLedgerException If acquiring the lock failed.
     */
    public function performWithLockForArticleDetail(ArticleDetail $articleDetail, Closure $closure)
    {
        $this->entityManager->beginTransaction();

        try {
            $this->articleDetailConcurrencyCoordinator->lockArticleDetailIds($articleDetail->getId());
            $result = $closure();
            $this->entityManager->commit();

            return $result;
        } catch (Exception $e) {
            $this->entityManager->rollback();

            throw $e;
        }
    }

    /**
     * @param ArticleDetailBinLocationMapping[] $articleDetailBinLocationMappings
     * @return ArticleDetailBinLocationMapping|null
     */
    private static function getFirstNullBinLocationMapping(array $articleDetailBinLocationMappings)
    {
        $filteredMappings = array_filter(
            $articleDetailBinLocationMappings,
            function (ArticleDetailBinLocationMapping $binLocationMapping) {
                return $binLocationMapping->getBinLocation()->isNullBinLocation();
            }
        );

        return array_shift($filteredMappings);
    }

    /**
     * @param ArticleDetailBinLocationMapping[] $binLocationMappings
     * @param BinLocation $binLocation
     * @return ArticleDetailBinLocationMapping|null
     */
    private static function getFirstMappingForBinLocation(array $binLocationMappings, BinLocation $binLocation)
    {
        $filteredMappings = array_filter(
            $binLocationMappings,
            function (ArticleDetailBinLocationMapping $binLocationMapping) use ($binLocation) {
                return $binLocationMapping->getBinLocation()->equals($binLocation);
            }
        );

        return array_shift($filteredMappings);
    }

    /**
     * @param ArticleDetailBinLocationMapping[] $articleDetailBinLocationMappings
     * @return ArticleDetailBinLocationMapping|null
     */
    private static function getFirstDefaultBinLocationMapping(array $articleDetailBinLocationMappings)
    {
        $filteredMappings = array_filter(
            $articleDetailBinLocationMappings,
            function (ArticleDetailBinLocationMapping $binLocationMapping) {
                return $binLocationMapping->isDefaultMapping();
            }
        );

        return array_shift($filteredMappings);
    }

    /**
     * {@see \Shopware\CustomModels\ViisonPickwareERP\Warehouse\WarehouseRepository::findSortedWarehouseBinLocationMappings}
     *
     * This method wraps the method above but also refreshes the mappings afterwards.
     *
     * We have to refresh the found bin location mapping because it may not be up-to-date with the database. This is
     * because querying entities with doctrine does not return an up-to-date entity in case the same entity has been
     * fetched from the database before. Hence any queried entities might have outdated values, if the same entity was
     * changed by another process between fetching it for the first time and re-fetching it later.
     * No refreshing the bin location mapping could lead to serious inconsistencies in the database, because its values
     * are used e.g. to determine how to update the bin location mappings of a product after recording stock changes.
     *
     * @see https://github.com/VIISON/ShopwarePickwareERP/issues/1027#issuecomment-453213903
     *
     * @param Warehouse $warehouse
     * @param ArticleDetail $articleDetail
     * @return ArticleDetailBinLocationMapping[]
     */
    private function findRefreshedSortedWarehouseBinLocationMappings(Warehouse $warehouse, ArticleDetail $articleDetail)
    {
        $mappings = $this->warehouseRepository->findSortedWarehouseBinLocationMappings($warehouse, $articleDetail);
        foreach ($mappings as $mapping) {
            $this->entityManager->refresh($mapping);
        }

        return $mappings;
    }

    /**
     * {@see \Shopware\CustomModels\ViisonPickwareERP\Warehouse\WarehouseRepository::findArticleDetailBinLocationMapping}
     *
     * This method wraps the method above but also refreshes the mappings afterwards.
     *
     * We have to refresh the found bin location mappings because they may not be up-to-date with the database. This is
     * because querying entities with doctrine does not return an up-to-date entity in case the same entity has been
     * fetched from the database before. Hence any queried entities might have outdated values, if the same entity was
     * changed by another process between fetching it for the first time and re-fetching it later.
     * No refreshing the bin location mappings could lead to serious inconsistencies in the database, because their
     * values are used e.g. to determine how to update the bin location mappings of a product after recording stock
     * changes.
     *
     * @see https://github.com/VIISON/ShopwarePickwareERP/issues/1027#issuecomment-453213903
     *
     * @param ArticleDetail $articleDetail
     * @param BinLocation $binLocation
     * @return ArticleDetailBinLocationMapping|null
     */
    private function findRefreshedArticleDetailBinLocationMapping(
        ArticleDetail $articleDetail,
        BinLocation $binLocation
    ) {
        $mapping = $this->warehouseRepository->findArticleDetailBinLocationMapping($articleDetail, $binLocation);
        if ($mapping) {
            $this->entityManager->refresh($mapping);
        }

        return $mapping;
    }
}
